<?php

namespace App\Filament\Pages;

use App\Settings\CardDefaultsSetting;
use Filament\Forms;
use Filament\Forms\Components\ColorPicker;
use Filament\Forms\Form;
use Filament\Forms\Components\Textarea;
use Filament\Pages\SettingsPage;
use Filament\Forms\Components\FileUpload;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use Filament\Forms\Set;

class GeneralSettingsPage extends SettingsPage
{
    protected static ?string $navigationIcon = 'heroicon-o-cog-6-tooth';
    protected static ?string $title = 'General Settings';
    protected static ?string $navigationGroup = 'Settings';
    protected static ?string $navigationLabel = 'General Settings';
    protected static ?string $slug = 'general-settings';
    protected static ?int $navigationSort = 7;

    protected static string $settings = CardDefaultsSetting::class;

    /**
     * Hold old values before save to handle cleanup
     */
    protected array $oldValues = [];

    protected function mutateFormDataBeforeSave(array $data): array
    {
        // Capture old file paths before they get replaced
        $settings = app(CardDefaultsSetting::class);

        $this->oldValues = [
            'logo' => $settings->logo,
            'thumbnail' => $settings->thumbnail,
            'mockup' => $settings->mockup,
        ];

        return $data;
    }

    protected function afterSave(): void
    {
        $settings = app(CardDefaultsSetting::class);

        foreach (['logo', 'thumbnail', 'mockup'] as $field) {
            $oldPath = $this->oldValues[$field] ?? null;
            $newPath = $settings->$field ?? null;

            // If the old file existed and is different from the new one, delete it
            if ($oldPath && $oldPath !== $newPath && Storage::disk('public')->exists($oldPath)) {
                Storage::disk('public')->delete($oldPath);
            }
        }
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([

                Forms\Components\Section::make('Site Defaults')
                    ->schema([
                        Forms\Components\TextInput::make('app_name')
                            ->label("App Name (Don't update unless you're the dev)")
                            ->required()
                            ->placeholder('ex: 10kcards')
                            ->maxLength(255)
                            ->columnSpanFull(),

                        Forms\Components\TextInput::make('site_name')
                            ->label('Site Name')
                            ->required()
                            ->placeholder('10k Cards')
                            ->maxLength(255),

                        Forms\Components\TextInput::make('site_url')
                            ->label('Site URL')
                            ->placeholder('https://10kcards.site')
                            ->required()
                            ->url()
                            ->maxLength(255)
                            ->afterStateUpdated(function ($state, $set) {
                                if (!empty($state) && substr($state, -1) !== '/') {
                                    $set('site_url', $state . '/');
                                }
                            }),

                        Forms\Components\TextInput::make('site_slogan')
                            ->label('Site Slogan')
                            ->columnSpanFull()
                            ->required()
                            ->maxLength(255),

                        ColorPicker::make('site_slogan_color_1')
                            ->label('Color Gradient #1 (Slogan)')
                            ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                            ->required()
                            ->default('#133a9d')
                            ->afterStateUpdated(fn($state, Set $set) => $set('link', trim($state))),

                        ColorPicker::make('site_slogan_color_2')
                            ->label('Color Gradient #2 (Slogan)')
                            ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                            ->required()
                            ->default('#D91656')
                            ->afterStateUpdated(fn($state, Set $set) => $set('link', trim($state))),

                        ColorPicker::make('site_button_color')
                            ->label('Welcome, Login & Signup Page Button Color')
                            ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                            ->required()
                            ->afterStateUpdated(fn($state, Set $set) => $set('link', trim($state))),

                        ColorPicker::make('site_button_text_color')
                            ->label('Welcome, Login & Signup Page Button Color (Text)')
                            ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                            ->required()
                            ->afterStateUpdated(fn($state, Set $set) => $set('link', trim($state))),

                        ColorPicker::make('site_link_color')
                            ->label('Welcome, Login & Signup Page Link Color')
                            ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                            ->required()
                            ->afterStateUpdated(fn($state, Set $set) => $set('link', trim($state))),

                        Textarea::make('site_desc')
                            ->label('Site Description')
                            ->placeholder('A short description or tagline about the site.')
                            ->required()
                            ->maxLength(255)
                            ->columnSpanFull(),

                        FileUpload::make('logo')
                            ->label('Upload Website Logo')
                            ->disk('public')
                            ->directory('logo')
                            ->uploadingMessage('Uploading Website Logo. Please wait...')
                            ->image()
                            ->required()
                            ->imageEditor()
                            ->columnSpanFull()
                            ->getUploadedFileNameForStorageUsing(
                                fn(TemporaryUploadedFile $file): string =>
                                (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                            ),

                        FileUpload::make('thumbnail')
                            ->label('Upload Website Thumbnail')
                            ->disk('public')
                            ->directory('thumbnails')
                            ->uploadingMessage('Uploading Website Thumbnail. Please wait...')
                            ->image()
                            ->imageEditor()
                            ->required()
                            ->columnSpanFull()
                            ->imageResizeMode('cover')
                            ->imageCropAspectRatio('1:1')
                            ->imageResizeTargetWidth('520')
                            ->imageResizeTargetHeight('520')
                            ->getUploadedFileNameForStorageUsing(
                                fn(TemporaryUploadedFile $file): string =>
                                (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                            )
                            ->imageEditorAspectRatios([
                                '1:1',
                            ]),

                        FileUpload::make('mockup')
                            ->label('Upload Website Mockup')
                            ->disk('public')
                            ->directory('mockup')
                            ->uploadingMessage('Uploading Website Mockup. Please wait...')
                            ->image()
                            ->imageEditor()
                            ->columnSpanFull()
                            ->imageResizeMode('cover')
                            ->required()
                            ->getUploadedFileNameForStorageUsing(
                                fn(TemporaryUploadedFile $file): string =>
                                (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                            )
                            ->imageEditorAspectRatios([
                                '1:1',
                            ]),
                    ])->columns(2),

                Forms\Components\Section::make('Account Defaults')
                    ->schema([
                        Forms\Components\TextInput::make('default_password')
                            ->required()
                            ->label('Default password for accounts created by Admin')
                            ->placeholder('Default password for accounts created by Admin')
                    ]),

                Forms\Components\Section::make('Footer Defaults')
                    ->description('The default footer value for the Free card and initial values for other versions.')
                    ->schema([
                        Forms\Components\TextInput::make('footer_text')
                            ->label('Footer Text')
                            ->required()
                            ->placeholder('Get your 10kcards now')
                            ->maxLength(255),

                        Forms\Components\TextInput::make('footer_link')
                            ->label('Footer Link')
                            ->placeholder('https://10kcards.com')
                            ->required()
                            ->url()
                            ->maxLength(255),
                    ])->columns(2),

                Forms\Components\Section::make('Ad Banner Defaults')
                    ->description('This will be the recommended and crop size of the ad banner.')
                    ->schema([
                        Forms\Components\TextInput::make('ad_banner_w')
                            ->label('Ad Banner Width')
                            ->numeric()
                            ->required()
                            ->maxLength(255),

                        Forms\Components\TextInput::make('ad_banner_h')
                            ->label('Ad Banner Height')
                            ->required()
                            ->numeric()
                            ->maxLength(255),
                    ])->columns(2),


                Forms\Components\Section::make('Membership')
                    ->description("When disabled, admin/users can't add/register with that specific card type.")
                    ->schema([
                        Forms\Components\Toggle::make('free_trial_enabled')
                            ->label('Free Trial'),

                        Forms\Components\TextInput::make('free_trial_duration')
                            ->label('Free Trial Duration')
                            ->required()
                            ->numeric(),

                        Forms\Components\Toggle::make('platinum_enabled')
                            ->label(app(CardDefaultsSetting::class)->term_platinum . ' Cards'),

                        Forms\Components\Toggle::make('influencer_enabled')
                            ->label('Influencer Cards'),

                        Forms\Components\Toggle::make('sponsored_enabled')
                            ->label('Sponsored Cards'),

                    ]),

                Forms\Components\Section::make('Menu')
                    ->description("This is optional. The menu will be displayed on the top left corner of the landing page.")
                    ->schema([
                        Forms\Components\Toggle::make('menu_enabled')
                            ->label('Enable Menu'),

                        Forms\Components\Select::make('menu_position_position')
                            ->required()
                            ->label('Position')
                            ->columnSpanFull()
                            ->default('left')
                            ->options([
                                'left' => 'Left',
                                'right' => 'Right',
                            ]),

                        Forms\Components\Repeater::make('menu_items')
                            ->label('Menu Items')
                            ->schema([
                                Forms\Components\TextInput::make('title')
                                    ->label('Menu Title')
                                    ->required(),

                                Forms\Components\TextInput::make('link')
                                    ->label('Menu Link')
                                    ->required()
                                    ->url()
                                    ->placeholder('https://example.com'),
                            ])
                            ->columns(2)
                            ->reorderable(),
                        // ->collapsed(),
                        // ->visible(fn($get) => $get('menu_enabled')),
                    ]),

                Forms\Components\Section::make('Landing Page Links')
                    ->description("This is optional. The menu will be displayed on landing page.")
                    ->schema([
                        Forms\Components\Toggle::make('landing_page_menu_enabled')
                            ->label('Enable Menu'),

                        Forms\Components\Select::make('landing_page_menu_position')
                            ->required()
                            ->label('Position')
                            ->columnSpanFull()
                            ->default('left')
                            ->options([
                                'left' => 'Left',
                                'right' => 'Right',
                            ]),

                        Forms\Components\Repeater::make('landing_page_menu_items')
                            ->label('Menu Items')
                            ->schema([
                                Forms\Components\TextInput::make('title')
                                    ->label('Menu Title')
                                    ->required(),

                                Forms\Components\TextInput::make('link')
                                    ->label('Menu Link')
                                    ->required()
                                    ->url()
                                    ->placeholder('https://example.com'),
                            ])
                            ->columns(2)
                            ->reorderable(),
                    ]),

                Forms\Components\Section::make('SMS Notification')
                    ->schema([

                        Forms\Components\Toggle::make('admin_sms_notification')
                            ->label('Enable Notification')
                            ->reactive()
                            ->helperText('Needs a Twilio account to make it work. When ON, admin will receive sms notifications on every action from the platform'),

                        // Forms\Components\TextInput::make('admin_sms_number')
                        //     ->label("Admin SMS Number")
                        //     ->helperText('Include country code. Number added here will received the sms notifications. If adding more, separate with comma.')
                        //     ->required(fn(callable $get) => $get('admin_sms_notification') === true)
                        //     ->afterStateUpdated(function ($state, callable $set) {
                        //         $clean = preg_replace('/[\s\-\(\)]/', '', $state);
                        //         $set('admin_sms_number', $clean);
                        //     })
                        //     ->regex('/^\+[1-9]\d{7,14}$/')
                        //     ->validationMessages([
                        //         'regex' => 'Please enter a valid phone number format with country code (e.g. +639058651234).',
                        //     ])
                        //     ->maxLength(16)
                        //     ->reactive()
                        //     ->columnSpanFull(),

                        Forms\Components\TextInput::make('admin_sms_number')
                            ->label("Admin SMS Number")
                            ->helperText('Include country code. Numbers separated by comma.')
                            ->required(fn(callable $get) => $get('admin_sms_notification') === true)
                            ->afterStateUpdated(function ($state, callable $set) {
                                // remove spaces/dashes for each number
                                $clean = preg_replace('/[\s\-\(\)]/', '', $state);
                                $set('admin_sms_number', $clean);
                            })
                            ->regex('/^(\+[1-9]\d{7,14})(,\s*\+[1-9]\d{7,14})*$/')
                            ->validationMessages([
                                'regex' => 'Use comma-separated valid phone numbers (e.g. +639058651234, +639171234567).',
                            ])
                            ->maxLength(255)
                            ->reactive()
                            ->columnSpanFull(),

                    ]),

                Forms\Components\Section::make('WhatsApp Notification')
                    ->schema([
                        Forms\Components\Toggle::make('admin_whatsapp_notification')
                            ->label('Enable Notification')
                            ->reactive()
                            ->helperText('Needs a Twilio account to make it work. When ON, admin will receive WhatsApp notifications on every action from the platform'),

                        // Forms\Components\TextInput::make('admin_whatsapp_number')
                        //     ->label("Admin WhatsApp Number")
                        //     ->helperText('Include country code. Number added here will received the whatsapp notifications. If adding more, separate with comma.')
                        //     ->required(fn(callable $get) => $get('admin_whatsapp_notification') === true)
                        //     ->afterStateUpdated(function ($state, callable $set) {
                        //         $clean = preg_replace('/[\s\-\(\)]/', '', $state);
                        //         $set('admin_whatsapp_number', $clean);
                        //     })
                        //     ->regex('/^\+[1-9]\d{7,14}$/')
                        //     ->validationMessages([
                        //         'regex' => 'Please enter a valid phone number format with country code (e.g. +639058651234).',
                        //     ])
                        //     ->maxLength(16)
                        //     ->reactive()
                        //     ->columnSpanFull(),

                        Forms\Components\TextInput::make('admin_whatsapp_number')
                            ->label("Admin WhatsApp Number")
                            ->helperText('Include country code. Numbers separated by comma.')
                            ->required(fn(callable $get) => $get('admin_whatsapp_notification') === true)
                            ->afterStateUpdated(function ($state, callable $set) {
                                $clean = preg_replace('/[\s\-\(\)]/', '', $state);
                                $set('admin_whatsapp_number', $clean);
                            })
                            ->regex('/^(\+[1-9]\d{7,14})(,\s*\+[1-9]\d{7,14})*$/')
                            ->validationMessages([
                                'regex' => 'Use comma-separated valid phone numbers (e.g. +639058651234, +639171234567).',
                            ])
                            ->maxLength(255)
                            ->reactive()
                            ->columnSpanFull(),
                    ]),

                Forms\Components\Section::make('Email Notification')
                    ->schema([
                        Forms\Components\Toggle::make('admin_email_notification')
                            ->label('Enable Notification')
                            ->reactive()
                            ->helperText('When ON, admin will receive email notifications on every action from the platform'),

                        Forms\Components\TextInput::make('admin_email')
                            ->label("Admin email(s)")
                            ->helperText('Email(s) listed here will received the email notifications. If adding more, separate with comma.')
                            ->rule('regex:/^([^\s,]+@[^\s,]+\.[^\s,]+)(\s*,\s*[^\s,]+@[^\s,]+\.[^\s,]+)*$/')
                            ->required(fn(callable $get) => $get('admin_email_notification') === true)
                            ->maxLength(255)
                            ->reactive()
                            ->columnSpanFull(),

                        Forms\Components\TextInput::make('sender_email')
                            ->label("Sender email")
                            ->helperText("Email added here will send the email notifications (Don't change, for dev only).")
                            ->rule('regex:/^([^\s,]+@[^\s,]+\.[^\s,]+)(\s*,\s*[^\s,]+@[^\s,]+\.[^\s,]+)*$/')
                            ->required(fn(callable $get) => $get('admin_email_notification') === true)
                            ->maxLength(255)
                            ->reactive()
                            ->columnSpanFull(),
                    ]),

                Forms\Components\Section::make('Site Functionality')
                    // ->description('Turn ON when doing some updates/maintenance work to disable the users from using the cards/logging in.')
                    ->schema([
                        Forms\Components\Toggle::make('maintenance_mode')
                            ->label('Maintenance Mode')
                            ->helperText('Turn ON when doing some updates/maintenance work to disable the users from using the cards/logging in.'),

                        Forms\Components\Toggle::make('manual_capture_mode')
                            ->label('Manual Capture Mode')
                            ->reactive()
                            ->helperText('If turned ON, subscriptions payment method will be manual'),

                        Forms\Components\Toggle::make('manual_capture_mode_annual_only')
                            ->label('Annual Only')
                            ->reactive()
                            ->visible(fn(callable $get) => $get('manual_capture_mode') === true)
                            ->helperText('If turned ON, manual capture will only show ANNUAL card subscription items.'),

                        Forms\Components\Toggle::make('affiliate_functionality')
                            ->label('Affiliate Functionality'),

                        Forms\Components\Toggle::make('replays_functionality')
                            ->label('Replay Functionality'),

                        Forms\Components\Toggle::make('reviews_functionality')
                            ->label('Review Functionality')


                    ]),
            ]);
    }
}
