<?php

namespace App\Filament\Pages;

use App\Models\Product;
use App\Models\User;
use App\Models\SubCard;
use Filament\Pages\Page;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ProductsByOwner extends Page implements Tables\Contracts\HasTable
{
    use Tables\Concerns\InteractsWithTable;

    protected static ?string $navigationIcon = 'heroicon-o-shopping-bag';
    protected static ?string $navigationLabel = 'Products';
    protected static ?string $navigationGroup = 'Client Management';
    protected static ?int $navigationSort = 5;
    protected static ?string $title = 'Products';
    protected static string $view = 'filament.pages.products-by-owner';

    public string $activeTab = 'owners'; // Default tab

    public function setTab(string $tab): void
    {
        $this->activeTab = $tab;
    }

    public function mount(): void
    {
        $this->activeTab = request()->query('tab', 'owners');
    }

    public function table(Table $table): Table
    {
        // 👇 Determine which query to use based on the tab
        if ($this->activeTab === 'all') {
            return $this->allProductsTable($table);
        }

        return $this->ownersTable($table);
    }

    protected function ownersTable(Table $table): Table
    {
        return $table
            ->query(
                Product::query()
                    ->selectRaw('
                        IFNULL(CONCAT("sub-", page_id), CONCAT("user-", user_id)) as owner_key,
                        COUNT(*) as product_count,
                        MAX(id) as id
                    ')
                    ->groupBy('owner_key')
            )
            ->headerActions([
                Tables\Actions\Action::make('create')
                    ->label('Add Product')
                    ->icon('heroicon-o-plus')
                    ->color('success')
                    ->url(route('filament.admin.resources.products.create')),
            ])
            ->columns([
                Tables\Columns\TextColumn::make('owner')
                    ->label('Owner')
                    ->getStateUsing(function ($record) {
                        $key = $record->owner_key;

                        if (str_starts_with($key, 'user-')) {
                            $userId = substr($key, 5);
                            $user = User::find($userId);
                            if ($user) {
                                return "{$user->firstname} {$user->lastname} - @$user->username (Main Card)";
                            }
                        }

                        if (str_starts_with($key, 'sub-')) {
                            $subId = substr($key, 4);
                            $sub = SubCard::find($subId);
                            if ($sub) {
                                return "{$sub->firstname} {$sub->lastname} - @$sub->username (Sub Card)";
                            }
                        }

                        return 'Unknown';
                    })
                    ->searchable(query: function (Builder $query, string $search): Builder {
                        $userIds = User::query()
                            ->where(DB::raw('CONCAT(firstname, " ", lastname)'), 'like', "%{$search}%")
                            ->orWhere('username', 'like', "%{$search}%")
                            ->pluck('id')
                            ->toArray();

                        $subIds = SubCard::query()
                            ->where(DB::raw('CONCAT(firstname, " ", lastname)'), 'like', "%{$search}%")
                            ->orWhere('username', 'like', "%{$search}%")
                            ->pluck('id')
                            ->toArray();

                        return $query->where(function ($q) use ($userIds, $subIds) {
                            $q->whereIn(DB::raw('user_id'), $userIds)
                                ->orWhereIn(DB::raw('page_id'), $subIds);
                        });
                    }),

                Tables\Columns\TextColumn::make('product_count')
                    ->label('Products')
                    ->sortable(),
            ])
            ->actions([
                Tables\Actions\Action::make('view_products')
                    ->label('View Products')
                    ->icon('heroicon-o-eye')
                    ->url(fn($record) => url("/admin/products/by-owner/{$record->owner_key}"))
                    ->color('primary'),
            ])
            ->defaultSort('product_count', 'desc')
            ->searchPlaceholder('Search by owner name or username');
    }

    protected function allProductsTable(Table $table): Table
    {
        return $table
            ->query(Product::query())
            ->columns([

                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success')
                    ->offColor('danger'),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Added')
                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),

                Tables\Columns\TextColumn::make('owner')
                    ->label('Owner')
                    ->getStateUsing(function ($record) {
                        // Check SubCard first
                        if ($record->page_id) {
                            $sub = SubCard::find($record->page_id);
                            if ($sub) {
                                return "{$sub->firstname} {$sub->lastname} - @$sub->username (Sub Card)";
                            }
                        }

                        // Then check User (Main Card)
                        if ($record->user_id) {
                            $user = User::find($record->user_id);
                            if ($user) {
                                return "{$user->firstname} {$user->lastname} - @$user->username (Main Card)";
                            }
                        }

                        return 'Unknown';
                    })
                    ->searchable(query: function (Builder $query, string $search): Builder {
                        // Allow searching by owner’s name or username
                        $userIds = User::query()
                            ->where(DB::raw('CONCAT(firstname, " ", lastname)'), 'like', "%{$search}%")
                            ->orWhere('username', 'like', "%{$search}%")
                            ->pluck('id')
                            ->toArray();

                        $subIds = SubCard::query()
                            ->where(DB::raw('CONCAT(firstname, " ", lastname)'), 'like', "%{$search}%")
                            ->orWhere('username', 'like', "%{$search}%")
                            ->pluck('id')
                            ->toArray();

                        return $query->where(function ($q) use ($userIds, $subIds) {
                            $q->whereIn('user_id', $userIds)
                                ->orWhereIn('page_id', $subIds);
                        });
                    }),

                Tables\Columns\TextColumn::make('product_name')
                    ->label('Product Name')
                    ->searchable(),

                // Tables\Columns\TextColumn::make('category.title')
                //     ->label('Category')
                //     ->searchable(),


                // Tables\Columns\TextColumn::make('collection.title')
                //     ->label('Collection')
                //     ->searchable(),

                Tables\Columns\TextColumn::make('price')
                    ->money('usd')
                    ->label('Price'),
            ])
            ->headerActions([
                Tables\Actions\Action::make('create')
                    ->label('Add Product')
                    ->icon('heroicon-o-plus')
                    ->color('success')
                    ->url(route('filament.admin.resources.products.create')),
            ])
            ->actions([
                Tables\Actions\Action::make('edit')
                    ->url(fn($record) => route('filament.admin.resources.products.edit', $record))
                    ->icon('heroicon-o-pencil')
                    ->color('primary'),
            ])
            ->defaultSort('created_at', 'desc');
    }


    public function getTabs(): array
    {
        return [
            'owners' => [
                'label' => 'By Owner',
                'icon' => 'heroicon-o-users',
                'badge' => Product::query()->distinct('user_id')->count(),
                'url' => route('filament.admin.pages.products-by-owner', ['tab' => 'owners']),
            ],
            'all' => [
                'label' => 'All Products',
                'icon' => 'heroicon-o-shopping-bag',
                'badge' => Product::count(),
                'url' => route('filament.admin.pages.products-by-owner', ['tab' => 'all']),
            ],
        ];
    }
}
