<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ColorPresetResource\Pages;
use App\Models\ColorPreset;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Forms\Components\ColorPicker;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Set;

class ColorPresetResource extends Resource
{
    protected static ?string $model = ColorPreset::class;

    protected static ?string $navigationIcon = 'heroicon-o-paint-brush';

    protected static ?string $title = 'Color Presets';
    protected static ?string $navigationGroup = 'Settings';
    protected static ?string $navigationLabel = 'Color Presets';
    protected static ?int $navigationSort = 9;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('title')
                    ->label('Preset Title')
                    ->columnSpanFull()
                    ->maxLength(255)
                    ->required(),

                ColorPicker::make('custom_button')
                    ->label('Button Color')
                    ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                    ->required()
                    ->default('#facc15')
                    ->afterStateUpdated(fn($state, Set $set) => $set('custom_button', trim($state))),

                ColorPicker::make('custom_button_text')
                    ->label('Button Text Color')
                    ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                    ->required()
                    ->default('#000000')
                    ->afterStateUpdated(fn($state, Set $set) => $set('custom_button_text', trim($state))),

                ColorPicker::make('link')
                    ->label('Link Color')
                    ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                    ->required()
                    ->default('#ca8a04')
                    ->afterStateUpdated(fn($state, Set $set) => $set('link', trim($state))),

                ColorPicker::make('profile_pic_border')
                    ->label('Profile Pic Border Color')
                    ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                    ->required()
                    ->default('#ca8a04')
                    ->afterStateUpdated(fn($state, Set $set) => $set('profile_pic_border', trim($state))),

                ColorPicker::make('card_light')
                    ->label('Light Mode Background Color')
                    ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                    ->required()
                    ->default('#ffffff')
                    ->afterStateUpdated(fn($state, Set $set) => $set('card_light', trim($state))),

                ColorPicker::make('card_dark')
                    ->label('Dark Mode Background Color')
                    ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                    ->required()
                    ->default('#18181b')
                    ->afterStateUpdated(fn($state, Set $set) => $set('card_dark', trim($state))),

                // Text Color Settings
                ColorPicker::make('text_light')
                    ->label('Light Mode Font Color')
                    ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                    ->required()
                    ->default('#1e293b')
                    ->afterStateUpdated(fn($state, Set $set) => $set('text_light', trim($state))),

                ColorPicker::make('text_dark')
                    ->label('Dark Mode Font Color')
                    ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                    ->required()
                    ->default('#cbd5e1')
                    ->afterStateUpdated(fn($state, Set $set) => $set('text_dark', trim($state))),

                Forms\Components\Toggle::make('is_active')
                    ->label('Preset Active')
                    ->default(true)
                    ->required(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->description('Color preset(s) saved here will be used by the clients.')
            ->columns([
                Tables\Columns\IconColumn::make('is_active')
                    ->columnSpanFull()
                    ->label('Active')
                    ->boolean()
                    ->default(true),
                Tables\Columns\TextColumn::make('title')
                    ->label('Preset Title'),
                Tables\Columns\TextColumn::make('colors')
                    ->label('Colors')
                    ->html()
                    ->state(function ($record) {
                        $colorFields = [
                            'custom_button',
                            'custom_button_text',
                            'link',
                            'profile_pic_border',
                            'card_light',
                            'card_dark',
                            'text_light',
                            'text_dark',
                        ];

                        return collect($colorFields)->map(function ($field) use ($record) {
                            $color = $record->{$field};
                            return "<span style='display: inline-block; width: 16px; height: 16px; border-radius: 50%; background-color: {$color}; margin-right: 4px; border: 1px solid #ccc;'></span>";
                        })->implode('');
                    }),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListColorPresets::route('/'),
            'create' => Pages\CreateColorPreset::route('/create'),
            'edit' => Pages\EditColorPreset::route('/{record}/edit'),
        ];
    }
}
