<?php

namespace App\Filament\Resources\PartnershipResource\Pages;

use App\Filament\Resources\PartnershipResource;
use App\Models\Partnership;
use Filament\Actions;
use Filament\Resources\Pages\ViewRecord;
use App\Models\User;
use Filament\Notifications\Notification;
use App\Notifications\UserNotification;

class ViewPartnership extends ViewRecord
{
    protected static string $resource = PartnershipResource::class;

    protected function getHeaderActions(): array
    {
        return [
            Actions\Action::make('update_partnership')
                ->label('Update Partnership')
                ->icon('heroicon-s-users')
                ->color('success')
                ->requiresConfirmation()
                ->visible(fn($record) => ! $record->is_upgraded)
                ->action(function ($record) {
                    // Map plans to card types
                    $planMap = [
                        'senior'  => 'Senior',
                        'junior'  => 'Junior',
                    ];

                    if (! isset($planMap[$record->requested_partnership])) {
                        Notification::make()
                            ->title('Invalid partnership level')
                            ->danger()
                            ->send();
                        return;
                    }

                    // Get user
                    $user = User::find($record->user_id);
                    $update = Partnership::find($record->id);

                    if (! $user) {
                        Notification::make()
                            ->title('User not found')
                            ->danger()
                            ->send();

                        return;
                    }

                    // Upgrade user
                    $user->update([
                        'partnership_level' => $planMap[$record->requested_partnership],
                    ]);

                    $update->update([
                        'is_upgraded' => true,
                    ]);

                    Notification::make()
                        ->title('Partnership level successfully updated')
                        ->body("{$user->firstname} {$user->lastname}'s partnership level was updated to {$planMap[$record->requested_partnership]} Level.")
                        ->success()
                        ->send();

                    // Notify the user using your existing UserNotification
                    if ($user && ! $user->is_admin) {
                        $user->notify(
                            new UserNotification(
                                title: 'Partnership updated',
                                message: "<span class='font-semibold'>{$user->firstname} {$user->lastname}</span>, your partnership level has been updated to <span class='font-semibold'>{$planMap[$record->requested_partnership]} Level</span>.",
                                url: route('partnership.index')
                            )
                        );
                    }
                }),
        ];
    }
}
