<?php

namespace App\Filament\Resources;

use App\Filament\Resources\PresentationPageResource\Pages;
use App\Filament\Resources\PresentationPageResource\RelationManagers;
use App\Filament\Resources\PresentationPageResource\RelationManagers\PageContentRelationManager;
use App\Models\PresentationPage;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Models\SubCard;
use App\Models\User;
use App\Settings\CardDefaultsSetting;
use Filament\Forms\Components\Select;
use Filament\Forms\Set;
use Illuminate\Support\HtmlString;
use Carbon\Carbon;

class PresentationPageResource extends Resource
{
    protected static ?string $model = PresentationPage::class;

    protected static ?string $navigationIcon = 'heroicon-o-document';

    protected static ?string $navigationGroup = 'Client Management';
    protected static ?string $navigationLabel = 'Pages';
    protected static ?string $modelLabel = 'Page';
    protected static ?int $navigationSort = 5;

    // hide this resource
    public static function shouldRegisterNavigation(): bool
    {
        return false;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make("Page Details")
                    ->schema([

                        // 👇 Added hidden fields so Filament will save them
                        Forms\Components\Hidden::make('user_id'),
                        Forms\Components\Hidden::make('page_id'),

                        Select::make('owner')
                            ->label('Owner')
                            ->placeholder('Search Client or SubCard')
                            ->getSearchResultsUsing(function (string $search): array {
                                // --- Users ---
                                $users = User::query()
                                    ->where(function ($query) use ($search) {
                                        if (strpos($search, '@') === 0 && strlen($search) > 1) {
                                            $query->where('username', 'like', "%" . substr($search, 1) . "%");
                                        } else {
                                            $query->where('firstname', 'like', "%{$search}%")
                                                ->orWhere('lastname', 'like', "%{$search}%")
                                                ->orWhere('username', 'like', "%{$search}%");
                                        }
                                    })
                                    ->whereIn('user_type', ['Platinum', 'Gold'])
                                    ->limit(25)
                                    ->get()
                                    ->mapWithKeys(fn($user) => [
                                        "user-{$user->id}" => "{$user->firstname} {$user->lastname} - @$user->username (User)"
                                    ]);

                                // --- SubCards ---
                                $subCards = SubCard::query()
                                    ->where(function ($query) use ($search) {
                                        $query->where('firstname', 'like', "%{$search}%")
                                            ->orWhere('lastname', 'like', "%{$search}%")
                                            ->orWhere('username', 'like', "%{$search}%");
                                    })
                                    ->limit(25)
                                    ->get()
                                    ->mapWithKeys(fn($sub) => [
                                        "sub-{$sub->id}" => "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)"
                                    ]);

                                return $users->toArray() + $subCards->toArray();
                            })
                            ->getOptionLabelUsing(function ($value) {
                                if (!$value) return '';

                                [$type, $id] = explode('-', $value);

                                if ($type === 'user') {
                                    $user = User::find($id);
                                    return $user ? "{$user->firstname} {$user->lastname} - @$user->username (User)" : '';
                                }

                                if ($type === 'sub') {
                                    $sub = SubCard::find($id);
                                    return $sub ? "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)" : '';
                                }

                                return '';
                            })
                            ->formatStateUsing(function ($state, $record) {
                                if (!$record) return null;

                                if ($record->page_id && $record->subcard) {
                                    return 'sub-' . $record->subcard->id;
                                }

                                if ($record->user_id && $record->user) {
                                    return 'user-' . $record->user->id;
                                }

                                return null;
                            })
                            ->afterStateHydrated(function (Set $set, $state, $record) {
                                if (!$record) return;

                                if ($record->page_id && $record->subcard) {
                                    $set('owner', 'sub-' . $record->subcard->id);
                                } elseif ($record->user_id && $record->user) {
                                    $set('owner', 'user-' . $record->user->id);
                                }
                            })
                            ->afterStateUpdated(function (Set $set, $state) {
                                if (!$state) return;

                                [$type, $id] = explode('-', $state);

                                if ($type === 'user') {
                                    // 👇 set hidden fields instead of saving directly
                                    $set('user_id', $id);
                                    $set('page_id', null);
                                } elseif ($type === 'sub') {
                                    // $set('page_id', $id);
                                    // $set('user_id', null);
                                    $sub = SubCard::find($id);
                                    if ($sub) {
                                        $set('page_id', $sub->id);
                                        $set('user_id', $sub->user_id); // <-- set the main user_id from subcard
                                    }
                                }
                            })
                            ->searchable()
                            ->required()
                            ->helperText(function () {
                                $settings = app(CardDefaultsSetting::class);

                                return new HtmlString(
                                    'Only <strong>' . e($settings->term_gold) . '</strong> & <strong>' . e($settings->term_platinum) . '</strong> users can add team(s). SubCards are also allowed.'
                                );
                            }),

                        Forms\Components\TextInput::make('title')
                            ->maxLength(300)
                            ->placeholder('ex: Presentation Page, Main Menu')
                            ->columnSpanFull()
                            ->required(),

                        Forms\Components\Textarea::make('description')
                            ->label('Description')
                            ->maxLength(1000)
                            ->columnSpanFull(),

                        Forms\Components\Toggle::make('is_active')
                            ->label('Active')
                            ->default(true)
                            ->columnSpanFull()
                            ->required(),

                        // Forms\Components\Hidden::make('type')
                        //     ->default('team'),

                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('title')
            ->modifyQueryUsing(
                fn(Builder $query) =>
                $query->orderBy('order', 'desc')
            )
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('owner')
                    ->label('Owner')
                    ->getStateUsing(function ($record) {
                        if (!$record) return 'N/A';

                        // If page_id is null, show main user
                        if (is_null($record->page_id) && $record->user) {
                            $user = $record->user;
                            return "{$user->firstname} {$user->lastname} - @$user->username (Main Card)";
                        }

                        // If page_id exists, show subcard
                        if ($record->page_id && $record->subcard) {
                            $sub = $record->subcard;
                            return "{$sub->firstname} {$sub->lastname} - @$sub->username (Sub Card)";
                        }

                        return 'N/A';
                    })
                    ->sortable()
                    // ->searchable(['user.firstname', 'user.lastname', 'user.username', 'subcard.firstname', 'subcard.lastname', 'subcard.username']),
                    ->searchable(query: function (Builder $query, string $search): Builder {
                        return $query
                            ->whereHas('user', function ($q) use ($search) {
                                $q->where('firstname', 'like', "%{$search}%")
                                    ->orWhere('lastname', 'like', "%{$search}%")
                                    ->orWhere('username', 'like', "%{$search}%");
                            })
                            ->orWhereHas('subcard', function ($q) use ($search) {
                                $q->where('firstname', 'like', "%{$search}%")
                                    ->orWhere('lastname', 'like', "%{$search}%")
                                    ->orWhere('username', 'like', "%{$search}%");
                            });
                    }),

                Tables\Columns\TextColumn::make('title'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->sortable()
                    ->formatStateUsing(fn($state) => $state ? Carbon::parse($state)->format('M d, Y') : null),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\Action::make('view_card')
                    ->label('View')
                    ->icon('heroicon-o-eye')
                    ->url(function ($record) {
                        if (!$record) return '#';

                        $baseUrl = app(\App\Settings\CardDefaultsSetting::class)->site_url;

                        if ($record->page_id && $record->subcard) {
                            return $baseUrl . $record->subcard->username . '/page/' . $record->slug;
                        }

                        if ($record->user) {
                            return $baseUrl . $record->user->username . '/page/' . $record->slug;
                        }

                        return '#';
                    })
                    ->openUrlInNewTab()
                    ->color('info'),
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            PageContentRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListPresentationPages::route('/'),
            'create' => Pages\CreatePresentationPage::route('/create'),
            'edit' => Pages\EditPresentationPage::route('/{record}/edit'),
        ];
    }
}
