<?php

namespace App\Filament\Resources\PresentationPageResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use Filament\Forms\Components\FileUpload;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Actions\Action;
use Malzariey\FilamentLexicalEditor\Enums\ToolbarItem;
use Malzariey\FilamentLexicalEditor\FilamentLexicalEditor;

class PageContentRelationManager extends RelationManager
{
    protected static string $relationship = 'page_content';
    protected static ?string $modelLabel = 'Content';

    protected array $types = [
        'button' => [
            'label' => 'Button',
            'field_label' => 'Link',
        ],
        'heading' => [
            'label' => 'Text Heading',
            'field_label' => 'Content',
        ],
        'html' => [
            'label' => 'Text Content',
            'field_label' => 'Content',
        ],
        'video' => [
            'label' => 'Video',
            'field_label' => 'Youtube or Vimeo Link',
        ],
        'image' => [
            'label' => 'Image',
            'field_label' => 'Upload Image',
            'upload_dir' => 'pages',
        ],
    ];

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                // Type selector
                Forms\Components\Select::make('type')
                    ->label('Element Type')
                    ->required()
                    ->reactive()
                    ->columnSpanFull()
                    ->options(array_map(fn($item) => $item['label'], $this->types))
                    ->default(fn($record) => $record?->type),

                // Title (only for button)
                Forms\Components\TextInput::make('title')
                    ->label('Title')
                    ->maxLength(255)
                    ->columnSpanFull()
                    ->required(fn(callable $get) => $get('type') === 'button')
                    ->visible(fn(callable $get, $livewire) => $get('type') === 'button')
                    ->afterStateHydrated(function ($component, $state, $record) {
                        if ($record && $record->type === 'button') {
                            $component->state($record->title);
                        }
                    }),

                // TextInput for button and video
                Forms\Components\TextInput::make('value')
                    ->label(fn(callable $get) => match ($get('type')) {
                        'button' => 'Link',
                        'heading' => 'Content',
                        'video' => 'Youtube or Vimeo Link',
                        default => 'Value',
                    })
                    ->url(fn(callable $get) => in_array($get('type'), ['button', 'video']))
                    ->columnSpanFull()
                    ->required(fn(callable $get) => in_array($get('type'), ['heading', 'button', 'video']))
                    ->visible(fn(callable $get, $livewire) => in_array($get('type'), ['heading', 'button', 'video']))
                    ->afterStateHydrated(function ($component, $state, $record) {
                        if ($record && in_array($record->type, ['heading', 'button', 'video'])) {
                            $component->state($record->value);
                            // $component->state([$record->value]);
                        }
                    }),

                FilamentLexicalEditor::make('value')
                    ->columnSpanFull()
                    ->reactive()
                    ->live()
                    ->label('Text Content')
                    ->enabledToolbars([
                        ToolbarItem::BOLD,
                        ToolbarItem::ITALIC,
                        ToolbarItem::UNDERLINE,
                        ToolbarItem::LEFT,
                        ToolbarItem::CENTER,
                        ToolbarItem::RIGHT,
                        ToolbarItem::JUSTIFY,
                        ToolbarItem::START,
                        ToolbarItem::END,
                        ToolbarItem::BULLET,
                        ToolbarItem::NUMBERED,
                    ])
                    ->required(fn(callable $get) => $get('type') === 'html')
                    ->hidden(fn(callable $get, $livewire) => $get('type') != 'html')
                    ->afterStateHydrated(function ($component, $state, $record) {
                        if ($record && $record->type === 'html') {
                            $component->state($record->value);
                        }
                    }),


                // RichEditor for HTML
                // Forms\Components\RichEditor::make('value')
                //     ->label('Text Content')
                //     ->columnSpanFull()
                //     ->disableGrammarly()
                //     ->toolbarButtons([
                //         // 'blockquote',
                //         'bold',
                //         'bulletList',
                //         // 'h1',
                //         // 'h2',
                //         // 'h3',
                //         'italic',
                //         'orderedList',
                //         // 'redo',
                //         'underline',
                //         // 'undo',
                //     ])
                //     ->required(fn(callable $get) => $get('type') === 'html')
                //     ->visible(fn(callable $get, $livewire) => $get('type') === 'html')
                //     ->afterStateHydrated(function ($component, $state, $record) {
                //         if ($record && $record->type === 'html') {
                //             $component->state($record->value);
                //         }
                //     }),


                // FileUpload::make('value')
                //     ->label('Upload Image')
                //     ->directory('pages')
                //     ->image()
                //     ->imageEditor()
                //     ->columnSpanFull()
                //     // ->imageQuality(75)
                //     ->imageResizeTargetWidth(520) // resize width
                //     ->imageResizeTargetHeight(520) // resize height
                //     ->required(fn($get) => $get('type') === 'image')
                //     ->visible(fn($get) => $get('type') === 'image')
                //     ->afterStateHydrated(function ($component, $state, $record) {
                //         if ($record && $record->type === 'image' && $record->value) {
                //             // Wrap in array because Filament expects an array for the state
                //             $component->state([$record->value]);
                //         }
                //     })
                //     ->getUploadedFileNameForStorageUsing(fn($file) => Str::random(40) . '.' . $file->getClientOriginalExtension()),

                FileUpload::make('value')
                    ->label('Upload Image')
                    ->directory('pages')
                    ->image()
                    ->imageEditor()
                    ->columnSpanFull()
                    ->imageResizeMode('cover')
                    ->imageResizeTargetWidth(720)
                    ->imageResizeTargetHeight(720)
                    ->required(fn($get) => $get('type') === 'image')
                    ->visible(fn($get) => $get('type') === 'image')
                    ->afterStateHydrated(function ($component, $state, $record) {
                        if ($record && $record->type === 'image' && $record->value) {
                            $component->state([$record->value]);
                        }
                    })
                    ->getUploadedFileNameForStorageUsing(fn($file) => Str::random(40) . '.' . $file->getClientOriginalExtension()),

            ]);
    }

    public static function getScripts(): array
    {
        return [
            <<<JS
        document.addEventListener('livewire:load', () => {

            const reloadLexicalEditors = () => {
                document.querySelectorAll('.lexical-editor').forEach(el => {
                    // Only reinit if editor is visible and not already initialized
                    const isVisible = el.offsetParent !== null;
                    if (isVisible && window.initLexicalEditor && !el.__lexicalInitialized) {
                        window.initLexicalEditor(el);
                        el.__lexicalInitialized = true;
                    }
                });
            };

            // Initial mount (in case not lazy)
            setTimeout(reloadLexicalEditors, 600);

            // Re-run when Livewire finishes rendering or lazy components load
            Livewire.hook('message.processed', () => {
                setTimeout(reloadLexicalEditors, 400);
            });

            // Also observe visibility changes (fix for lazy tabs, collapsibles, etc.)
            const observer = new MutationObserver(() => {
                reloadLexicalEditors();
            });
            observer.observe(document.body, { attributes: true, childList: true, subtree: true });
        });
        JS,
        ];
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('title')
            ->modifyQueryUsing(
                fn(Builder $query) =>
                $query->orderBy('order', 'asc')
            )
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success')
                    ->offColor('danger'),

                Tables\Columns\TextColumn::make('title')
                    ->label('Title'),

                Tables\Columns\TextColumn::make('type')
                    ->formatStateUsing(fn($state) => ucfirst($state)),

                Tables\Columns\TextColumn::make('value')
                    ->label('Value')
                    ->formatStateUsing(function ($state, $record) {
                        return match ($record->type) {
                            'image' => "<img src='" . asset('storage/' . $state) . "' class='h-10 w-10 object-cover rounded' />",
                            'button', 'video' => "<a href='{$state}' target='_blank'>{$state}</a>",
                            'html' => Str::limit(strip_tags($state), 50),
                            default => $state,
                        };
                    })
                    ->html(),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->sortable()
                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),
            ])
            ->filters([])
            ->headerActions([
                Tables\Actions\CreateAction::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
