<?php

namespace App\Filament\Resources\ProductResource\Pages;

use App\Filament\Resources\ProductResource;
use App\Models\Product;
use App\Models\User;
use App\Models\SubCard;
use Filament\Resources\Pages\Page;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Str;
use Filament\Actions\Action;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use Filament\Notifications\Notification;

class ProductsByOwnerDetail extends Page implements Tables\Contracts\HasTable
{
    use Tables\Concerns\InteractsWithTable;

    protected static string $resource = ProductResource::class;
    protected static string $view = 'filament.resources.product-resource.pages.products-by-owner-detail';

    public ?string $ownerKey = null;
    public ?string $ownerLabel = null;

    protected static ?string $navigationLabel = null;
    protected static ?string $navigationIcon = null;
    protected static ?string $title = 'Products';

    public static function shouldRegisterNavigation(array $parameters = []): bool
    {
        return false;
    }

    public function mount(string $ownerKey)
    {
        $this->ownerKey = $ownerKey;

        if (Str::startsWith($ownerKey, 'user-')) {
            $user = User::find(substr($ownerKey, 5));
            $this->ownerLabel = $user
                ? "{$user->firstname} {$user->lastname} - @$user->username (Main Card)"
                : 'Unknown Owner';
        } elseif (Str::startsWith($ownerKey, 'sub-')) {
            $sub = SubCard::find(substr($ownerKey, 4));
            $this->ownerLabel = $sub
                ? "{$sub->firstname} {$sub->lastname} - @$sub->username (Sub Card)"
                : 'Unknown Owner';
        }
    }

    public function table(Table $table): Table
    {
        return $table
            // ->query(function (): Builder {
            //     if (Str::startsWith($this->ownerKey, 'user-')) {
            //         $userId = substr($this->ownerKey, 5);
            //         return Product::query()->where('user_id', $userId);
            //     } elseif (Str::startsWith($this->ownerKey, 'sub-')) {
            //         $pageId = substr($this->ownerKey, 4);
            //         return Product::query()->where('page_id', $pageId);
            //     }
            //     return Product::query()->whereRaw('0=1');
            // })
            ->query(function (): Builder {
                if (Str::startsWith($this->ownerKey, 'user-')) {
                    $userId = substr($this->ownerKey, 5);
                    return Product::query()
                        ->where('user_id', $userId)
                        ->orderByDesc('created_at');
                } elseif (Str::startsWith($this->ownerKey, 'sub-')) {
                    $pageId = substr($this->ownerKey, 4);
                    return Product::query()
                        ->where('page_id', $pageId)
                        ->orderByDesc('created_at');
                }

                return Product::query()->whereRaw('0=1');
            })

            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success')
                    ->offColor('danger'),

                Tables\Columns\TextColumn::make('product_name')->label('Product Name')->searchable(),
                Tables\Columns\TextColumn::make('type'),
                Tables\Columns\TextColumn::make('price')->money(),
                Tables\Columns\TextColumn::make('stock'),
                Tables\Columns\TextColumn::make('created_at')->date('M d, Y'),
            ])
            ->actions([
                Tables\Actions\Action::make('view')
                    ->label('View')
                    ->icon('heroicon-o-eye')
                    ->color('info')
                    // ->url(fn($record) => route('filament.admin.resources.products.view', ['record' => $record->id])),
                    ->url(function ($record) {
                        if (!$record) return '#';

                        $baseUrl = app(\App\Settings\CardDefaultsSetting::class)->site_url;

                        // if ($record->page_id && $record->subcard) {
                        //     return $baseUrl . $record->subcard->username . '/products/' . $record->id;
                        // }

                        // if ($record->user) {
                        //     return $baseUrl . $record->user->username . '/products/' . $record->id;
                        // }

                        if ($record->type != "subscription") {
                            if ($record->page_id && $record->subcard) {
                                return $baseUrl . $record->subcard->username . '/products/' . $record->id;
                            }

                            if ($record->user) {
                                return $baseUrl . $record->user->username . '/products/' . $record->id;
                            }
                        } else {
                            if ($record->page_id && $record->subcard) {
                                return $baseUrl . $record->subcard->username . '/subscription/' . $record->id;
                            }

                            if ($record->user) {
                                return $baseUrl . $record->user->username . '/subscription/' . $record->id;
                            }
                        }

                        return '#';
                    }),

                Tables\Actions\Action::make('edit')
                    ->label('Edit')
                    ->icon('heroicon-o-pencil')
                    ->url(fn($record) => route('filament.admin.resources.products.edit', ['record' => $record->id])),

                Tables\Actions\Action::make('customAction')
                    ->icon('heroicon-o-document-duplicate')
                    ->label('Duplicate')
                    ->color('warning')
                    ->requiresConfirmation()
                    ->action(function (Model $record) {
                        DB::transaction(function () use ($record) {

                            // Helper function to safely duplicate a file
                            $duplicateFile = function (?string $filePath) {
                                if (!$filePath || !Storage::disk('public')->exists($filePath)) {
                                    return null;
                                }

                                $directory = dirname($filePath);
                                $extension = pathinfo($filePath, PATHINFO_EXTENSION);
                                $originalName = pathinfo($filePath, PATHINFO_FILENAME);

                                $maxBaseLength = 50;
                                $shortName = Str::limit($originalName, $maxBaseLength, '');
                                $newFileName = $shortName . '_' . Str::random(6) . '.' . $extension;
                                $newPath = $directory . '/' . $newFileName;

                                Storage::disk('public')->makeDirectory($directory);
                                Storage::disk('public')->copy($filePath, $newPath);

                                return $newPath;
                            };

                            // ✅ Duplicate the main product
                            $newProduct = $record->replicate();
                            $newProduct->product_name = $record->product_name . ' (Copy)';
                            $newProduct->featured_image = $duplicateFile($record->featured_image);
                            $newProduct->created_at = now();
                            $newProduct->updated_at = now();
                            $newProduct->save();

                            // ✅ Duplicate related media (if any)
                            if (method_exists($record, 'media') && $record->media()->exists()) {
                                foreach ($record->media()->orderBy('order')->get() as $media) {
                                    $newMedia = $media->replicate();

                                    // Duplicate image files only
                                    if ($media->type === 'image') {
                                        $newPath = $duplicateFile($media->path);
                                        if (!$newPath) {
                                            continue;
                                        }
                                        $newMedia->path = $newPath;
                                    } else {
                                        $newMedia->path = $media->path; // reuse path for non-image
                                    }

                                    $newMedia->product_id = $newProduct->id;
                                    $newMedia->order = $media->order;
                                    $newMedia->save();
                                }
                            }

                            Notification::make()
                                ->title('Success')
                                ->body('Product duplicated successfully.')
                                ->success()
                                ->send();
                        });
                    }),

            ]);
    }

    protected function getHeaderActions(): array
    {
        return [
            Action::make('create_product')
                ->label('Create Product')
                ->icon('heroicon-o-plus')
                ->color('success')
                ->url(fn() => route('filament.admin.resources.products.create', [
                    'ownerKey' => $this->ownerKey,
                ])),
        ];
    }
}
