<?php

namespace App\Filament\Resources\ReferralResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use App\Models\Referral;
use App\Models\User;
use Carbon\Carbon;
use App\Settings\CardDefaultsSetting;

class ReferralsRelationManager extends RelationManager
{
    protected static string $relationship = 'referralsAsMain';
    protected static ?string $title = 'Referrals';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('post')
                    ->required()
                    ->maxLength(255),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->query(User::join('referrals', 'users.id', '=', 'referrals.referred_id')
                ->where('referrals.referrer_id_main', '=', $this->ownerRecord->id)
                // ->select('users.*', 'referrals.referred_id')
                ->select('users.*', 'users.id as user_id', 'referrals.referred_id')
                ->with('referralsAsMain.referredUser'))
            ->columns([
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean(),

                Tables\Columns\TextColumn::make('username')
                    ->searchable()
                    ->badge()
                    ->label('Card Link')
                    ->color('primary')
                    ->url(function ($record) {
                        $siteUrl = app(CardDefaultsSetting::class)->site_url;
                        return "{$siteUrl}{$record->username}";
                    })
                    ->formatStateUsing(fn($state) => '@' . $state)
                    ->openUrlInNewTab(),


                Tables\Columns\TextColumn::make('firstname')
                    ->label('Fullname')
                    ->formatStateUsing(function ($state, $record) {
                        return $record->firstname . ' ' . $record->lastname;
                    }),

                Tables\Columns\TextColumn::make('user_type')
                    ->badge()
                    ->label('Card Type')
                    ->formatStateUsing(function ($state) {
                        $settings = app(CardDefaultsSetting::class);

                        return match ($state) {
                            'Basic' => $settings->term_basic,
                            'Gold' => $settings->term_gold,
                            'Platinum' => $settings->term_platinum,
                            'setup_incomplete' => 'Setup Incomplete',
                            default => $state,
                        };
                    })
                    ->color(fn(string $state): string => match ($state) {
                        'Free' => 'info',
                        'Basic' => 'success',
                        'Gold' => 'warning',
                        'Platinum' => 'gray',
                        'Sponsored' => 'danger',
                        'Influencer' => 'influencer',
                    }),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Registration Date')
                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),
            ])
            ->filters([
                //
            ])
            ->headerActions([])
            ->actions([

                Tables\Actions\Action::make('edit_user')
                    ->label('Edit')
                    ->icon('heroicon-o-pencil')
                    ->url(fn($record) => \App\Filament\Resources\UserResource::getUrl('edit', ['record' => $record->user_id]))
                    ->color('primary'),

                Tables\Actions\Action::make('view_card')
                    ->label('View')
                    ->icon('heroicon-o-eye')
                    ->url(fn($record) => app(CardDefaultsSetting::class)->site_url . $record->username)
                    ->openUrlInNewTab()
                    ->color('info'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
