<?php

namespace App\Filament\Resources;

use App\Filament\Resources\SaleResource\Pages;
use App\Filament\Resources\SaleResource\RelationManagers;
use App\Models\Sale;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Notifications\Notification;
use App\Models\User;
use Filament\Forms\Components\Select;
use Illuminate\Support\HtmlString;
use Filament\Forms\Set;
use App\Settings\CardDefaultsSetting;
use Filament\Forms\Components\FileUpload;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use App\Models\SubCard;
use Carbon\Carbon;
use Intervention\Image\Facades\Image;
use App\Filament\Resources\SaleResource\RelationManagers\SaleVideosRelationManager;
use App\Filament\Resources\SaleResource\RelationManagers\SaleImagesRelationManager;

class SaleResource extends Resource
{
    protected static ?string $model = Sale::class;

    protected static ?string $navigationIcon = 'heroicon-o-shopping-bag';

    protected static ?string $navigationGroup = 'Client Management';
    protected static ?string $navigationLabel = 'Sales Page';
    protected static ?int $navigationSort = 3;

    // hide this resource
    public static function shouldRegisterNavigation(): bool
    {
        return false;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make("Sale's Page Details")
                    ->schema([

                        // 👇 Added hidden fields so Filament will save them
                        Forms\Components\Hidden::make('user_id'),
                        Forms\Components\Hidden::make('page_id'),

                        Select::make('owner')
                            ->label('Owner')
                            ->placeholder('Search Client or SubCard')
                            ->getSearchResultsUsing(function (string $search): array {
                                // --- Users ---
                                $users = User::query()
                                    ->where(function ($query) use ($search) {
                                        if (strpos($search, '@') === 0 && strlen($search) > 1) {
                                            $query->where('username', 'like', "%" . substr($search, 1) . "%");
                                        } else {
                                            $query->where('firstname', 'like', "%{$search}%")
                                                ->orWhere('lastname', 'like', "%{$search}%")
                                                ->orWhere('username', 'like', "%{$search}%");
                                        }
                                    })
                                    ->whereIn('user_type', ['Platinum', 'Gold'])
                                    ->limit(25)
                                    ->get()
                                    ->mapWithKeys(fn($user) => [
                                        "user-{$user->id}" => "{$user->firstname} {$user->lastname} - @$user->username (User)"
                                    ]);

                                // --- SubCards ---
                                $subCards = SubCard::query()
                                    ->where(function ($query) use ($search) {
                                        $query->where('firstname', 'like', "%{$search}%")
                                            ->orWhere('lastname', 'like', "%{$search}%")
                                            ->orWhere('username', 'like', "%{$search}%");
                                    })
                                    ->limit(25)
                                    ->get()
                                    ->mapWithKeys(fn($sub) => [
                                        "sub-{$sub->id}" => "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)"
                                    ]);

                                return $users->toArray() + $subCards->toArray();
                            })
                            ->getOptionLabelUsing(function ($value) {
                                if (!$value) return '';

                                [$type, $id] = explode('-', $value);

                                if ($type === 'user') {
                                    $user = User::find($id);
                                    return $user ? "{$user->firstname} {$user->lastname} - @$user->username (User)" : '';
                                }

                                if ($type === 'sub') {
                                    $sub = SubCard::find($id);
                                    return $sub ? "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)" : '';
                                }

                                return '';
                            })
                            ->formatStateUsing(function ($state, $record) {
                                if (!$record) return null;

                                if ($record->page_id && $record->subcard) {
                                    return 'sub-' . $record->subcard->id;
                                }

                                if ($record->user_id && $record->user) {
                                    return 'user-' . $record->user->id;
                                }

                                return null;
                            })
                            ->afterStateHydrated(function (Set $set, $state, $record) {
                                if (!$record) return;

                                if ($record->page_id && $record->subcard) {
                                    $set('owner', 'sub-' . $record->subcard->id);
                                } elseif ($record->user_id && $record->user) {
                                    $set('owner', 'user-' . $record->user->id);
                                }
                            })
                            ->afterStateUpdated(function (Set $set, $state) {
                                if (!$state) return;

                                [$type, $id] = explode('-', $state);

                                if ($type === 'user') {
                                    $user = User::find($id);
                                    if ($user) {
                                        if ($user->user_type === "Platinum" && $user->sales()->count() >= 6) {
                                            Notification::make()
                                                ->title('Limit reached')
                                                ->body('Platinum card holders can only add up to 6 sales page!')
                                                ->warning()
                                                ->send();
                                            $set('owner', null);
                                            return;
                                        }
                                        if ($user->user_type === "Gold" && $user->sales()->count() >= 3) {
                                            Notification::make()
                                                ->title('Limit reached')
                                                ->body('Gold card holders can only add up to 3 sales page!')
                                                ->warning()
                                                ->send();
                                            $set('owner', null);
                                            return;
                                        }
                                    }

                                    // 👇 set hidden fields instead of saving directly
                                    $set('user_id', $id);
                                    $set('page_id', null);
                                } elseif ($type === 'sub') {
                                    $set('page_id', $id);
                                    $set('user_id', null);
                                }
                            })
                            ->searchable()
                            ->required()
                            ->helperText(new HtmlString('Only <strong>Gold</strong> & <strong>Platinum</strong> users can add sales page(s). SubCards are also allowed.')),

                        FileUpload::make('thumbnail')
                            ->required()
                            ->label('Upload Thumbnail')
                            ->placeholder('Upload Thumbnail')
                            ->directory('sales')
                            ->uploadingMessage('Uploading. Please wait...')
                            ->image()
                            ->imageEditor()
                            ->columnSpanFull()
                            ->imageResizeMode('cover')
                            ->imageCropAspectRatio('1:1')
                            ->imageResizeTargetWidth('600')
                            ->imageResizeTargetHeight('600')
                            ->getUploadedFileNameForStorageUsing(
                                fn(TemporaryUploadedFile $file): string =>
                                (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                            )
                            ->saveUploadedFileUsing(function (TemporaryUploadedFile $file) {
                                $image = Image::make($file->getRealPath())
                                    ->resize(800, null, function ($constraint) {
                                        $constraint->aspectRatio();
                                        $constraint->upsize();
                                    })
                                    ->encode('jpg', 75);

                                $filename = Str::random(40) . '.jpg';

                                \Storage::disk('public')->put('sales/' . $filename, (string)$image);

                                return 'sales/' . $filename;
                            })
                            ->imageEditorAspectRatios(['1:1']),

                        Forms\Components\TextInput::make('title')
                            ->maxLength(300)
                            ->placeholder('Ex: Gym Equipments Sale')
                            ->columnSpanFull()
                            ->required(),

                        Forms\Components\Textarea::make('description')
                            ->maxLength(2000)
                            ->columnSpanFull()
                            ->helperText('Write a few sentences about this page. Max character 2000.'),

                        Forms\Components\Toggle::make('is_active')
                            ->label('Active')
                            ->default(true)
                            ->columnSpanFull()
                            ->required(),
                    ]),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean()
                    ->searchable(),

                Tables\Columns\TextColumn::make('owner')
                    ->label('Owner')
                    ->getStateUsing(function ($record) {
                        if (!$record) return 'N/A';

                        if ($record->page_id && $record->subcard) {
                            $sub = $record->subcard;
                            return "{$sub->firstname} {$sub->lastname} - @$sub->username (Sub Card)";
                        }

                        if ($record->user_id && $record->user) {
                            $user = $record->user;
                            return "{$user->firstname} {$user->lastname} - @$user->username (Main Card)";
                        }

                        return 'N/A';
                    })
                    ->sortable()
                    ->searchable(['user.firstname', 'user.lastname', 'user.username', 'subcard.firstname', 'subcard.lastname', 'subcard.username']),

                Tables\Columns\TextColumn::make('title')
                    ->searchable(),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->sortable()
                    ->formatStateUsing(fn($state) => $state ? Carbon::parse($state)->format('M d, Y') : null),
            ])
            ->defaultSort('created_at', 'desc')
            ->actions([
                Tables\Actions\Action::make('view_card')
                    ->label('View')
                    ->icon('heroicon-o-eye')
                    ->url(function ($record) {
                        if (!$record) return '#';

                        $baseUrl = app(\App\Settings\CardDefaultsSetting::class)->site_url;

                        if ($record->page_id && $record->subcard) {
                            return $baseUrl . $record->subcard->username . '/sales/' . $record->id;
                        }

                        if ($record->user) {
                            return $baseUrl . $record->user->username . '/sales/' . $record->id;
                        }

                        return '#';
                    })
                    ->openUrlInNewTab()
                    ->color('info'),

                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            SaleVideosRelationManager::class,
            SaleImagesRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSales::route('/'),
            'create' => Pages\CreateSale::route('/create'),
            'edit' => Pages\EditSale::route('/{record}/edit'),
        ];
    }
}
