<?php

namespace App\Filament\Resources;

use App\Filament\Resources\SendCardResource\Pages;
use App\Filament\Resources\SendCardResource\RelationManagers;
use App\Models\SendCard;
use App\Models\SubCard;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Infolist;
use Filament\Infolists\Components;
use Filament\Infolists\Components\ImageEntry;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Storage;
use App\Settings\CardDefaultsSetting;

class SendCardResource extends Resource
{
    protected static ?string $model = SendCard::class;

    protected static ?string $navigationIcon = 'heroicon-o-rectangle-stack';

    protected static ?string $navigationGroup = 'Forms';

    protected static ?string $navigationLabel = 'Physical Cards';

    protected static ?string $modelLabel = 'Physical Card';

    protected static ?int $navigationSort = 3;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                //
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(
                fn(Builder $query) =>
                $query->orderBy('created_at', 'desc')
            )
            ->columns([
                Tables\Columns\TextColumn::make('email')
                    ->label('Email'),

                Tables\Columns\TextColumn::make('firstname')
                    ->label('Full Name')
                    ->formatStateUsing(function ($state, $record) {
                        return $record->firstname . ' ' . $record->lastname;
                    }),

            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\Action::make('convert_to_card')
                    ->label('Convert to Card')
                    ->icon('heroicon-o-credit-card')
                    ->color('success')
                    ->requiresConfirmation()
                    ->action(function ($record, $livewire) {

                        $email = $record->email;

                        // Check if email exists in Users or SubCards
                        $existsInUsers = User::where('email', $email)->exists();
                        $existsInSubCards = SubCard::where('email', $email)->exists();

                        if ($existsInUsers || $existsInSubCards) {
                            Notification::make()
                                ->title('Email already exists')
                                ->body("The email '{$email}' is already registered.")
                                ->danger()
                                ->send();
                            return;
                        }

                        // Generate a unique username
                        $username = strtolower(preg_replace('/\s+/', '', $record->firstname)) . '_' . uniqid();

                        // Handle profile picture duplication if it exists
                        $profilePicPath = $record->profile_pic;
                        $newProfilePicPath = null;

                        if ($profilePicPath && Storage::disk('public')->exists($profilePicPath)) {
                            $filename = basename($profilePicPath);
                            $newProfilePicPath = 'profile_pic/' . uniqid() . '_' . $filename;
                            Storage::disk('public')->copy($profilePicPath, $newProfilePicPath);
                        }

                        // Create the new user
                        $user = User::create([
                            'firstname' => $record->firstname,
                            'lastname' => $record->lastname,
                            'email' => $record->email,
                            'phone' => $record->phone,
                            'username' => $username,
                            'profile_pic' => $newProfilePicPath,
                            'password' => bcrypt(app(CardDefaultsSetting::class)->default_password), // optional default password
                        ]);

                        Notification::make()
                            ->title('Card successfully converted')
                            ->body("User {$user->firstname} {$user->lastname} was created successfully.")
                            ->success()
                            ->send();
                    }),

                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function infolist(Infolist $infolist): Infolist
    {
        return $infolist
            ->schema([
                Components\Section::make('Form Details')
                    ->schema([
                        ImageEntry::make('profile_pic')
                            ->label('')
                            ->width(250)
                            ->height(250)
                            ->square()
                            ->defaultImageUrl(asset('/img/profile-placeholder.jpg')),

                        Components\Group::make([
                            TextEntry::make('firstname')
                                ->label('Full Name')
                                ->formatStateUsing(fn($state, $record) => $record->firstname . ' ' . $record->lastname),

                            TextEntry::make('email')
                                ->label('Email'),

                            TextEntry::make('phone')
                                ->label('Phone'),
                        ]),
                    ])
                    ->columns(2),

                Components\Section::make('Physical Business Card')
                    ->schema([
                        ImageEntry::make('physical_card')
                            ->label('Front')
                            ->square()
                            ->width('100%')
                            ->height('100%')
                            ->extraImgAttributes([
                                'alt' => 'Business Card (Front)',
                                'loading' => 'lazy',
                            ]),

                        Components\Group::make([
                            ImageEntry::make('physical_card_back')
                                ->label('Back')
                                ->square()
                                ->width('100%')
                                ->height('100%')
                                ->extraImgAttributes([
                                    'alt' => 'Business Card (Back)',
                                    'loading' => 'lazy',
                                ]),
                        ]),
                    ])->columns(2),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSendCards::route('/'),
            'create' => Pages\CreateSendCard::route('/create'),
            'edit' => Pages\EditSendCard::route('/{record}/edit'),
            'view' => Pages\ViewSendCard::route('/{record}'),
        ];
    }
}
