<?php

namespace App\Filament\Resources;

use App\Filament\Resources\SocialResource\Pages;
use App\Filament\Resources\SocialResource\RelationManagers;
use App\Models\Social;
use App\Models\User;
use Filament\Forms;
use Filament\Forms\Components\Select;
use Filament\Forms\Form;
use Filament\Forms\Set;
use Filament\Notifications\Notification;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use Filament\Support\Enums\IconPosition;

class SocialResource extends Resource
{
    protected static ?string $model = Social::class;

    protected static ?string $navigationIcon = 'heroicon-o-heart';

    protected static ?string $navigationGroup = 'Client Management';

    protected static ?int $navigationSort = 3;

    // hide this resource
    public static function shouldRegisterNavigation(): bool
    {
        return false;
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Select::make('user_id')
                    ->label('Owner')
                    ->getSearchResultsUsing(
                        fn(string $search): array =>
                        User::where(function ($query) use ($search) {
                            $query->where('firstname', 'like', "%{$search}%")
                                ->orWhere('lastname', 'like', "%{$search}%");
                        })
                            ->limit(50)
                            ->pluck(DB::raw('CONCAT(firstname, " ", lastname)'), 'id')
                            ->toArray()
                    )
                    ->getOptionLabelUsing(
                        fn($value) =>
                        optional(User::find($value))->firstname . ' ' . optional(User::find($value))->lastname
                    )
                    ->searchable()
                    ->required()
                    ->afterStateUpdated(function (Set $set, $state) {
                        // Check if the selected user has 5 or more videos
                        $user = User::find($state);
                        // dd($user->user_type);
                        if ($user->user_type == "Free" && $user->socials()->count() >= 3) {
                            // Trigger a notification
                            Notification::make()
                                ->title('Limit reached')
                                ->body('Free card holders can only add up to 3 social links!')
                                ->warning()
                                ->send();
                            // Optionally, you can reset the state or show a validation error:
                            $set('user_id', null); // Reset the user_id if limit is reached
                        }
                    }),
                Forms\Components\Select::make('type')
                    ->required()
                    ->label('Social Media')
                    ->options([
                        'facebook' => 'Facebook',
                        'x' => 'X',
                        'linkedin' => 'Linkedin',
                        'instagram' => 'Instagram',
                        'tiktok' => 'Tiktok',
                        'youtube' => 'Youtube',
                        'clubhouse' => 'Clubhouse',
                        'soundcloud' => 'Sound Cloud',
                        'apple_podcasts' => 'Apple Podcasts',
                        'spotify' => 'Spotify',
                        'discord' => 'Discord',
                    ]),
                Forms\Components\TextInput::make('link')
                    ->required()
                    ->columnSpanFull()
                    ->maxLength(255),
                Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
                    ->columnSpanFull(),
                // Forms\Components\TextInput::make('order')
                //     ->numeric()
                //     ->default(null),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean(),
                Tables\Columns\TextColumn::make('user.firstname')
                    ->label('Owner')
                    ->getStateUsing(function ($record) {
                        return $record->user ? $record->user->firstname . ' ' . $record->user->lastname : 'N/A';
                    })
                    ->sortable()
                    ->searchable(['firstname', 'lastname']),
                Tables\Columns\TextColumn::make('type')
                    ->searchable()
                    ->formatStateUsing(fn($state) => ucfirst($state))
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    ->url(fn($record) => $record->link)
                    ->openUrlInNewTab(),
                // Tables\Columns\TextColumn::make('link')
                //     ->searchable()
                //     ->icon('heroicon-m-arrow-top-right-on-square')
                //     ->iconPosition(IconPosition::After)
                //     ->url(fn($record) => $record->link)
                //     ->openUrlInNewTab(),
                // Tables\Columns\TextColumn::make('link')
                //     ->searchable()
                //     ->limit(30)
                //     ->tooltip(function (TextColumn $column): ?string {
                //         $state = $column->getState();

                //         if (strlen($state) <= $column->getCharacterLimit()) {
                //             return null;
                //         }

                //         // Only render the tooltip if the column content exceeds the length limit.
                //         return $state;
                //     })
                // Tables\Columns\TextColumn::make('order')
                //     ->numeric()
                //     ->sortable(),
            ])
            ->filters([
                Filter::make('is_active')
                    ->label('Active')
                    ->query(fn(Builder $query) => $query->where('is_active', true)),
                SelectFilter::make('type')
                    ->label('Type')
                    ->options([
                        'facebook' => 'Facebook',
                        'x' => 'X',
                        'linkedin' => 'Linkedin',
                        'instagram' => 'Instagram',
                        'tiktok' => 'Tiktok',
                        'youtube' => 'Youtube',
                        'clubhouse' => 'Clubhouse',
                        'soundcloud' => 'Sound Cloud',
                        'apple_podcasts' => 'Apple Podcasts',
                        'spotify' => 'Spotify',
                        'discord' => 'Discord',
                    ]),
            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                // Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->with('user');
    }


    public static function getPages(): array
    {
        return [
            'index' => Pages\ListSocials::route('/'),
            'create' => Pages\CreateSocial::route('/create'),
            'view' => Pages\ViewSocial::route('/{record}'),
            'edit' => Pages\EditSocial::route('/{record}/edit'),
        ];
    }
}
