<?php

namespace App\Filament\Resources\SubCardsResource\RelationManagers;

use App\Settings\CardDefaultsSetting;
use App\Models\Event;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Database\Eloquent\Model;
use Filament\Tables\Actions\Action;
use Filament\Support\Enums\IconPosition;
use Filament\Forms\Components\FileUpload;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use Filament\Notifications\Notification;
use Filament\Tables\Actions\CreateAction;
use Filament\Tables\Actions\EditAction;
use Intervention\Image\Facades\Image;
use Malzariey\FilamentLexicalEditor\Enums\ToolbarItem;
use Malzariey\FilamentLexicalEditor\FilamentLexicalEditor;
use Kahusoftware\FilamentCkeditorField\CKEditor;

class EventsRelationManager extends RelationManager
{
    protected static string $relationship = 'events';

    public function form(Form $form): Form
    {
        $contacts = $this->ownerRecord->buttons()->where('view', 'contacts')->get();

        return $form
            ->schema([

                // Forms\Components\Select::make('event_type')
                //     ->label('Event Type')
                //     ->columnSpanFull()
                //     ->default('Photo')
                //     ->options([
                //         'Photo' => 'Photo',
                //         'Video' => 'Video',
                //     ])
                //     ->reactive(), // 🔑 this triggers updates

                FileUpload::make('event_poster')
                    ->label("Event Poster")
                    ->placeholder('Upload Event Poster')
                    ->directory('events')
                    ->disk('public')
                    ->image()
                    ->imageEditor()
                    ->imageResizeMode('cover')
                    ->columnSpanFull()
                    ->required()
                    // ->required(fn($get) => $get('event_type') === 'Photo')
                    // ->visible(fn($get) => $get('event_type') === 'Photo')
                    ->getUploadedFileNameForStorageUsing(
                        fn(TemporaryUploadedFile $file): string =>
                        (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                    )
                    ->saveUploadedFileUsing(function (TemporaryUploadedFile $file) {
                        // Resize & compress
                        $image = Image::make($file->getRealPath())
                            ->resize(800, null, function ($constraint) {
                                $constraint->aspectRatio();
                                $constraint->upsize();
                            })
                            ->encode('jpg', 75);

                        // Generate unique filename
                        $filename = Str::random(40) . '.jpg';

                        // Save to storage
                        \Storage::disk('public')->put('events/' . $filename, (string)$image);

                        // Return path for Filament to store in DB
                        return 'events/' . $filename;
                    }),

                Forms\Components\Toggle::make('poster_active')
                    ->label('Poster Visible on Page')
                    ->default(true)
                    ->columnSpanFull(),

                Forms\Components\Toggle::make('ad_active')
                    ->label('Ad Visible on Page')
                    ->default(false)
                    ->columnSpanFull(),

                Forms\Components\TextInput::make('title')
                    ->columnSpanFull()
                    ->required()
                    ->maxLength(255),

                Forms\Components\Toggle::make('title_active')
                    ->label('Title Visible on Page')
                    ->default(true)
                    ->columnSpanFull(),

                Forms\Components\DatePicker::make('start_datetime')
                    ->label('Event Date')
                    ->columnSpanFull(),

                // Forms\Components\Section::make('Description')
                //     ->schema([
                //         FilamentLexicalEditor::make('description')
                //             ->label('Write the details about the event')
                //             ->columnSpanFull()
                //             ->enabledToolbars([
                //                 ToolbarItem::BOLD,
                //                 ToolbarItem::ITALIC,
                //                 ToolbarItem::UNDERLINE,
                //                 ToolbarItem::LINK,
                //                 ToolbarItem::LEFT,
                //                 ToolbarItem::CENTER,
                //                 ToolbarItem::RIGHT,
                //                 ToolbarItem::JUSTIFY,
                //                 ToolbarItem::BULLET,
                //                 ToolbarItem::NUMBERED,
                //             ])
                //             ->extraAttributes(['class' => 'mb-6'])
                //             ->helperText('Write a few sentences about the event. Max character 5000.')
                //             // ->dehydrateStateUsing(fn($state) => $state ? $state : null) // ✅ ensures saving
                //             ->afterStateHydrated(function ($state, $set) {
                //                 $plainText = strip_tags($state ?? '');
                //             })
                //             // Update counter on typing
                //             ->afterStateUpdated(function ($state, $set) {
                //                 $plainText = strip_tags($state ?? '');
                //             })
                //             ->rule(fn($get) => function ($attribute, $value, $fail) {
                //                 $plainText = strip_tags($value ?? '');
                //                 if (Str::length($plainText) > 5000) {
                //                     $fail('The long bio must not exceed 5000 characters.');
                //                 }
                //             }),
                //     ])
                //     ->columnSpanFull()
                //     ->columns(1),

                Forms\Components\Repeater::make('event_video_link')
                    ->label('Event Video Link(s)')
                    ->columns(1)
                    ->schema([
                        Forms\Components\TextInput::make('link')
                            ->label('Video Link')
                            ->url()
                            // ->required()
                            ->maxLength(255),
                    ])
                    // ->default([['link' => '']])
                    ->default([])
                    // ->minItems(1)
                    ->maxItems(10)
                    ->columnSpanFull()
                    // Only extract 'link' values for saving
                    // ->dehydrateStateUsing(fn($state) => array_map(fn($item) => $item['link'] ?? null, $state))
                    ->dehydrateStateUsing(function ($state) {
                        if (! is_array($state)) {
                            return [];
                        }

                        return array_values(
                            array_filter(
                                array_map(
                                    fn($item) => $item['link'] ?? null,
                                    $state
                                )
                            )
                        );
                    })
                    ->afterStateHydrated(function ($state, $set, $record) {
                        if (!$record || empty($record->event_video_link)) return;

                        $links = is_string($record->event_video_link) ? json_decode($record->event_video_link, true) : $record->event_video_link;

                        if (!is_array($links)) {
                            $set('event_video_link', []);
                            return;
                        }

                        // Map strings back to Repeater format
                        $set('event_video_link', array_map(fn($link) => ['link' => $link], $links));
                    }),

                Forms\Components\TextInput::make('button_title')
                    ->required()
                    ->maxLength(255),

                Forms\Components\TextInput::make('button_link')
                    ->label('Button Link')
                    ->url()
                    ->required()
                    ->maxLength(255),

                Forms\Components\Repeater::make('optional_cta')
                    ->reactive()
                    ->label('Optional Call to Action Buttons')
                    ->columnSpanFull()
                    ->schema([
                        Forms\Components\TextInput::make('title')
                            ->label('Button Title')
                            ->required(),

                        Forms\Components\TextInput::make('link')
                            ->label('Button Link')
                            ->url()
                            ->required(),
                    ])
                    ->reorderable(),

                Forms\Components\TextInput::make('partner_link')
                    ->label("Partnership link")
                    ->columnSpanFull()
                    ->maxLength(255),

                Forms\Components\TextInput::make('shop_link')
                    ->label("External shop link. Shop page by default")
                    ->columnSpanFull()
                    ->maxLength(255),

                Forms\Components\Toggle::make('shop_active')
                    ->label('Shop Button Visible on Page')
                    ->default(false)
                    ->columnSpanFull(),

                Forms\Components\Toggle::make('contact_active')
                    ->label('Contact Points Visible on Page')
                    ->default(true)
                    ->columnSpanFull(),

                $this->getContactToggles($contacts),

                Forms\Components\RichEditor::make('description')
                    ->disableGrammarly()
                    ->label('Write the details about the event')
                    ->columnSpanFull()
                    ->helperText('Write a few sentences about the event. Max character 5000.')
                    ->toolbarButtons([
                        // 'blockquote',
                        'bold',
                        'bulletList',
                        // 'h1',
                        // 'h2',
                        // 'h3',
                        'italic',
                        'link',
                        'orderedList',
                        // 'redo',
                        // 'underline',
                        // 'undo',
                    ])
                    ->helperText('Write a few sentences about the event. Max character 5000.')
                    // ->dehydrateStateUsing(fn($state) => $state ? $state : null) // ✅ ensures saving
                    ->afterStateHydrated(function ($state, $set) {
                        $plainText = strip_tags($state ?? '');
                    })
                    // Update counter on typing
                    ->afterStateUpdated(function ($state, $set) {
                        $plainText = strip_tags($state ?? '');
                    })
                    ->rule(fn($get) => function ($attribute, $value, $fail) {
                        $plainText = strip_tags($value ?? '');
                        if (Str::length($plainText) > 5000) {
                            $fail('The description must not exceed 5000 characters.');
                        }
                    }),

                // FilamentLexicalEditor::make('description')
                //     ->label('Write the details about the event')
                //     ->columnSpanFull()
                //     ->enabledToolbars([
                //         ToolbarItem::BOLD,
                //         ToolbarItem::ITALIC,
                //         ToolbarItem::UNDERLINE,
                //         ToolbarItem::LINK,
                //         ToolbarItem::LEFT,
                //         ToolbarItem::CENTER,
                //         ToolbarItem::RIGHT,
                //         ToolbarItem::JUSTIFY,
                //         ToolbarItem::BULLET,
                //         ToolbarItem::NUMBERED,
                //     ])
                //     ->extraAttributes(['class' => 'mb-6'])
                //     ->helperText('Write a few sentences about the event. Max character 5000.')
                //     // ->dehydrateStateUsing(fn($state) => $state ? $state : null) // ✅ ensures saving
                //     ->afterStateHydrated(function ($state, $set) {
                //         $plainText = strip_tags($state ?? '');
                //     })
                //     // Update counter on typing
                //     ->afterStateUpdated(function ($state, $set) {
                //         $plainText = strip_tags($state ?? '');
                //     })
                //     ->rule(fn($get) => function ($attribute, $value, $fail) {
                //         $plainText = strip_tags($value ?? '');
                //         if (Str::length($plainText) > 5000) {
                //             $fail('The description must not exceed 5000 characters.');
                //         }
                //     }),


                // Forms\Components\Toggle::make('is_active')
                //     ->label('Active')
                //     ->default(true)
                //     ->columnSpanFull()
                //     ->required(),
            ]);
    }

    protected function getContactToggles($contacts)
    {
        // Only keep active contacts
        $activeContacts = $contacts->filter(fn($contact) => $contact->is_active);

        return Forms\Components\Section::make('Contact Points')
            ->schema(
                $activeContacts->map(function ($contact) {
                    return Forms\Components\Toggle::make("contact_settings.{$contact->id}")
                        ->label($contact->title)
                        ->default(false); // fallback for new records
                })->toArray()
            )
            ->columns(2)
            ->afterStateHydrated(
                function ($state, $set, $record) use ($activeContacts) {
                    if (!$record) return;

                    // Decode JSON if stored as string
                    $contactSettings = $record->contact_settings;
                    if (is_string($contactSettings)) {
                        $contactSettings = json_decode($contactSettings, true) ?? [];
                    }

                    foreach ($activeContacts as $contact) {
                        $set("contact_settings.{$contact->id}", $contactSettings[$contact->id] ?? false);
                    }
                }
            )
            ->visible(fn($get) => $get('contact_active'))
            ->reactive();
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->where('page_id', $livewire->ownerRecord->id)->orderBy('order')
            )
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('title'),
                Tables\Columns\TextColumn::make('button_link')
                    ->label('Button Link')
                    ->searchable()
                    ->limit(40)
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    ->url(fn($record) => $record->button_link)
                    ->openUrlInNewTab(),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->before(function (CreateAction $action, $livewire) {
                        // Access form data from the action
                        $data = $action->getFormData();

                        $eventCount = $livewire->ownerRecord->events()->count();
                        $userType   = $livewire->ownerRecord->user?->user_type; // 🔑 Get from users table

                        $limits = [
                            'Platinum' => 20,
                            'Gold' => 3,
                            'Basic' => 1,
                        ];

                        if (isset($limits[$userType]) && $eventCount >= $limits[$userType]) {
                            Notification::make()
                                ->warning()
                                ->title("{$userType} card holders can only add up to {$limits[$userType]} " . ($userType === in_array($userType, ['Platinum', 'Gold'])  ? 'events' : 'event') . "!")
                                ->persistent()
                                ->send();

                            $action->halt();
                        }

                        if (!empty($data['event_video_link']) && is_array($data['event_video_link'])) {
                            try {
                                $links = [];

                                // foreach ($data['event_video_link'] as $item) {
                                //     $link = trim($item['link'] ?? '');

                                //     // ✅ Skip empty rows
                                //     if ($link === '') {
                                //         continue;
                                //     }

                                //     $isYoutube = preg_match('#^(https?://)?(www\.)?(youtube\.com|youtu\.be)/#', $link);
                                //     $isVimeo   = preg_match('#^(https?://)?(www\.)?(vimeo\.com|player\.vimeo\.com/video)/#', $link);

                                //     if (!$isYoutube && !$isVimeo) {
                                //         throw new \InvalidArgumentException("Invalid video link: {$link}");
                                //     }

                                //     $links[] = $link;
                                // }

                                // ✅ Only set JSON if there are actual valid links
                                if (!empty($links)) {
                                    $data['event_video_link'] = json_encode($links);
                                } else {
                                    $data['event_video_link'] = null;
                                }
                            } catch (\InvalidArgumentException $e) {
                                Notification::make()
                                    ->danger()
                                    ->title('Validation Error')
                                    ->body($e->getMessage())
                                    ->send();

                                $action->halt();
                                return;
                            }
                        }

                        // try {
                        //     if (!empty($data['event_video_link']) && is_array($data['event_video_link'])) {
                        //         $links = array_map(fn($item) => $item['link'], $data['event_video_link']);

                        //         // --- VALIDATE LINKS FIRST ---
                        //         $invalidLinks = [];
                        //         foreach ($links as $link) {
                        //             $link = trim($link);
                        //             $isYoutube = preg_match('#^(https?://)?(www\.)?(youtube\.com|youtu\.be)/#', $link);
                        //             $isVimeo = preg_match('#^(https?://)?(www\.)?(vimeo\.com|player\.vimeo\.com/video)/#', $link);

                        //             if (!$isYoutube && !$isVimeo) {
                        //                 $invalidLinks[] = $link;
                        //             }
                        //         }

                        //         if (!empty($invalidLinks)) {
                        //             throw new \InvalidArgumentException(
                        //                 'Invalid video link(s): ' . implode(', ', $invalidLinks)
                        //             );
                        //         }

                        //         // --- LINKS ARE VALID, THEN FORMAT ---
                        //         $data['event_video_link'] = json_encode($links);
                        //     }

                        //     // Now safe to call handleLinkFormatting
                        //     Event::handleLinkFormatting($data);

                        //     // Save the new event
                        //     $livewire->ownerRecord->events()->create($data);

                        //     Notification::make()
                        //         ->success()
                        //         ->title('Success')
                        //         ->body('Event created successfully.')
                        //         ->send();

                        //     $action->cancel();
                        // } catch (\InvalidArgumentException $e) {
                        //     Notification::make()
                        //         ->danger()
                        //         ->title('Validation Error')
                        //         ->body($e->getMessage())
                        //         ->send();

                        //     $action->halt();
                        // }
                    })
                    ->disabled(function (RelationManager $livewire) {
                        $limits = [
                            'Platinum' => 20,
                            'Gold' => 3,
                            'Basic' => 1,
                        ];

                        // $userType = $livewire->ownerRecord->user_type;
                        $userType   = $livewire->ownerRecord->user?->user_type; // 🔑 Get from users table
                        $eventCount = $livewire->ownerRecord->events()->count();

                        return isset($limits[$userType]) && $eventCount >= $limits[$userType];
                    })
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\Action::make('view_card')
                        ->label('View')
                        ->icon('heroicon-o-eye')
                        ->url(fn($record) => app(CardDefaultsSetting::class)->site_url . $record->subcard->username . '/events/' . $record->slug)
                        ->openUrlInNewTab()
                        ->color('info'),

                    Tables\Actions\EditAction::make()
                        // ->before(function (EditAction $action, $livewire) {
                        //     // Access form data from the action
                        //     $data = $action->getFormData();
                        //     $record = $action->getRecord();

                        //     try {
                        //         // If there are video links, extract them and encode as JSON
                        //         if (!empty($data['event_video_link']) && is_array($data['event_video_link'])) {
                        //             $links = array_map(fn($item) => $item['link'], $data['event_video_link']);
                        //             $data['event_video_link'] = json_encode($links);
                        //         }

                        //         Event::handleLinkFormatting($data);

                        //         // Update the record
                        //         $record->update($data);

                        //         Notification::make()
                        //             ->success()
                        //             ->title('Success')
                        //             ->body('Event updated successfully.')
                        //             ->send();

                        //         $action->cancel();
                        //     } catch (\InvalidArgumentException $e) {
                        //         Notification::make()
                        //             ->danger()
                        //             ->title('Validation Error')
                        //             ->body($e->getMessage()) // Shows the invalid link error
                        //             ->send();

                        //         $action->halt();
                        //     }
                        // }),

                        ->before(function (EditAction $action) {
                            $data = $action->getFormData();
                            $record = $action->getRecord();

                            try {
                                // Normalize event_video_link
                                $videoLinks = $data['event_video_link'] ?? [];

                                // If stored JSON string → decode it
                                if (is_string($videoLinks)) {
                                    $videoLinks = json_decode($videoLinks, true) ?? [];
                                }

                                // If simple list of strings → convert to repeater format
                                if (!empty($videoLinks) && is_string($videoLinks[0] ?? null)) {
                                    $videoLinks = array_map(fn($link) => ['link' => $link], $videoLinks);
                                }

                                // Now extract only the 'link' values
                                $data['event_video_link'] = json_encode(
                                    array_filter(array_map(
                                        fn($item) => $item['link'] ?? null,
                                        $videoLinks
                                    ))
                                );

                                Event::handleLinkFormatting($data);

                                $record->update($data);

                                Notification::make()
                                    ->success()
                                    ->title('Success')
                                    ->body('Event updated successfully.')
                                    ->send();

                                $action->cancel();
                            } catch (\InvalidArgumentException $e) {
                                Notification::make()
                                    ->danger()
                                    ->title('Validation Error')
                                    ->body($e->getMessage())
                                    ->send();

                                $action->halt();
                            }
                        }),
                    Tables\Actions\DeleteAction::make(),
                ])
                    // ->dropdown(false)
                    ->label('Actions') // optional — can be hidden if you prefer
                    ->icon('heroicon-o-cog-6-tooth') // optional
                    ->color('gray'), // ✅ ensures buttons are left-aligned

            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
