<?php

namespace App\Filament\Resources\SubCardsResource\RelationManagers;

use App\Models\ProductCategory;
use App\Models\ProductCollection;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Filters\Filter;
use Illuminate\Database\Eloquent\Model;
use Filament\Support\Enums\IconPosition;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use Filament\Forms\Components\FileUpload;
use Filament\Tables\Actions\Action;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\DB;
use Filament\Forms\Components\Select;

class ProductsRelationManager extends RelationManager
{
    protected static string $relationship = 'products';

    // public function form(Form $form): Form
    // {
    //     return $form
    //         ->schema([
    //             FileUpload::make('featured_image')
    //                 ->label('Featured Image')
    //                 ->placeholder('Upload Featured Image')
    //                 ->directory('products')
    //                 ->uploadingMessage('Uploading Featured Image. Please wait...')
    //                 ->image()
    //                 ->imageEditor()
    //                 ->columnSpanFull()
    //                 ->imageResizeMode('cover')
    //                 ->imageCropAspectRatio(null)
    //                 // ->imageCropAspectRatio('1:1')
    //                 // ->imageResizeTargetWidth('520')
    //                 // ->imageResizeTargetHeight('520')
    //                 ->getUploadedFileNameForStorageUsing(
    //                     fn(TemporaryUploadedFile $file): string =>
    //                     (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
    //                 )
    //                 ->imageEditorAspectRatios([
    //                     '1:1',
    //                 ]),

    //             Forms\Components\TextInput::make('product_name')
    //                 ->label('Product Name')
    //                 ->columnSpanFull()
    //                 ->required()
    //                 ->maxLength(255),

    //             // Forms\Components\Select::make('type')
    //             //     ->label('Product Type')
    //             //     ->columnSpanFull()
    //             //     ->options([
    //             //         'physical' => 'Physical',
    //             //         'digital' => 'Digital',
    //             //     ])
    //             //     ->default('physical')
    //             //     ->reactive()
    //             //     ->required(),

    //             Forms\Components\Select::make('type')
    //                 ->label('Product Type')
    //                 ->columnSpanFull()
    //                 ->options([
    //                     'physical' => 'Physical',
    //                     'digital' => 'Digital',
    //                     'subscription' => 'Subscription',
    //                 ])
    //                 ->default('physical')
    //                 ->afterStateUpdated(function (callable $set, $state) {
    //                     $set(
    //                         'shop_online_button',
    //                         $state === 'subscription'
    //                             ? 'Subscribe'
    //                             : 'Shop Online'
    //                     );
    //                 })
    //                 ->reactive()
    //                 ->required(),

    //             Forms\Components\TagsInput::make('sizes')
    //                 ->label('Sizes')
    //                 ->placeholder('Add sizes (e.g. S, M, L, XL)')
    //                 ->reorderable()
    //                 ->suggestions(['XS', 'S', 'M', 'L', 'XL', 'XXL'])
    //                 ->columnSpanFull()
    //                 // ->visible(fn(callable $get) => $get('type') !== 'digital'),
    //                 ->visible(fn(callable $get) => !in_array($get('type'), ['digital', 'subscription'])),

    //             Forms\Components\Repeater::make('colors')
    //                 ->label('Colors')
    //                 ->schema([
    //                     Forms\Components\TextInput::make('name')
    //                         ->label('Color Name')
    //                         ->placeholder('e.g. Sky Blue')
    //                         ->required(),
    //                     Forms\Components\ColorPicker::make('value')
    //                         ->label('Color Code')
    //                         ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
    //                         ->required()
    //                         ->default('#ffffff'),
    //                 ])
    //                 ->collapsible()
    //                 ->reorderable()
    //                 ->addActionLabel('Add Color')
    //                 ->columns(2)
    //                 ->columnSpanFull()
    //                 // ->visible(fn(callable $get) => $get('type') !== 'digital'),
    //                 ->visible(fn(callable $get) => !in_array($get('type'), ['digital', 'subscription'])),

    //             Forms\Components\TagsInput::make('styles')
    //                 ->label('Styles')
    //                 ->placeholder('Add styles (e.g. Casual, Formal, Sport)')
    //                 ->reorderable()
    //                 ->suggestions(['Casual', 'Formal', 'Sport', 'Vintage', 'Modern'])
    //                 ->columnSpanFull()
    //                 // ->visible(fn(callable $get) => $get('type') !== 'digital'),
    //                 ->visible(fn(callable $get) => !in_array($get('type'), ['digital', 'subscription'])),

    //             Forms\Components\Select::make('category_id')
    //                 ->visible(fn(callable $get) => $get('type') != 'subscription')
    //                 ->label('Category')
    //                 ->relationship(
    //                     name: 'category',
    //                     titleAttribute: 'title',
    //                     modifyQueryUsing: fn(Builder $query, $livewire) =>
    //                     $query->where('page_id', $livewire->ownerRecord->id)
    //                 )
    //                 ->searchable()
    //                 ->preload()
    //                 ->nullable()
    //                 ->createOptionForm([
    //                     Forms\Components\TextInput::make('title')
    //                         ->label('New Category Name')
    //                         ->required()
    //                         ->maxLength(255),

    //                     Forms\Components\Textarea::make('description')
    //                         ->maxLength(500)
    //                         ->columnSpanFull()
    //                         ->helperText('Write a few sentences about the product category. Max character 500.'),
    //                 ])
    //                 ->createOptionUsing(function (array $data, $livewire) {
    //                     // Inject the current user_id from the parent record (ownerRecord)
    //                     $data['page_id'] = $livewire->ownerRecord->id;

    //                     return ProductCategory::create($data)->getKey();
    //                 }),

    //             Forms\Components\Select::make('collection_id')
    //                 ->visible(fn(callable $get) => $get('type') != 'subscription')
    //                 ->label('Collection')
    //                 ->relationship(
    //                     name: 'collection',
    //                     titleAttribute: 'title',
    //                     modifyQueryUsing: fn(Builder $query, $livewire) =>
    //                     $query->where('page_id', $livewire->ownerRecord->id)
    //                 )
    //                 ->searchable()
    //                 ->preload()
    //                 ->nullable()
    //                 ->createOptionForm([

    //                     FileUpload::make('image')
    //                         ->label('Upload Collection Banner')
    //                         ->placeholder('Upload Collection Banner')
    //                         ->directory('product_collection')
    //                         ->uploadingMessage('Uploading Collection Banner. Please wait...')
    //                         ->image()
    //                         ->required()
    //                         ->imageEditor()
    //                         ->columnSpanFull()
    //                         ->imageResizeMode('cover')
    //                         ->imageCropAspectRatio('1080:600')
    //                         ->imageResizeTargetWidth('1080')
    //                         ->imageResizeTargetHeight('600')
    //                         ->getUploadedFileNameForStorageUsing(
    //                             fn(TemporaryUploadedFile $file): string =>
    //                             (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
    //                         )
    //                         ->imageEditorAspectRatios([
    //                             '1080:600',
    //                         ]),

    //                     Forms\Components\TextInput::make('title')
    //                         ->label('New Collection Name')
    //                         ->required()
    //                         ->maxLength(255),

    //                     Forms\Components\Textarea::make('description')
    //                         ->maxLength(500)
    //                         ->columnSpanFull()
    //                         ->helperText('Write a few sentences about the product collection. Max character 500.'),
    //                 ])
    //                 ->createOptionUsing(function (array $data, $livewire) {
    //                     // Inject the current user_id from the parent record (ownerRecord)
    //                     $data['page_id'] = $livewire->ownerRecord->id;

    //                     return ProductCollection::create($data)->getKey();
    //                 }),

    //             Forms\Components\TextInput::make('price')
    //                 ->required()
    //                 ->numeric()
    //                 ->prefix('$'),

    //             Forms\Components\TextInput::make('stock')
    //                 ->numeric()
    //                 ->default(1)
    //                 ->visible(
    //                     fn(callable $get) =>
    //                     !in_array($get('type'), ['subscription', 'digital'])
    //                 )
    //                 ->required(
    //                     fn(callable $get) =>
    //                     !in_array($get('type'), ['subscription', 'digital'])
    //                 ),

    //             Forms\Components\TextInput::make('shop_online_button')
    //                 ->label('Button')
    //                 ->reactive()
    //                 ->default('Shop Online')
    //                 ->afterStateHydrated(function ($component, $state, callable $get) {
    //                     if (blank($state)) {
    //                         $component->state(
    //                             $get('type') === 'subscription'
    //                                 ? 'Subscribe'
    //                                 : 'Shop Online'
    //                         );
    //                     }
    //                 })
    //                 ->dehydrated()
    //                 ->required()
    //                 ->maxLength(255),

    //             Forms\Components\TextInput::make('shop_online_link')
    //                 ->label('Shop Online Link')
    //                 ->columnSpanFull()
    //                 ->url()
    //                 ->required(),

    //             Forms\Components\RichEditor::make('description')
    //                 ->maxLength(1000)
    //                 ->columnSpanFull()
    //                 ->disableGrammarly()
    //                 ->toolbarButtons([
    //                     // 'blockquote',
    //                     'bold',
    //                     'bulletList',
    //                     'h1',
    //                     'h2',
    //                     'h3',
    //                     'italic',
    //                     'orderedList',
    //                     'redo',
    //                     'underline',
    //                     'undo',
    //                 ])
    //                 ->helperText('Write a few sentences about the product. Max character 1000.'),

    //             Forms\Components\Toggle::make('is_active')
    //                 ->label('Active')
    //                 ->default(true)
    //                 ->columnSpanFull()
    //                 ->required(),
    //         ]);
    // }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                FileUpload::make('featured_image')
                    ->label('Featured Image')
                    ->placeholder('Upload Featured Image')
                    ->directory('products')
                    ->uploadingMessage('Uploading Featured Image. Please wait...')
                    ->image()
                    ->imageEditor()
                    ->columnSpanFull()
                    ->imageResizeMode('cover')
                    ->imageCropAspectRatio(null)
                    // ->imageCropAspectRatio('1:1')
                    // ->imageResizeTargetWidth('520')
                    // ->imageResizeTargetHeight('520')
                    ->getUploadedFileNameForStorageUsing(
                        fn(TemporaryUploadedFile $file): string =>
                        (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                    )
                    ->imageEditorAspectRatios([
                        '1:1',
                    ]),

                Forms\Components\TextInput::make('product_name')
                    ->label('Product Name')
                    ->columnSpanFull()
                    ->required()
                    ->maxLength(255),


                Forms\Components\Select::make('type')
                    ->label('Product Type')
                    ->columnSpanFull()
                    ->options([
                        'physical' => 'Physical',
                        'digital' => 'Digital',
                        'subscription' => 'Subscription',
                    ])
                    ->default('physical')
                    ->afterStateUpdated(function (callable $set, $state) {
                        $set(
                            'shop_online_button',
                            $state === 'subscription'
                                ? 'Subscribe'
                                : 'Shop Online'
                        );
                    })
                    ->reactive()
                    ->required(),

                Forms\Components\TextInput::make('stock')
                    ->columnSpanFull()
                    ->numeric()
                    ->default(1)
                    ->visible(
                        fn(callable $get) =>
                        !in_array($get('type'), ['subscription', 'digital'])
                    )
                    ->required(
                        fn(callable $get) =>
                        !in_array($get('type'), ['subscription', 'digital'])
                    ),

                Forms\Components\Toggle::make('has_packages')
                    ->visible(fn(callable $get) => $get('type') == 'subscription')
                    ->label('Has Packages')
                    ->reactive()
                    ->default(false)
                    ->columnSpanFull(),

                Forms\Components\TextInput::make('price')
                    ->visible(fn(callable $get) => $get('type') !== 'subscription' || $get('has_packages') == false)
                    ->required()
                    ->columnSpanFull()
                    ->numeric()
                    ->prefix('$'),

                Forms\Components\Toggle::make('is_on_deal')
                    ->visible(fn(callable $get) => $get('type') !== 'subscription')
                    ->label('Is on Deal')
                    ->default(false)
                    ->reactive()
                    ->columnSpanFull(),

                Forms\Components\DatePicker::make('deal_starts_at')
                    ->label('Deal Starts At')
                    ->visible(fn(callable $get) => $get('is_on_deal') === true && $get('type') !== 'subscription')
                    ->required(fn(callable $get) => $get('is_on_deal') === true && $get('type') !== 'subscription')
                    ->reactive()
                    ->rules([
                        fn(callable $get) =>
                        $get('is_on_deal') === true
                            ? 'required'
                            : 'nullable',
                    ])
                    ->minDate(now()->startOfDay())
                    ->native(false),

                Forms\Components\DatePicker::make('deal_ends_at')
                    ->label('Deal Ends At')
                    ->visible(fn(callable $get) => $get('is_on_deal') === true && $get('type') !== 'subscription')
                    ->required(fn(callable $get) => $get('is_on_deal') === true && $get('type') !== 'subscription')
                    ->native(false)
                    ->reactive()

                    // UI restriction
                    ->minDate(function (callable $get) {
                        $start = $get('deal_starts_at');

                        if ($start) {
                            return \Carbon\Carbon::parse($start)->addDay()->startOfDay();
                        }

                        return now()->addDay()->startOfDay();
                    })

                    // Backend validation
                    ->rules([
                        function (callable $get) {
                            if ($get('is_on_deal') === true) {
                                return function (string $attribute, $value, $fail) use ($get) {
                                    $start = $get('deal_starts_at');

                                    if ($start && \Carbon\Carbon::parse($value)->lt(
                                        \Carbon\Carbon::parse($start)->addDay()->startOfDay()
                                    )) {
                                        $fail('The deal end date must be at least 1 day after the start date.');
                                    }
                                };
                            }

                            return 'nullable';
                        },
                    ]),

                Forms\Components\Select::make('deal_type')
                    ->label('Deal Type')
                    ->options([
                        'price' => 'Deal Price',
                        'discount' => 'Discount Percentage',
                    ])
                    ->default('price')
                    ->afterStateHydrated(function ($component, $state, callable $get) {
                        // If user turns on a deal and deal_type is empty, force it to "price"
                        if ($get('is_on_deal') === true && empty($state)) {
                            $component->state('price');
                        }
                    })
                    ->reactive()
                    ->visible(fn(callable $get) => $get('is_on_deal') === true && $get('type') !== 'subscription')
                    ->dehydrated(false)
                    ->helperText(fn($get) => $get('deal_type') === 'price'
                        ? 'Enter the final deal price; discount will be calculated automatically.'
                        : 'Enter the discount percentage; deal price will be calculated automatically.'),

                Forms\Components\TextInput::make('deal_price')
                    ->label('Deal Price')
                    ->numeric()
                    ->reactive()
                    ->visible(fn($get) => $get('is_on_deal') === true && $get('deal_type') === 'price' && $get('type') !== 'subscription')
                    ->afterStateUpdated(function ($state, callable $set, callable $get) {
                        // When deal_price changes, recalc discount_percentage
                        if ($state && $get('price')) {
                            $set('discount_percentage', round((1 - $state / $get('price')) * 100, 2));
                        }
                    }),

                Forms\Components\TextInput::make('discount_percentage')
                    ->label('Discount (%)')
                    ->numeric()
                    ->minValue(0)
                    ->maxValue(100)
                    ->reactive()
                    ->visible(fn($get) => $get('is_on_deal') === true && $get('deal_type') === 'discount' && $get('type') !== 'subscription')
                    ->afterStateUpdated(function ($state, callable $set, callable $get) {
                        // When discount changes, recalc deal_price
                        if ($state !== null && $get('price')) {
                            $set('deal_price', round($get('price') * (1 - $state / 100), 2));
                        }
                    }),

                Forms\Components\Toggle::make('has_coins')
                    ->visible(fn(callable $get) => $get('type') == 'subscription')
                    ->label('Has Coin')
                    ->default(true)
                    ->reactive()
                    ->columnSpanFull(),

                Forms\Components\TextInput::make('coin')
                    ->reactive()
                    ->label('Coin')
                    ->columnSpanFull()
                    ->visible(fn(callable $get) => $get('type') === 'subscription' && $get('has_packages') === false)
                    ->required(fn(callable $get) => $get('type') === 'subscription' && $get('has_packages') === false)
                    ->numeric(),

                Forms\Components\TextInput::make('shop_online_button')
                    ->visible(fn(callable $get) => $get('type') !== 'subscription')
                    ->label('Button')
                    ->reactive()
                    ->columnSpanFull()
                    ->default('Shop Online')
                    ->afterStateHydrated(function ($component, $state, callable $get) {
                        if (blank($state)) {
                            $component->state(
                                $get('type') === 'subscription'
                                    ? 'Subscribe'
                                    : 'Shop Online'
                            );
                        }
                    })
                    ->dehydrated()
                    ->required()
                    ->maxLength(255),

                Forms\Components\TextInput::make('shop_online_link')
                    ->visible(fn(callable $get) => $get('type') !== 'subscription' || $get('has_packages') === false)
                    ->label('Shop Online Link')
                    ->columnSpanFull()
                    ->label(fn(callable $get) => $get('type') == 'subscription' ? 'Subscription Link' : 'Shop Online Link')
                    ->url()
                    ->required(),

                Forms\Components\TagsInput::make('sizes')
                    ->label('Sizes')
                    ->placeholder('Add sizes (e.g. S, M, L, XL)')
                    ->reorderable()
                    ->suggestions(['XS', 'S', 'M', 'L', 'XL', 'XXL'])
                    ->columnSpanFull()
                    // ->visible(fn(callable $get) => $get('type') !== 'digital'),
                    ->visible(fn(callable $get) => !in_array($get('type'), ['digital', 'subscription']))
                    ->dehydrateStateUsing(function ($state) {
                        return array_filter($state); // removes empty strings
                    }),

                Forms\Components\Repeater::make('colors')
                    ->label('Colors')
                    ->schema([
                        Forms\Components\TextInput::make('name')
                            ->label('Color Name')
                            ->placeholder('e.g. Sky Blue')
                            ->required(),
                        Forms\Components\ColorPicker::make('value')
                            ->label('Color Code')
                            ->regex('/^#([a-f0-9]{6}|[a-f0-9]{3})$/i')
                            ->required()
                            ->default('#ffffff'),
                    ])
                    ->collapsible()
                    ->reorderable()
                    ->addActionLabel('Add Color')
                    ->columns(2)
                    ->columnSpanFull()
                    ->visible(fn(callable $get) => !in_array($get('type'), ['digital', 'subscription'])),

                Forms\Components\TagsInput::make('styles')
                    ->label('Styles')
                    ->placeholder('Add styles (e.g. Casual, Formal, Sport)')
                    ->reorderable()
                    ->suggestions(['Casual', 'Formal', 'Sport', 'Vintage', 'Modern'])
                    ->columnSpanFull()
                    ->visible(fn(callable $get) => !in_array($get('type'), ['digital', 'subscription']))
                    ->dehydrateStateUsing(function ($state) {
                        return array_filter($state); // removes empty strings
                    }),

                // Select::make('category_id')
                Forms\Components\MultiSelect::make('category_id')
                    ->visible(fn(callable $get) => $get('type') != 'subscription')
                    ->label('Category')
                    // ->relationship(
                    //     name: 'category',
                    //     titleAttribute: 'title',
                    //     modifyQueryUsing: fn(Builder $query, $livewire) =>
                    //     $query->where('page_id', $livewire->ownerRecord->id)
                    // )
                    ->options(function (callable $get, $livewire) {
                        $options = ProductCategory::where('page_id', $livewire->ownerRecord->id)
                            ->pluck('title', 'id')
                            ->filter(fn($title) => $title !== null)
                            ->toArray();

                        // $current = $get('category_id') ?? [];
                        // foreach ($current as $id) {
                        //     if (!array_key_exists($id, $options)) {
                        //         $options[$id] = 'Unknown'; // fallback label for deleted/null categories
                        //     }
                        // }

                        return $options;
                    })
                    ->searchable()
                    ->preload()
                    ->nullable()
                    ->afterStateHydrated(function ($component, $state) {
                        if (is_array($state)) {
                            $component->state(array_values(array_filter($state, fn($v) => !is_null($v))));
                        }
                    })
                    ->dehydrateStateUsing(fn($state) => array_map('intval', array_filter($state)))
                    ->createOptionForm([
                        Forms\Components\TextInput::make('title')
                            ->label('New Category Name')
                            ->required()
                            ->maxLength(255),

                        Forms\Components\Textarea::make('description')
                            ->maxLength(500)
                            ->columnSpanFull()
                            ->helperText('Write a few sentences about the product category. Max character 500.'),
                    ])
                    ->createOptionUsing(function (array $data, $livewire) {
                        // Inject the current user_id from the parent record (ownerRecord)
                        // $data['user_id'] = $livewire->ownerRecord->id;
                        $data['page_id'] = $livewire->ownerRecord->id;

                        return ProductCategory::create($data)->getKey();
                    }),

                Forms\Components\MultiSelect::make('collection_id')
                    ->visible(fn(callable $get) => $get('type') != 'subscription')
                    ->label('Collection')
                    ->options(function (callable $get, $livewire) {
                        $options = ProductCollection::where('page_id', $livewire->ownerRecord->id)
                            ->pluck('title', 'id')
                            ->filter(fn($title) => $title !== null)
                            ->toArray();

                        // $current = $get('collection_id') ?? [];
                        // foreach ($current as $id) {
                        //     if (!array_key_exists($id, $options)) {
                        //         $options[$id] = 'Unknown'; // fallback label
                        //     }
                        // }

                        return $options;
                    })
                    ->searchable()
                    ->preload()
                    ->nullable()
                    ->afterStateHydrated(function ($component, $state) {
                        if (is_array($state)) {
                            $component->state(array_values(array_filter($state, fn($v) => !is_null($v))));
                        }
                    })
                    ->dehydrateStateUsing(fn($state) => array_map('intval', array_filter($state)))
                    ->createOptionForm([
                        FileUpload::make('image')
                            ->label('Upload Collection Banner')
                            ->placeholder('Upload Collection Banner')
                            ->directory('product_collection')
                            ->uploadingMessage('Uploading Collection Banner. Please wait...')
                            ->image()
                            ->required()
                            ->imageEditor()
                            ->columnSpanFull()
                            ->imageResizeMode('cover')
                            ->imageCropAspectRatio('1080:600')
                            ->imageResizeTargetWidth('1080')
                            ->imageResizeTargetHeight('600')
                            ->getUploadedFileNameForStorageUsing(
                                fn(TemporaryUploadedFile $file): string =>
                                (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                            ),

                        Forms\Components\TextInput::make('title')
                            ->label('New Collection Name')
                            ->required()
                            ->maxLength(255),

                        Forms\Components\Textarea::make('description')
                            ->maxLength(500)
                            ->columnSpanFull()
                            ->helperText('Write a few sentences about the product collection. Max character 500.'),

                    ])
                    ->createOptionUsing(function (array $data, $livewire) {
                        // Inject the current user_id from the parent record (ownerRecord)
                        // $data['user_id'] = $livewire->ownerRecord->id;
                        $data['page_id'] = $livewire->ownerRecord->id;

                        return ProductCollection::create($data)->getKey();
                    }),

                Forms\Components\RichEditor::make('description')
                    ->maxLength(1000)
                    ->columnSpanFull()
                    ->disableGrammarly()
                    ->toolbarButtons([
                        // 'blockquote',
                        'bold',
                        'bulletList',
                        // 'h1',
                        // 'h2',
                        // 'h3',
                        'italic',
                        'orderedList',
                        // 'redo',
                        'underline',
                        // 'undo',
                    ])
                    ->helperText('Write a few sentences about the product. Max character 1000.'),

                Forms\Components\Repeater::make('subscription_options')
                    ->reactive()
                    ->visible(fn(callable $get) => $get('type') === 'subscription' && $get('has_packages') === true)
                    ->label('Subscription Packages')
                    ->columnSpanFull()
                    ->schema([

                        Forms\Components\TextInput::make('title')
                            ->label('Package Title')
                            ->required(),

                        Forms\Components\TextInput::make('price')
                            ->required()
                            ->numeric()
                            ->prefix('$'),

                        Forms\Components\TextInput::make('coin')
                            ->visible(fn(callable $get) => $get('../../has_coins') === true)
                            ->reactive()
                            ->label('Coin')
                            ->required()
                            ->numeric(),

                        Forms\Components\TextInput::make('subscription_link')
                            ->label('Subscription Link')
                            ->url()
                            ->required(),

                        Forms\Components\RichEditor::make('tier_description')
                            ->label('Package Description')
                            ->maxLength(5000)
                            ->required()
                            ->columnSpanFull()
                            ->disableGrammarly()
                            ->toolbarButtons([
                                // 'blockquote',
                                'bold',
                                'bulletList',
                                // 'h1',
                                // 'h2',
                                // 'h3',
                                'italic',
                                'orderedList',
                                // 'redo',
                                'underline',
                                // 'undo',
                            ])
                            ->helperText('Write a few sentences about the product. Max character 5000.'),
                        // ->afterStateUpdated(function ($state, $set) {
                        //     $plainText = strip_tags($state ?? '');
                        //     $set('description_count', Str::length($plainText));
                        // })

                        // ->rule(fn($get) => function ($attribute, $value, $fail) {
                        //     $plainText = strip_tags($value ?? '');
                        //     if (Str::length($plainText) > 5000) {
                        //         $fail('The description must not exceed 5000 characters.');
                        //     }
                        // }),

                    ])
                    ->reorderable(),

                // Forms\Components\Toggle::make('is_active')
                //     ->label('Active')
                //     ->default(true)
                //     ->columnSpanFull()
                //     ->required(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->description('All items listed/added here will be displayed under the "Shop" button on the card.')
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->where('page_id', $livewire->ownerRecord->id)->orderBy('order')
            )
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('product_name')
                    ->label('Product Name')
                    ->searchable(),

                Tables\Columns\TextColumn::make('type')
                    ->label('Type'),

                Tables\Columns\TextColumn::make('category.title')
                    ->label('Category')
                    ->searchable(),

                Tables\Columns\TextColumn::make('collection.title')
                    ->label('Collection')
                    ->searchable(),

                Tables\Columns\TextColumn::make('price')
                    ->money('USD'),

                Tables\Columns\TextColumn::make('stock'),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->sortable()
                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),

            ])
            ->filters([
                Filter::make('is_active')
                    ->label('Active')
                    ->query(fn(Builder $query) => $query->where('is_active', true)),
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['user_id'] = $livewire->ownerRecord->user_id;
                        return $data;
                    }),
            ])
            ->actions([
                Tables\Actions\ActionGroup::make([
                    Tables\Actions\Action::make('view_card')
                        ->label('View')
                        ->icon('heroicon-o-eye')
                        ->url(function ($record) {
                            if (!$record) return '#';

                            $baseUrl = app(\App\Settings\CardDefaultsSetting::class)->site_url;

                            // if ($record->page_id && $record->subcard) {
                            //     return $baseUrl . $record->subcard->username . '/products/' . $record->id;
                            // }

                            // if ($record->user) {
                            //     return $baseUrl . $record->user->username . '/products/' . $record->id;
                            // }

                            if ($record->type != "subscription") {
                                if ($record->page_id && $record->subcard) {
                                    return $baseUrl . $record->subcard->username . '/products/' . $record->id;
                                }

                                if ($record->user) {
                                    return $baseUrl . $record->user->username . '/products/' . $record->id;
                                }
                            } else {
                                if ($record->page_id && $record->subcard) {
                                    return $baseUrl . $record->subcard->username . '/subscription/' . $record->id;
                                }

                                if ($record->user) {
                                    return $baseUrl . $record->user->username . '/subscription/' . $record->id;
                                }
                            }

                            return '#';
                        })
                        ->openUrlInNewTab()
                        ->color('info'),

                    Tables\Actions\EditAction::make(),

                    Tables\Actions\Action::make('addMedia')
                        ->icon('heroicon-o-photo')
                        ->label('Add Media')
                        ->url(fn($record) => route('filament.admin.resources.products.edit', $record)),

                    Tables\Actions\Action::make('customAction')
                        ->icon('heroicon-o-document-duplicate')
                        ->label('Duplicate')
                        ->color('warning')
                        ->requiresConfirmation()
                        ->action(function (Model $record) {
                            DB::transaction(function () use ($record) {

                                // Helper function to safely duplicate a file
                                $duplicateFile = function (?string $filePath) {
                                    if (!$filePath || !Storage::disk('public')->exists($filePath)) {
                                        return null;
                                    }

                                    $directory = dirname($filePath);
                                    $extension = pathinfo($filePath, PATHINFO_EXTENSION);
                                    $originalName = pathinfo($filePath, PATHINFO_FILENAME);

                                    $maxBaseLength = 50;
                                    $shortName = Str::limit($originalName, $maxBaseLength, '');
                                    $newFileName = $shortName . '_' . Str::random(6) . '.' . $extension;
                                    $newPath = $directory . '/' . $newFileName;

                                    Storage::disk('public')->makeDirectory($directory);
                                    Storage::disk('public')->copy($filePath, $newPath);

                                    return $newPath;
                                };

                                // ✅ Duplicate the main product
                                $newProduct = $record->replicate();
                                $newProduct->product_name = $record->product_name . ' (Copy)';
                                $newProduct->featured_image = $duplicateFile($record->featured_image);
                                $newProduct->created_at = now();
                                $newProduct->updated_at = now();
                                $newProduct->save();

                                // ✅ Duplicate related media (if any)
                                if (method_exists($record, 'media') && $record->media()->exists()) {
                                    foreach ($record->media()->orderBy('order')->get() as $media) {
                                        $newMedia = $media->replicate();

                                        // Duplicate image files only
                                        if ($media->type === 'image') {
                                            $newPath = $duplicateFile($media->path);
                                            if (!$newPath) {
                                                continue;
                                            }
                                            $newMedia->path = $newPath;
                                        } else {
                                            $newMedia->path = $media->path; // reuse path for non-image
                                        }

                                        $newMedia->product_id = $newProduct->id;
                                        $newMedia->order = $media->order;
                                        $newMedia->save();
                                    }
                                }

                                Notification::make()
                                    ->title('Success')
                                    ->body('Product duplicated successfully.')
                                    ->success()
                                    ->send();
                            });
                        }),

                    Tables\Actions\DeleteAction::make(),
                ])
                    // ->dropdown(false)
                    ->label('Actions') // optional — can be hidden if you prefer
                    ->icon('heroicon-o-cog-6-tooth') // optional
                    ->color('gray'),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
