<?php

namespace App\Filament\Resources\SubCardsResource\RelationManagers;

use App\Models\Replay;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Database\Eloquent\Model;
use App\Settings\CardDefaultsSetting;
use Filament\Tables\Actions\Action;
use Filament\Support\Enums\IconPosition;
use Filament\Notifications\Notification;
use Filament\Tables\Actions\CreateAction;
use Filament\Tables\Actions\EditAction;

class ReplaysRelationManager extends RelationManager
{
    protected static string $relationship = 'replays';

    public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    {
        return app(CardDefaultsSetting::class)->replays_functionality === true;
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('title')
                    ->required()
                    ->maxLength(255)
                    ->columnSpanFull(),

                Forms\Components\TextInput::make('link')
                    ->required()
                    ->columnSpanFull(),

                Forms\Components\Toggle::make('is_active')
                    ->columnSpanFull()
                    ->label('Active')
                    ->default(true),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->recordTitleAttribute('title')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(function (Builder $query, RelationManager $livewire) {
                $query->orderBy('order');
            })
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success')
                    ->offColor('danger'),

                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    ->url(fn($record) => $record->link)
                    ->openUrlInNewTab(),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->before(function (CreateAction $action, $livewire) {
                        // Access form data from the action
                        $data = $action->getFormData();
                        try {
                            // Assuming $data is the input data for button creation
                            Replay::handleLinkFormatting($data);

                            // Check if the user is on a free plan and has exceeded the button limit
                            if ($livewire->ownerRecord->replays()->count() >= 20) {
                                Notification::make()
                                    ->warning()
                                    ->title('Can only add up to 20 replay links!')
                                    ->persistent()
                                    ->send();

                                // Cancel the action to prevent further processing
                                $action->cancel(); // This ensures that the action is canceled and no further action is executed
                            } else {
                                // Proceed to create the button if no issues
                                $livewire->ownerRecord->replays()->create($data);

                                Notification::make()
                                    ->title('Success')
                                    ->body('Replay link created successfully.')
                                    ->success()  // Success notification style
                                    ->send();

                                // Cancel the action after manual creation to prevent default action execution
                                $action->cancel();  // Prevent the default action (which would also try to create the button)
                            }
                        } catch (\InvalidArgumentException $e) {
                            Notification::make()
                                ->title('Validation Error')
                                ->body($e->getMessage())  // Show the exception message
                                ->danger()  // Set notification style to error
                                ->send();

                            // Cancel the action to prevent the model creation
                            $action->halt(); // Halt the action to prevent further processing
                        }
                    })
                    ->disabled(fn(RelationManager $livewire) => $livewire->ownerRecord->replays()->count() >= 20)
                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['user_id'] = $livewire->ownerRecord->user_id;
                        return $data;
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make()->before(function (EditAction $action, $livewire) {
                    $data = $action->getFormData();

                    try {
                        Replay::handleLinkFormatting($data);

                        // Fetch the existing button to update
                        $button = $action->getRecord();

                        if ($livewire->ownerRecord->replays()->count() >= 20) {
                            Notification::make()
                                ->warning()
                                ->title('Can only add up to 20 replay links!')
                                ->persistent()
                                ->send();

                            $action->cancel();
                        } else {
                            $button->update($data);

                            Notification::make()
                                ->title('Success')
                                ->body('Replay link created successfully.')
                                ->success()
                                ->send();

                            $action->cancel();
                        }
                    } catch (\InvalidArgumentException $e) {
                        Notification::make()
                            ->title('Validation Error')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();

                        $action->halt();
                    }
                })
                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['user_id'] = $livewire->ownerRecord->user_id;
                        return $data;
                    }),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
