<?php

namespace App\Filament\Resources\SubCardsResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Support\Enums\IconPosition;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\CreateAction;
use Filament\Tables\Actions\EditAction;
use App\Models\Social;
use Filament\Notifications\Notification;

class SocialsRelationManager extends RelationManager
{
    protected static string $relationship = 'socials';

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('type')
                    ->required()
                    ->label('Social Media')
                    ->reactive()
                    ->afterStateUpdated(function ($state, callable $get, callable $set) {
                        $urls = [
                            'facebook' => 'https://www.facebook.com/',
                            'x' => 'https://x.com/',
                            'linkedin' => 'https://www.linkedin.com/in/',
                            'instagram' => 'https://www.instagram.com/',
                            'tiktok' => 'https://www.tiktok.com/@',
                            'youtube' => 'https://www.youtube.com/@', // you might want to handle channels differently
                            'clubhouse' => 'https://www.clubhouse.com/@',
                            'threads' => 'https://www.threads.net/@',
                            'fanbase' => 'https://www.fanbase.app/@',
                            'snapchat' => 'https://www.snapchat.com/add/',
                            'pinterest' => 'https://www.pinterest.com/username',
                            'soundcloud' => 'https://soundcloud.com/username',
                            'apple_podcasts' => 'https://podcasts.apple.com/podcast/id',
                            'spotify' => 'https://open.spotify.com/user/username',
                            'discord' => 'https://discord.gg/invitecode',
                            'skool' => 'https://skool.com/join/',
                        ];

                        $set('link', $urls[$state] ?? '');
                    })
                    ->options([
                        'facebook' => 'Facebook',
                        'x' => 'X',
                        'linkedin' => 'Linkedin',
                        'instagram' => 'Instagram',
                        'tiktok' => 'Tiktok',
                        'youtube' => 'Youtube',
                        'clubhouse' => 'Clubhouse',
                        'threads' => 'Threads',
                        'fanbase' => 'Fanbase',
                        'snapchat' => 'Snapchat',
                        'pinterest' => 'Pinterest',
                        'soundcloud' => 'Sound Cloud',
                        'apple_podcasts' => 'Apple Podcasts',
                        'spotify' => 'Spotify',
                        'discord' => 'Discord',
                        'skool' => 'Skool',
                    ]),
                Forms\Components\TextInput::make('link')
                    ->required()
                    ->maxLength(255),
                Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->description('All links listed/added here will be displayed as a social icon on the card.')
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->where('page_id', $livewire->ownerRecord->id)->orderBy('order')
            )
            ->columns([
                // Tables\Columns\IconColumn::make('is_active')
                //     ->label('Active')
                //     ->boolean(),

                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                // Tables\Columns\TextColumn::make('type')
                //     ->searchable()
                //     ->formatStateUsing(fn($state) => ucfirst($state)),
                Tables\Columns\TextColumn::make('type')
                    ->searchable()
                    ->formatStateUsing(fn($state) => ucfirst($state))
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    ->url(fn($record) => $record->link)
                    ->openUrlInNewTab(),
            ])
            ->filters([
                Filter::make('is_active')
                    ->label('Active')
                    ->query(fn(Builder $query) => $query->where('is_active', true)),
                SelectFilter::make('type')
                    ->label('Type')
                    ->options([
                        'facebook' => 'Facebook',
                        'x' => 'X',
                        'linkedin' => 'Linkedin',
                        'instagram' => 'Instagram',
                        'tiktok' => 'Tiktok',
                        'youtube' => 'Youtube',
                        'clubhouse' => 'Clubhouse',
                        'threads' => 'Threads',
                        'fanbase' => 'Fanbase',
                        'snapchat' => 'Snapchat',
                        'pinterest' => 'Pinterest',
                        'soundcloud' => 'Sound Cloud',
                        'apple_podcasts' => 'Apple Podcasts',
                        'spotify' => 'Spotify',
                        'discord' => 'Discord',
                        'skool' => 'Skool',
                    ]),
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->before(function (CreateAction $action, $livewire) {
                        // Access form data from the action
                        $data = $action->getFormData();
                        // Validate and process the link
                        try {
                            // Assuming $data is the input data for button creation
                            Social::handleLinkFormatting($data);

                            // Check if the user is on a free plan and has exceeded the button limit
                            if ($livewire->ownerRecord->user_type == 'Free' && $livewire->ownerRecord->socials()->count() >= 3) {
                                Notification::make()
                                    ->warning()
                                    ->title('Free card holders can only add up to 3 social links!')
                                    ->persistent()
                                    ->send();

                                // Cancel the action to prevent further processing
                                $action->cancel(); // This ensures that the action is canceled and no further action is executed
                            } else {
                                // Proceed to create the button if no issues
                                $livewire->ownerRecord->socials()->create($data);

                                Notification::make()
                                    ->title('Success')
                                    ->body('Social link created successfully.')
                                    ->success()  // Success notification style
                                    ->send();

                                // Cancel the action after manual creation to prevent default action execution
                                $action->cancel();  // Prevent the default action (which would also try to create the button)
                            }
                        } catch (\InvalidArgumentException $e) {
                            Notification::make()
                                ->title('Validation Error')
                                ->body($e->getMessage())  // Show the exception message
                                ->danger()  // Set notification style to error
                                ->send();

                            // Cancel the action to prevent the model creation
                            $action->halt(); // Halt the action to prevent further processing
                        }
                    })
                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['user_id'] = $livewire->ownerRecord->user_id;
                        return $data;
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make()->before(function (EditAction $action, $livewire) {
                    $data = $action->getFormData();
                    try {
                        Social::handleLinkFormatting($data);

                        // Fetch the existing button to update
                        $button = $action->getRecord();

                        if ($livewire->ownerRecord->user_type == 'Free' && $livewire->ownerRecord->socials()->count() >= 3) {
                            Notification::make()
                                ->warning()
                                ->title('Free card holders can only add up to 3 social links!')
                                ->persistent()
                                ->send();

                            $action->cancel();
                        } else {
                            $button->update($data);

                            Notification::make()
                                ->title('Success')
                                ->body('Social link has been updated successfully.')
                                ->success()
                                ->send();

                            $action->cancel();
                        }
                    } catch (\InvalidArgumentException $e) {
                        Notification::make()
                            ->title('Validation Error')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();

                        $action->halt();
                    }
                })
                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['user_id'] = $livewire->ownerRecord->user_id;
                        return $data;
                    }),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
