<?php

namespace App\Filament\Resources\SubCardsResource\RelationManagers;

use App\Models\Video;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Columns\TextColumn;
use Illuminate\Database\Eloquent\Model;
use Filament\Forms\Components\Select;
use Filament\Support\Enums\IconPosition;
use Filament\Notifications\Notification;
use Filament\Tables\Actions\Action;
use Filament\Forms\Components\FileUpload;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;

class VideosRelationManager extends RelationManager
{
    protected static string $relationship = 'videos';

    // public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    // {
    //     //return $ownerRecord->user_type === 'Platinum';
    //     $videos = Video::where('user_id', $ownerRecord->id)->get()->count();
    //     return $ownerRecord->user_type == 'Platinum';
    // }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('title')
                    ->required()
                    ->maxLength(255),
                Select::make('video_type')
                    ->label('Vide Type')
                    ->required()
                    ->searchable()
                    ->options([
                        'youtube' => 'Youtube',
                        'vimeo' => 'Vimeo',
                    ]),
                Forms\Components\TextInput::make('link')
                    ->label('Video Link')
                    ->url()
                    ->required()
                    ->columnSpanFull()
                    ->maxLength(255),

                FileUpload::make('thumbnail')
                    ->label("Custom Video Thumbnail")
                    ->placeholder('Upload Custom Video Thumbnail')
                    ->directory('video_thumbnails')
                    ->disk('public')
                    ->image()
                    ->imageEditor()
                    ->imageResizeMode('cover')
                    ->columnSpanFull()
                    ->imageCropAspectRatio('1280:720')
                    ->imageResizeTargetWidth('1280')
                    ->imageResizeTargetHeight('720')
                    ->getUploadedFileNameForStorageUsing(
                        fn(TemporaryUploadedFile $file): string =>
                        (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                    ),

                Forms\Components\TextInput::make('button_title')
                    ->label('Button Title')
                    ->placeholder('Optional')
                    ->maxLength(255),
                Forms\Components\TextInput::make('button_link')
                    ->label('Button Link')
                    ->url()
                    ->placeholder('Optional')
                    ->maxLength(255),
                Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
                    ->columnSpanFull()
                    ->required(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->where('page_id', $livewire->ownerRecord->id)->orderBy('order')
            )
            ->description('Platinum card holders can only add up to 5 videos')
            // ->recordTitleAttribute('user_id')
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->limit(40)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();

                        if (strlen($state) <= $column->getCharacterLimit()) {
                            return null;
                        }

                        // Only render the tooltip if the column content exceeds the length limit.
                        return $state;
                    }),

                Tables\Columns\TextColumn::make('link')
                    ->searchable()
                    ->limit(40)
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    ->url(fn($record) => $record->link)
                    ->openUrlInNewTab(),
                // ->tooltip(function (TextColumn $column): ?string {
                //     $state = $column->getState();

                //     if (strlen($state) <= $column->getCharacterLimit()) {
                //         return null;
                //     }

                //     // Only render the tooltip if the column content exceeds the length limit.
                //     return $state;
                // }),
                Tables\Columns\TextColumn::make('video_type')
                    ->label('Video Type')
                    ->searchable()
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['user_id'] = $livewire->ownerRecord->user_id;
                        return $data;
                    })
                    ->before(function (Tables\Actions\CreateAction $action, RelationManager $livewire) {
                        if ($livewire->ownerRecord->videos()->count() >= 5) {
                            Notification::make()
                                ->warning()
                                ->title('Platinum card holders can only add up to 5 videos!')
                                ->persistent()
                                ->send();

                            $action->halt(); // <- This is what actually stops the action.
                        }
                    })
                    ->disabled(fn(RelationManager $livewire) => $livewire->ownerRecord->videos()->count() >= 5)

            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
