<?php

namespace App\Filament\Resources;

use App\Filament\Resources\TeamResource\Pages;
use App\Filament\Resources\TeamResource\RelationManagers;
use App\Models\Button;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Str;
use App\Models\SubCard;
use App\Models\User;
use Filament\Notifications\Notification;
use Filament\Forms\Components\Select;
use Filament\Forms\Set;
use Illuminate\Support\HtmlString;
use App\Filament\Resources\TeamResource\RelationManagers\MembersRelationManager;
use App\Settings\CardDefaultsSetting;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Filament\Forms\Components\FileUpload;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

class TeamResource extends Resource
{
    protected static ?string $model = Button::class;

    protected static ?string $navigationIcon = 'heroicon-o-users';

    protected static ?string $navigationGroup = 'Client Management';
    protected static ?string $navigationLabel = 'Teams';
    protected static ?int $navigationSort = 4;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make("Team Details")
                    ->schema([

                        // 👇 Added hidden fields so Filament will save them
                        Forms\Components\Hidden::make('user_id'),
                        Forms\Components\Hidden::make('page_id'),

                        // Select::make('owner')
                        //     ->label('Owner')
                        //     ->placeholder('Search Client or SubCard')
                        //     ->getSearchResultsUsing(function (string $search): array {
                        //         // --- Users ---
                        //         $users = User::query()
                        //             ->where(function ($query) use ($search) {
                        //                 if (strpos($search, '@') === 0 && strlen($search) > 1) {
                        //                     $query->where('username', 'like', "%" . substr($search, 1) . "%");
                        //                 } else {
                        //                     $query->where('firstname', 'like', "%{$search}%")
                        //                         ->orWhere('lastname', 'like', "%{$search}%")
                        //                         ->orWhere('username', 'like', "%{$search}%");
                        //                 }
                        //             })
                        //             ->whereIn('user_type', ['Platinum', 'Gold'])
                        //             ->limit(25)
                        //             ->get()
                        //             ->mapWithKeys(fn($user) => [
                        //                 "user-{$user->id}" => "{$user->firstname} {$user->lastname} - @$user->username (User)"
                        //             ]);

                        //         // --- SubCards ---
                        //         $subCards = SubCard::query()
                        //             ->where(function ($query) use ($search) {
                        //                 $query->where('firstname', 'like', "%{$search}%")
                        //                     ->orWhere('lastname', 'like', "%{$search}%")
                        //                     ->orWhere('username', 'like', "%{$search}%");
                        //             })
                        //             ->limit(25)
                        //             ->get()
                        //             ->mapWithKeys(fn($sub) => [
                        //                 "sub-{$sub->id}" => "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)"
                        //             ]);

                        //         return $users->toArray() + $subCards->toArray();
                        //     })
                        //     ->getOptionLabelUsing(function ($value) {
                        //         if (!$value) return '';

                        //         [$type, $id] = explode('-', $value);

                        //         if ($type === 'user') {
                        //             $user = User::find($id);
                        //             return $user ? "{$user->firstname} {$user->lastname} - @$user->username (User)" : '';
                        //         }

                        //         if ($type === 'sub') {
                        //             $sub = SubCard::find($id);
                        //             return $sub ? "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)" : '';
                        //         }

                        //         return '';
                        //     })
                        //     ->formatStateUsing(function ($state, $record) {
                        //         if (!$record) return null;

                        //         if ($record->page_id && $record->subcard) {
                        //             return 'sub-' . $record->subcard->id;
                        //         }

                        //         if ($record->user_id && $record->user) {
                        //             return 'user-' . $record->user->id;
                        //         }

                        //         return null;
                        //     })
                        //     ->afterStateHydrated(function (Set $set, $state, $record) {
                        //         if (!$record) return;

                        //         if ($record->page_id && $record->subcard) {
                        //             $set('owner', 'sub-' . $record->subcard->id);
                        //         } elseif ($record->user_id && $record->user) {
                        //             $set('owner', 'user-' . $record->user->id);
                        //         }
                        //     })
                        //     ->afterStateUpdated(function (Set $set, $state) {
                        //         if (!$state) return;

                        //         [$type, $id] = explode('-', $state);

                        //         if ($type === 'user') {
                        //             // 👇 set hidden fields instead of saving directly
                        //             $set('user_id', $id);
                        //             $set('page_id', null);
                        //         } elseif ($type === 'sub') {
                        //             // $set('page_id', $id);
                        //             // $set('user_id', null);
                        //             $sub = SubCard::find($id);
                        //             if ($sub) {
                        //                 $set('page_id', $sub->id);
                        //                 $set('user_id', $sub->user_id); // <-- set the main user_id from subcard
                        //             }
                        //         }
                        //     })

                        //     ->helperText(function () {
                        //         $settings = app(CardDefaultsSetting::class);

                        //         return new HtmlString(
                        //             'Only <strong>' . e($settings->term_gold) . '</strong> & <strong>' . e($settings->term_platinum) . '</strong> users can add team(s). SubCards are also allowed.'
                        //         );
                        //     }),

                        Select::make('owner')
                            ->label('Owner')
                            ->disabled(fn() => request()->has('ownerKey'))
                            ->placeholder('Search Client or SubCard')
                            ->getSearchResultsUsing(function (string $search): array {
                                // --- Users ---
                                $users = User::query()
                                    ->where(function ($query) use ($search) {
                                        if (strpos($search, '@') === 0 && strlen($search) > 1) {
                                            $query->where('username', 'like', "%" . substr($search, 1) . "%");
                                        } else {
                                            $query->where('firstname', 'like', "%{$search}%")
                                                ->orWhere('lastname', 'like', "%{$search}%")
                                                ->orWhere('username', 'like', "%{$search}%");
                                        }
                                    })
                                    ->whereIn('user_type', ['Platinum', 'Gold'])
                                    ->limit(25)
                                    ->get()
                                    ->mapWithKeys(fn($user) => [
                                        "user-{$user->id}" => "{$user->firstname} {$user->lastname} - @$user->username (User)"
                                    ]);

                                // --- SubCards ---
                                $subCards = SubCard::query()
                                    ->where(function ($query) use ($search) {
                                        $query->where('firstname', 'like', "%{$search}%")
                                            ->orWhere('lastname', 'like', "%{$search}%")
                                            ->orWhere('username', 'like', "%{$search}%");
                                    })
                                    ->limit(25)
                                    ->get()
                                    ->mapWithKeys(fn($sub) => [
                                        "sub-{$sub->id}" => "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)"
                                    ]);

                                return $users->toArray() + $subCards->toArray();
                            })
                            ->getOptionLabelUsing(function ($value) {
                                if (!$value) return '';

                                [$type, $id] = explode('-', $value);

                                if ($type === 'user') {
                                    $user = User::find($id);
                                    return $user ? "{$user->firstname} {$user->lastname} - @$user->username (User)" : '';
                                }

                                if ($type === 'sub') {
                                    $sub = SubCard::find($id);
                                    return $sub ? "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)" : '';
                                }

                                return '';
                            })
                            ->formatStateUsing(function ($state, $record) {
                                if (!$record) return null;

                                if ($record->page_id && $record->subcard) {
                                    return 'sub-' . $record->subcard->id;
                                }

                                if ($record->user_id && $record->user) {
                                    return 'user-' . $record->user->id;
                                }


                                return null;
                            })
                            ->afterStateHydrated(function (Set $set, $state, $record) {
                                if (!$record) return;

                                if ($record->page_id && $record->subcard) {
                                    $set('owner', 'sub-' . $record->subcard->id);
                                } elseif ($record->user_id && $record->user) {
                                    $set('owner', 'user-' . $record->user->id);
                                }
                            })
                            ->afterStateUpdated(function (Set $set, $state) {
                                if (!$state) return;

                                [$type, $id] = explode('-', $state);

                                if ($type === 'user') {
                                    // 👇 set hidden fields instead of saving directly
                                    $set('user_id', $id);
                                    $set('page_id', null);
                                } elseif ($type === 'sub') {
                                    // $set('page_id', $id);
                                    // $set('user_id', null);

                                    $sub = SubCard::find($id);
                                    if ($sub) {
                                        $set('page_id', $sub->id);
                                        $set('user_id', $sub->user_id); // <-- set the main user_id from subcard
                                    }
                                }
                            })
                            ->searchable()
                            ->required()
                            ->columnSpanFull()
                            ->helperText(new HtmlString('Only <strong>Gold</strong> & <strong>Platinum</strong> users can add sales page(s). SubCards are also allowed.')),

                        FileUpload::make('thumbnail')
                            ->label('Custom Thumbnail')
                            ->placeholder('Upload Custom Thumbnail')
                            ->directory('thumbnails')
                            ->uploadingMessage('Uploading Custom Thumbnail. Please wait...')
                            ->helperText("This will be displayed on the Team's page. If no custom thumbnail, the user's profile picture will be displayed.")
                            ->image()
                            ->imageEditor()
                            ->multiple(false) // ✅ only one image
                            ->columnSpanFull()
                            ->imageResizeMode('cover')
                            ->imageCropAspectRatio('1:1')
                            ->imageResizeTargetWidth('520')
                            ->imageResizeTargetHeight('520')
                            ->getUploadedFileNameForStorageUsing(
                                fn(TemporaryUploadedFile $file): string =>
                                (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                            )
                            ->imageEditorAspectRatios([
                                '1:1',
                            ]),

                        Forms\Components\TextInput::make('title')
                            ->maxLength(300)
                            ->placeholder('ex: Meet the Team, Meet the Guests, Meet the Sponsors')
                            ->columnSpanFull()
                            ->required(),

                        Forms\Components\Textarea::make('description')
                            ->label('Team Description')
                            ->maxLength(1000)
                            ->columnSpanFull(),

                        Forms\Components\Toggle::make('is_active')
                            ->label('Active')
                            ->default(true)
                            ->columnSpanFull()
                            ->required(),

                        Forms\Components\Hidden::make('type')
                            ->default('team'),
                    ]),
            ]);
    }


    public static function table(Table $table): Table
    {
        return $table
            ->modifyQueryUsing(
                fn(Builder $query) =>
                $query->where('type', 'team')->orderBy('order', 'desc')
            )
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('owner')
                    ->label('Owner')
                    ->getStateUsing(function ($record) {
                        if (!$record) return 'N/A';

                        // If page_id is null, show main user
                        if (is_null($record->page_id) && $record->user) {
                            $user = $record->user;
                            return "{$user->firstname} {$user->lastname} - @$user->username (Main Card)";
                        }

                        // If page_id exists, show subcard
                        if ($record->page_id && $record->subcard) {
                            $sub = $record->subcard;
                            return "{$sub->firstname} {$sub->lastname} - @$sub->username (Sub Card)";
                        }

                        return 'N/A';
                    })
                    ->sortable()
                    // ->searchable(['user.firstname', 'user.lastname', 'user.username', 'subcard.firstname', 'subcard.lastname', 'subcard.username']),
                    ->searchable(query: function (Builder $query, string $search): Builder {
                        return $query
                            ->whereHas('user', function ($q) use ($search) {
                                $q->where('firstname', 'like', "%{$search}%")
                                    ->orWhere('lastname', 'like', "%{$search}%")
                                    ->orWhere('username', 'like', "%{$search}%");
                            })
                            ->orWhereHas('subcard', function ($q) use ($search) {
                                $q->where('firstname', 'like', "%{$search}%")
                                    ->orWhere('lastname', 'like', "%{$search}%")
                                    ->orWhere('username', 'like', "%{$search}%");
                            });
                    }),

                Tables\Columns\TextColumn::make('title'),
                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->sortable()
                    ->formatStateUsing(fn($state) => $state ? Carbon::parse($state)->format('M d, Y') : null),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\Action::make('view_card')
                    ->label('View')
                    ->icon('heroicon-o-eye')
                    ->url(function ($record) {
                        if (!$record) return '#';

                        $baseUrl = app(\App\Settings\CardDefaultsSetting::class)->site_url;

                        if ($record->page_id && $record->subcard) {
                            return $baseUrl . $record->subcard->username . '/team/' . $record->id;
                        }

                        if ($record->user) {
                            return $baseUrl . $record->user->username . '/team/' . $record->id;
                        }

                        return '#';
                    })
                    ->openUrlInNewTab()
                    ->color('info'),
                Tables\Actions\EditAction::make(),
                Tables\Actions\Action::make('customAction')
                    ->icon('heroicon-o-document-duplicate')
                    ->label('Duplicate')
                    ->color('warning')
                    ->action(function (Model $record) {
                        DB::transaction(function () use ($record) {

                            // Duplicate logic here
                            $newRecord = $record->replicate(); // Clone the record

                            $newRecord->title = $record->title;
                            $newRecord->order = $record->order;
                            $newRecord->is_active = $record->is_active;
                            $newRecord->type = $record->type;
                            $newRecord->view = $record->view;
                            $newRecord->description = $record->description;
                            $newRecord->created_at = Carbon::now();
                            $newRecord->updated_at = Carbon::now();

                            $newRecord->save();

                            //Duplicate members
                            if ($record->members) {
                                foreach ($record->members as $member) {
                                    $newTeam = $member->replicate();
                                    $newTeam->team_id = $newRecord->id;
                                    $newTeam->member_id = $member->member_id;
                                    $newTeam->card_type = $member->card_type;
                                    $newTeam->order = $member->order;
                                    $newTeam->custom_card_link = $member->custom_card_link;
                                    $newTeam->save();
                                }
                            }

                            // Optional: You could redirect or notify
                            Notification::make()
                                ->title('Success')
                                ->body('Team duplicated successfully.')
                                ->success()
                                ->send();
                        });
                    }),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),

                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            MembersRelationManager::class
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListTeams::route('/'),
            'create' => Pages\CreateTeam::route('/create'),
            'edit' => Pages\EditTeam::route('/{record}/edit'),
        ];
    }
}
