<?php

namespace App\Filament\Resources;

use App\Filament\Resources\UserAdResource\Pages;
use App\Filament\Resources\UserAdResource\RelationManagers;
use App\Models\Ad;
use App\Models\User;
use Filament\Tables\Actions\Action;
use Filament\Forms;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;
use Filament\Forms\Set;
use Filament\Notifications\Notification;
use Filament\Forms\Components\Select;

class UserAdResource extends Resource
{
    protected static ?string $model = Ad::class;

    protected static ?string $navigationIcon = 'heroicon-o-megaphone';
    protected static ?string $navigationGroup = 'Client Management';
    protected static ?int $navigationSort = 6;

    // hide this resource
    public static function shouldRegisterNavigation(): bool
    {
        return false;
    }

    public static function form(Form $form): Form
    {
        return $form->schema([
            Select::make('user_id')
                ->columnSpanFull()
                ->label('Owner')
                ->getSearchResultsUsing(
                    fn(string $search): array =>
                    User::where(function ($query) use ($search) {
                        $query->where('firstname', 'like', "%{$search}%")
                            ->orWhere('lastname', 'like', "%{$search}%");
                    })
                        ->where('user_type', 'Platinum')  // Only Platinum users
                        ->limit(50)
                        ->pluck(DB::raw('CONCAT(firstname, " ", lastname)'), 'id')
                        ->toArray()
                )
                ->getOptionLabelUsing(
                    fn($value) =>
                    optional(User::find($value))->firstname . ' ' . optional(User::find($value))->lastname
                )
                ->searchable()
                ->required(),
            // ->helperText(new HtmlString('Only <strong>Platinum</strong> card holders can add video(s)'))
            // ->afterStateUpdated(
            //     function (Set $set, $state) {
            //         // Check if the selected user has 5 or more videos
            //         $user = User::find($state);
            //         if ($user->user_type == "Platinum" && $user->videos()->count() >= 5) {
            //             // Trigger a notification
            //             Notification::make()
            //                 ->title('Limit reached')
            //                 ->body('Platinum card holders can only add up to 5 videos!')
            //                 ->warning()
            //                 ->send();
            //             // Optionally, you can reset the state or show a validation error:
            //             $set('user_id', null); // Reset the user_id if limit is reached
            //         }
            //     }
            // ),
            FileUpload::make('image')
                ->label('Upload Ad Banner (2104x345)')
                ->placeholder('Upload Ad Banner')
                ->directory('ads')
                ->disk('public')
                ->image()
                ->imageResizeTargetWidth('2104')
                ->imageResizeTargetHeight('345')
                ->columnSpanFull()
                ->getUploadedFileNameForStorageUsing(
                    fn(TemporaryUploadedFile $file): string =>
                    (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                ),

            Forms\Components\TextInput::make('caption')->label('Ad Title')
                ->required(),
            Forms\Components\TextInput::make('link')->required(),
            Forms\Components\Select::make('position')
                ->required()
                ->label('Position')
                ->columnSpanFull()
                ->options([
                    'top' => 'Top',
                    'bottom' => 'Bottom',
                ]),
            Forms\Components\Toggle::make('is_active')
                ->label('Active Ad')
                ->default(true)
                ->required(),
        ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            // ->reorderable('order')
            // ->reorderRecordsTriggerAction(
            //     fn(Action $action, bool $isReordering) => $action
            //         ->button()
            //         ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            // )
            // ->description('Ads posted here will display on Free users.')
            ->columns([
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean(),
                Tables\Columns\TextColumn::make('user.firstname')
                    ->label('Owner')
                    ->getStateUsing(function ($record) {
                        return $record->user ? $record->user->firstname . ' ' . $record->user->lastname : 'N/A';
                    })
                    ->sortable()
                    ->searchable(['firstname', 'lastname']),
                Tables\Columns\TextColumn::make('caption')
                    ->label('Ad Title'),
                Tables\Columns\TextColumn::make('position')
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->filters([
                //
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }


    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery()->where('is_admin', false)->orderBy('order');
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListUserAds::route('/'),
            'create' => Pages\CreateUserAd::route('/create'),
            'edit' => Pages\EditUserAd::route('/{record}/edit'),
        ];
    }
}
