<?php

namespace App\Filament\Resources\UserResource\Pages;

use App\Filament\Resources\UserResource;
use Filament\Resources\Pages\CreateRecord;
use App\Models\User;
use App\Models\SubCard;
use App\Models\ColorPreset;
use App\Models\Referral;
use App\Services\CommissionService;
use Illuminate\Support\Facades\Http;

class CreateUser extends CreateRecord
{
    protected static string $resource = UserResource::class;

    // capture full form data including non-model fields
    protected array $rawFormData = [];

    public $availableStates = [];

    public function fetchStates($country)
    {
        if (!$country) {
            $this->availableStates = [];
            return;
        }

        $response = Http::post('https://countriesnow.space/api/v0.1/countries/states', [
            'country' => $country,
        ]);

        if ($response->ok()) {
            $states = data_get($response->json(), 'data.states', []);
            $this->availableStates = collect($states)
                ->mapWithKeys(fn($state) => [$state['name'] => $state['name']])
                ->toArray();
        } else {
            $this->availableStates = [];
        }
    }

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        // Save full form data (including non-model fields like preset, button_style)
        $this->rawFormData = $data;

        return $data; // must return for actual user creation
    }

    protected function afterCreate(): void
    {
        /** @var User $user */
        $user = $this->record;

        $formState = $this->rawFormData;

        // --- Referral Logic ---
        $referrerValue = $formState['referrer_id_main'] ?? null;

        if ($referrerValue) {
            if (str_starts_with($referrerValue, 'user_')) {
                $id = (int) str_replace('user_', '', $referrerValue);

                Referral::updateOrCreate(
                    ['referred_id' => $user->id],
                    [
                        'referrer_id_main' => $id,
                        'referrer_id_sub'  => null,
                        'referrer_type'    => 'main',
                    ]
                );
            } elseif (str_starts_with($referrerValue, 'sub_')) {
                $id = (int) str_replace('sub_', '', $referrerValue);

                Referral::updateOrCreate(
                    ['referred_id' => $user->id],
                    [
                        'referrer_id_main' => null,
                        'referrer_id_sub'  => $id,
                        'referrer_type'    => 'sub',
                    ]
                );
            }
        }

        // --- Commission ---
        if (in_array($user->access, ['Month', 'Year'], true)) {
            app(CommissionService::class)->createCommission($user);
        }

        // --- Color Preset & Defaults ---
        if (!empty($formState['preset']) && $formState['preset'] !== 'Custom') {
            $preset = ColorPreset::find($formState['preset']);

            if ($preset) {
                $formState['color_settings_custom_button']       = $preset->custom_button;
                $formState['color_settings_custom_button_text']  = $preset->custom_button_text;
                $formState['color_settings_link']                = $preset->link;
                $formState['color_settings_profile_pic_border']  = $preset->profile_pic_border;
                $formState['color_settings_card_light']          = $preset->card_light;
                $formState['color_settings_card_dark']           = $preset->card_dark;
                $formState['color_settings_text_light']          = $preset->text_light;
                $formState['color_settings_text_dark']           = $preset->text_dark;
            }
        }

        $defaults = [
            'button_style'                       => 'rounded',
            'color_settings_custom_button'       => '#facc15',
            'color_settings_custom_button_text'  => '#000000',
            'color_settings_link'                => '#ca8a04',
            'color_settings_profile_pic_border'  => '#ca8a04',
            'color_settings_card_light'          => '#ffffff',
            'color_settings_card_dark'           => '#18181b',
            'color_settings_text_light'          => '#1e293b',
            'color_settings_text_dark'           => '#cbd5e1',
            'profile_layout'                     => 'default',
            'bio_text'                           => 'My Bio',
            'click_gallery'                      => 'Click here',
            'click_videos'                       => 'Click here',
            'click_gallery_link'                 => null,
            'click_videos_link'                  => null,
            'searchable'                         => true,
            'display_username'                   => true,
            'display_short_bio'                  => true,
            'display_country'                    => true,
            'connect_main'                       => true,
            'connect_bio'                        => false,
            'gallery_main'                       => true,
            'gallery_bio'                        => false,
            'product_price_enabled'              => true,
            'services_price_enabled'             => true,
            'click_videos_enabled'               => false,
            'click_gallery_enabled'              => false,
            'footer_enabled'                     => true,
            'getacard_enabled'                   => false,
            'shop_text'                          => 'Shop',
            'shop_more'                          => null,
            'shop_thumbnail'                     => null,
            'services_text'                      => 'Services',
            'contact_text'                       => 'Contact me',
            'gallery_two_slide'                  => true,
            'follow_text'                        => 'Follow me',
            'event_text'                         => 'Events & Promotions',
            'payment_text'                       => 'Payment Links',
        ];

        foreach ($defaults as $key => $default) {
            $value = $formState[$key] ?? $default;

            $user->settings()->updateOrCreate(
                ['setting_key' => $key, 'page_id' => null],
                ['setting_value' => $value]
            );
        }

        if (!empty($formState['preset'])) {
            $user->settings()->updateOrCreate(
                ['setting_key' => 'preset', 'page_id' => null],
                ['setting_value' => $formState['preset']]
            );
        }
    }

    protected function getRedirectUrl(): string
    {
        return static::$resource::getUrl('edit', ['record' => $this->record->getKey()]);
    }
}
