<?php

namespace App\Filament\Resources\UserResource\Pages;

use App\Filament\Resources\UserResource;
use Filament\Actions;
use Filament\Resources\Pages\EditRecord;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\Schema;
use Filament\Tables\Actions\Action;
use Illuminate\Database\Eloquent\Model;
use Filament\Notifications\Notification;
use Carbon\Carbon;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use App\Models\Referral;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;

class EditUser extends EditRecord
{
    protected static string $resource = UserResource::class;
    public $availableStates = [];

    public function mount($record): void
    {
        parent::mount($record);

        if ($this->record->country) {
            $this->fetchStates($this->record->country);
        }
    }

    public $defaults = [
        'button_style'                          => 'rounded',
        'color_settings_custom_button'          => '#facc15',
        'color_settings_custom_button_text'     => '#000000',
        'color_settings_link'                   => '#ca8a04',
        'color_settings_profile_pic_border'     => '#ca8a04',
        'color_settings_card_light'             => '#ffffff',
        'color_settings_card_dark'              => '#18181b',
        'color_settings_text_light'             => '#1e293b',
        'color_settings_text_dark'              => '#cbd5e1',
        'profile_layout'                        => 'default',
        'bio_text'                              => 'My Bio',
        'click_gallery'                         => 'Click here',
        'click_videos'                          => 'Click here',
        'click_gallery_link'                    => NULL,
        'click_videos_link'                     => NULL,
        'searchable'                            => true,
        'display_username'                      => true,
        'display_short_bio'                     => true,
        'display_country'                       => true,
        'connect_main'                          => true,
        'connect_bio'                           => false,
        'gallery_main'                          => true,
        'gallery_bio'                           => false,
        'product_price_enabled'                 => true,
        'services_price_enabled'                => true,
        'click_videos_enabled'                  => false,
        'click_gallery_enabled'                 => false,
        'footer_enabled'                        => true,
        'getacard_enabled'                      => false,
        'shop_text'                             => 'Shop',
        'shop_thumbnail'                        => NULL,
        'shop_more'                             => NULL,
        'services_text'                         => 'Services',
        'contact_text'                          => 'Contact me',
        'gallery_two_slide'                     => true,
        'follow_text'                           => 'Follow me',
        'event_text'                            => 'Events & Promotions',
        'payment_text'                          => 'Payment Links',
    ];

    // public function fetchStates($country)
    // {
    //     if (!$country) {
    //         $this->availableStates = [];
    //         return;
    //     }

    //     $response = Http::post('https://countriesnow.space/api/v0.1/countries/states', [
    //         'country' => $country,
    //     ]);

    //     if ($response->ok()) {
    //         $states = data_get($response->json(), 'data.states', []);

    //         // Convert API response to key => value for Filament select
    //         $this->availableStates = collect($states)
    //             ->mapWithKeys(fn($state) => [$state['state_code'] => $state['name']])
    //             ->toArray();
    //     } else {
    //         $this->availableStates = [];
    //     }
    // }

    public function fetchStates($country)
    {
        if (!$country) {
            $this->availableStates = [];
            return;
        }

        $response = Http::post('https://countriesnow.space/api/v0.1/countries/states', [
            'country' => $country,
        ]);

        if ($response->ok()) {
            $states = data_get($response->json(), 'data.states', []);

            // key => value array for Filament select
            $this->availableStates = collect($states)
                ->mapWithKeys(fn($state) => [$state['name'] => $state['name']])
                ->toArray();
        } else {
            $this->availableStates = [];
        }
    }

    protected function afterSave(): void
    {
        $referrerValue = $this->form->getState()['referrer_id_main'] ?? null;

        if ($referrerValue) {
            if (str_starts_with($referrerValue, 'user_')) {
                $referrerId = (int) str_replace('user_', '', $referrerValue);

                Referral::updateOrCreate(
                    ['referred_id' => $this->record->id],
                    [
                        'referrer_id_main' => $referrerId,
                        'referrer_id_sub'  => null,
                        'referrer_type'    => 'main',
                    ]
                );
            } elseif (str_starts_with($referrerValue, 'sub_')) {
                $subId = (int) str_replace('sub_', '', $referrerValue);

                Referral::updateOrCreate(
                    ['referred_id' => $this->record->id],
                    [
                        'referrer_id_main' => null,
                        'referrer_id_sub'  => $subId,
                        'referrer_type'    => 'sub',
                    ]
                );
            }
        }
    }

    protected function mutateFormDataBeforeFill(array $data): array
    {
        $user = $this->record;

        if ($user->referral) {
            if ($user->referral->referrer_type === 'main' && $user->referral->referrer_id_main) {
                $data['referrer_id_main'] = 'user_' . $user->referral->referrer_id_main;
            } elseif ($user->referral->referrer_type === 'sub' && $user->referral->referrer_id_sub) {
                $data['referrer_id_main'] = 'sub_' . $user->referral->referrer_id_sub;
            }
        }

        // This will fetch settings with page_id = null
        foreach ($user->settings()->whereNull('page_id')->get() as $setting) {
            $data[$setting->setting_key] = $setting->setting_value;
        }

        foreach ($this->defaults as $key => $value) {
            if (!array_key_exists($key, $data) || is_null($data[$key])) {
                $data[$key] = $value;
            }
        }

        return $data;
    }

    protected function mutateFormDataBeforeSave(array $data): array
    {
        $user = $this->record;

        // If a preset was selected (and not "Custom"), load the preset and override the colors
        if (!empty($data['preset']) && $data['preset'] !== 'Custom') {
            $preset = \App\Models\ColorPreset::find($data['preset']);

            if ($preset) {
                $data['color_settings_custom_button'] = $preset->custom_button;
                $data['color_settings_custom_button_text'] = $preset->custom_button_text;
                $data['color_settings_link'] = $preset->link;
                $data['color_settings_profile_pic_border'] = $preset->profile_pic_border;
                $data['color_settings_card_light'] = $preset->card_light;
                $data['color_settings_card_dark'] = $preset->card_dark;
                $data['color_settings_text_light'] = $preset->text_light;
                $data['color_settings_text_dark'] = $preset->text_dark;
            }
        }

        // Apply defaults to missing/null keys
        foreach ($this->defaults as $key => $value) {
            if (!array_key_exists($key, $data) || is_null($data[$key])) {
                $data[$key] = $value;
            }
        }

        // ✅ Handle old `shop_thumbnail` file removal or replacement
        $oldShopThumb = optional(
            $user->settings()->where('setting_key', 'shop_thumbnail')->whereNull('page_id')->first()
        )->setting_value;

        if (
            array_key_exists('shop_thumbnail', $data)
            && $oldShopThumb
            && $data['shop_thumbnail'] !== $oldShopThumb
            && Storage::disk('public')->exists($oldShopThumb)
        ) {
            Storage::disk('public')->delete($oldShopThumb);
        }

        foreach ($data as $key => $value) {
            if (!Schema::hasColumn('users', $key)) {
                // Ensure user_id is set in updateOrCreate
                $user->settings()->updateOrCreate(
                    ['setting_key' => $key, 'page_id' => null],
                    ['setting_value' => $value]
                );

                unset($data[$key]);
            }
        }

        return $data;
    }

    protected function getHeaderActions(): array
    {
        return [
            // Actions\ViewAction::make(),

            Actions\Action::make('customAction')
                ->icon('heroicon-o-eye')
                ->label('View') // Button text
                ->url(fn($record) => app(CardDefaultsSetting::class)->site_url . $record->username)
                ->openUrlInNewTab()
                ->color('info'),

            Actions\Action::make('customAction')
                ->icon('heroicon-o-document-duplicate')
                ->label('Duplicate')
                ->color('warning')
                ->action(function (Model $record) {
                    DB::transaction(function () use ($record) {

                        // Helper to duplicate a file into a specific folder
                        // $duplicateFile = function (?string $filePath) {
                        //     if (!$filePath || !Storage::disk('public')->exists($filePath)) {
                        //         return null;
                        //     }

                        //     $directory = dirname($filePath); // e.g. profile_pic
                        //     $extension = pathinfo($filePath, PATHINFO_EXTENSION);
                        //     $originalName = pathinfo($filePath, PATHINFO_FILENAME);
                        //     $newFileName = $originalName . '_' . Str::random(6) . '.' . $extension;
                        //     $newPath = $directory . '/' . $newFileName;

                        //     Storage::disk('public')->makeDirectory($directory);
                        //     Storage::disk('public')->copy($filePath, $newPath);

                        //     return $newPath;
                        // };

                        $duplicateFile = function (?string $filePath) {
                            if (!$filePath || !Storage::disk('public')->exists($filePath)) {
                                return null;
                            }

                            $directory = dirname($filePath); // e.g. profile_pic
                            $extension = pathinfo($filePath, PATHINFO_EXTENSION);
                            $originalName = pathinfo($filePath, PATHINFO_FILENAME);

                            // Limit the original name length to avoid filesystem errors
                            $maxBaseLength = 50; // adjust as needed (e.g. 50 chars)
                            $shortName = Str::limit($originalName, $maxBaseLength, '');

                            $newFileName = $shortName . '_' . Str::random(6) . '.' . $extension;
                            $newPath = $directory . '/' . $newFileName;

                            Storage::disk('public')->makeDirectory($directory);
                            Storage::disk('public')->copy($filePath, $newPath);

                            return $newPath;
                        };

                        // Duplicate logic here
                        $newRecord = $record->replicate(); // Clone the record

                        // Modify unique fields to avoid conflicts
                        $newRecord->username = $record->username . '_' . uniqid(); // Add random suffix
                        $newRecord->email = uniqid() . '_' . $record->email; // Or generate a dummy email if needed

                        // Duplicate each image to its respective folder
                        $newRecord->thumbnail = $duplicateFile($record->thumbnail);
                        $newRecord->profile_pic = $duplicateFile($record->profile_pic);
                        $newRecord->cover_pic = $duplicateFile($record->cover_pic);

                        //dont replicate the following
                        $newRecord->phone = NULL;
                        $newRecord->email = NULL;
                        $newRecord->created_at = Carbon::now();
                        $newRecord->updated_at = Carbon::now();

                        $newRecord->email_verified_at = NULL;
                        $newRecord->password = NULL;
                        $newRecord->firstname = NULL;
                        $newRecord->lastname = NULL;
                        $newRecord->gender = NULL;
                        $newRecord->job_title = NULL;
                        $newRecord->bio = NULL;
                        $newRecord->bio_long = NULL;
                        $newRecord->company = NULL;
                        $newRecord->company_website = NULL;
                        $newRecord->country = NULL;
                        // $newRecord->user_type = NULL;
                        $newRecord->sponsor_id = NULL;
                        $newRecord->stripe_id = NULL;
                        $newRecord->pm_type = NULL;
                        $newRecord->pm_last_four = NULL;
                        $newRecord->trial_ends_at = NULL;
                        $newRecord->access = NULL;

                        $newRecord->is_duplicated = true;

                        $newRecord->save();

                        //replicate settings
                        if ($record->settings) {
                            foreach ($record->settings as $settings) {
                                if (is_null($settings->page_id)) {
                                    $newSettings = $settings->replicate();
                                    $newSettings->user_id = $newRecord->id;
                                    $newSettings->save();
                                }
                            }
                        }

                        //replicate buttons
                        // if ($record->buttons) {
                        //     foreach ($record->buttons()->orderBy('order')->get() as $buttons) {
                        //         if (is_null($buttons->page_id)) {
                        //             $newButtons = $buttons->replicate();
                        //             $newButtons->user_id = $newRecord->id;
                        //             $newButtons->order = $buttons->order;
                        //             $newButtons->save();
                        //         }
                        //     }
                        // }

                        if ($record->buttons()->exists()) {
                            foreach ($record->buttons()->orderBy('order')->get() as $button) {
                                // Skip buttons attached to a page
                                if (!is_null($button->page_id)) {
                                    continue;
                                }

                                // Replicate the button
                                $newButton = $button->replicate();
                                $newButton->user_id = $newRecord->id;
                                $newButton->order = $button->order;
                                $newButton->save();

                                // If it's a "team" type button, replicate its members
                                if ($newButton->type === 'team') {
                                    foreach ($button->members as $member) {
                                        $newMember = $member->replicate();
                                        $newMember->team_id = $newButton->id;
                                        $newMember->member_id = $member->member_id;
                                        $newMember->card_type = $member->card_type;
                                        $newMember->order = $member->order;
                                        $newMember->custom_card_link = $member->custom_card_link;
                                        $newMember->save();
                                    }
                                }
                            }
                        }

                        //replicate socials
                        if ($record->socials) {
                            foreach ($record->socials()->orderBy('order')->get() as $socials) {
                                if (is_null($socials->page_id)) {
                                    $newSocials = $socials->replicate();
                                    $newSocials->user_id = $newRecord->id;
                                    $newSocials->order = $socials->order;
                                    $newSocials->save();
                                }
                            }
                        }

                        //replicate videos
                        if ($record->videos) {
                            foreach ($record->videos()->orderBy('order')->get() as $videos) {
                                if (is_null($videos->page_id)) {
                                    $newVideos = $videos->replicate();
                                    $newVideos->user_id = $newRecord->id;
                                    $newVideos->order = $videos->order;
                                    $newVideos->save();
                                }
                            }
                        }

                        //replicate ads
                        if ($record->ads) {
                            foreach ($record->ads()->orderBy('order')->get() as $ads) {
                                if (is_null($ads->page_id)) {
                                    $newAds = $ads->replicate();
                                    $newAds->image = $duplicateFile($ads->image);
                                    $newAds->user_id = $newRecord->id;
                                    $newAds->order = $ads->order;
                                    $newAds->save();
                                }
                            }
                        }

                        //replicate driving directions
                        if ($record->directions) {
                            foreach ($record->directions()->orderBy('order')->get() as $directions) {
                                if (is_null($directions->page_id)) {
                                    $newDirections = $directions->replicate();
                                    $newDirections->user_id = $newRecord->id;
                                    $newDirections->order = $directions->order;
                                    $newDirections->save();
                                }
                            }
                        }

                        // replicate products and their media
                        if ($record->products) {
                            foreach ($record->products()->orderBy('order')->get() as $product) {
                                if (is_null($product->page_id)) {
                                    $newProduct = $product->replicate();
                                    $newProduct->featured_image = $duplicateFile($product->featured_image);
                                    $newProduct->user_id = $newRecord->id;
                                    $newProduct->order = $product->order;
                                    $newProduct->save();

                                    // replicate associated product media
                                    if ($product->media && $product->media()->exists()) {
                                        foreach ($product->media()->orderBy('order')->get() as $media) {
                                            $newMedia = $media->replicate();
                                            // ✅ Only duplicate the file if it's an image
                                            if ($media->type === 'image') {
                                                $newPath = $duplicateFile($media->path);

                                                // skip if file missing or invalid
                                                if (!$newPath) {
                                                    continue;
                                                }

                                                $newMedia->path = $newPath;
                                            } else {
                                                // non-image types (video, pdf, etc.) keep the same path
                                                $newMedia->path = $media->path;
                                            }

                                            $newMedia->product_id = $newProduct->id; // link to new product
                                            // $newMedia->user_id = $newRecord->id; // link to new duplicated user/card
                                            $newMedia->order = $media->order;
                                            $newMedia->save();
                                        }
                                    }
                                }
                            }
                        }

                        //replicate services
                        if ($record->services) {
                            foreach ($record->services()->orderBy('order')->get() as $services) {
                                if (is_null($services->page_id)) {
                                    $newServices = $services->replicate();
                                    $newServices->service_image = $duplicateFile($services->service_image);
                                    $newServices->user_id = $newRecord->id;
                                    $newServices->order = $services->order;
                                    $newServices->save();
                                }
                            }
                        }

                        //replicate images
                        if ($record->images) {
                            foreach ($record->images()->orderBy('order')->get() as $images) {
                                if (is_null($images->page_id)) {
                                    $newImages = $images->replicate();
                                    $newImages->image_path = $duplicateFile($images->image_path);
                                    $newImages->user_id = $newRecord->id;
                                    $newImages->order = $images->order;
                                    $newImages->save();
                                }
                            }
                        }

                        //replicate events
                        if ($record->events) {
                            foreach ($record->events()->orderBy('order')->get() as $events) {
                                if (is_null($events->page_id)) {
                                    $newEvents = $events->replicate();
                                    $newEvents->event_poster = $duplicateFile($events->event_poster);
                                    $newEvents->user_id = $newRecord->id;
                                    $newEvents->order = $events->order; // ✅ preserve original order
                                    $newEvents->save();
                                }
                            }
                        }

                        // Optional: You could redirect or notify
                        Notification::make()
                            ->title('Success')
                            ->body('Card duplicated successfully.')
                            ->success()
                            ->send();
                    });
                }),

            Actions\DeleteAction::make()
                ->icon('heroicon-o-trash'),
            Actions\Action::make('save')
                ->label('Save')
                ->icon('heroicon-o-check-circle')
                ->color('success')
                ->action('save'),
        ];
    }
}
