<?php

namespace App\Filament\Resources\UserResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Support\Enums\IconPosition;
use Filament\Tables\Actions\Action;
use Filament\Forms\Components\FileUpload;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;
use App\Settings\CardDefaultsSetting;

class AdRelationManager extends RelationManager
{
    protected static string $relationship = 'ads';

    public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    {
        return in_array($ownerRecord->user_type, ['Basic', 'Gold', 'Platinum']);
    }

    public function form(Form $form): Form
    {
        $settings = app(CardDefaultsSetting::class);
        $width = $settings->ad_banner_w;
        $height = $settings->ad_banner_h;
        $aspectRatio = "{$width}:{$height}";

        return $form
            ->schema([
                FileUpload::make('image')
                    ->label("Upload Ad Banner ({$width}x{$height})")
                    ->placeholder('Upload Ad Banner')
                    ->directory('ads')
                    ->disk('public')
                    ->image()
                    ->required()
                    ->imageEditor()
                    ->imageCropAspectRatio($aspectRatio) // This sets the default crop shape
                    ->imageEditorAspectRatios([$aspectRatio]) // Only allow this one crop shape
                    ->imageResizeMode('cover') // Resize after crop
                    ->imageResizeTargetWidth($width)
                    ->imageResizeTargetHeight($height)
                    ->columnSpanFull()
                    ->getUploadedFileNameForStorageUsing(
                        fn(TemporaryUploadedFile $file): string =>
                        (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                    ),

                Forms\Components\TextInput::make('caption')->label('Ad Title')
                    ->required(),
                Forms\Components\TextInput::make('link')
                    ->url()
                    ->required(),
                Forms\Components\Select::make('position')
                    ->required()
                    ->label('Position')
                    ->columnSpanFull()
                    ->options([
                        'top' => 'Top',
                        'bottom' => 'Bottom',
                    ]),
                Forms\Components\Toggle::make('is_active')
                    ->label('Active Ad')
                    ->default(true)
                    ->required(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->whereNull('page_id')
                    ->where('is_admin', false)->orderBy('order')
            )
            ->description("Ads posted here will display on the owner's card.")
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('caption')
                    ->label('Ad Title'),
                Tables\Columns\TextColumn::make('position')
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
