<?php

namespace App\Filament\Resources\UserResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Notifications\Notification;
use Filament\Tables\Actions\Action;
use Filament\Support\Enums\IconPosition;
use App\Models\Button;
use Filament\Tables\Actions\CreateAction;
use Filament\Tables\Actions\EditAction;
use Illuminate\Database\Eloquent\Model;

class CommunitiesRelationManager extends RelationManager
{
    protected static string $relationship = 'buttons';

    protected static ?string $title = 'Communities';
    protected static ?string $modelLabel = 'Communities';

    public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    {
        return in_array($ownerRecord->user_type, ['Influencer', 'Sponsored', 'Gold', 'Platinum', 'Basic']);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('type')
                    ->label('Community Type')
                    ->columnSpanFull()
                    ->options([
                        'fb_community' => 'Facebook Community',
                        'whatsapp_community' => 'WhatsApp Community',
                        'discord_community' => 'Discord Community',
                        'telegram_community' => 'Telegram Community',
                        'skool_community' => 'Skool Community'
                    ]),
                Forms\Components\TextInput::make('title')
                    ->required()
                    ->columnSpanFull()
                    ->maxLength(255),
                Forms\Components\TextInput::make('link')
                    ->columnSpanFull()
                    ->required()
                    ->maxLength(255),
                Forms\Components\Toggle::make('is_active')
                    ->columnSpanFull()
                    ->label('Active')
                    ->default(true)
                    ->required(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->description('All links listed/added here will be displayed as a community link on the card.')
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->whereNull('page_id')->where('view', 'communities')->orderBy('order')
            )
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    ->url(
                        fn($record) =>
                        $record->type === 'email' ? 'mailto:' . $record->link : ($record->type === 'sms' ? 'sms:' . $record->link : ($record->type === 'phone' ? 'tel:' . $record->link : $record->link))
                    )
                    ->openUrlInNewTab(),

                Tables\Columns\TextColumn::make('type')
                    ->searchable()
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->disabled(function (RelationManager $livewire) {
                        return $livewire->ownerRecord->buttons()->where('view', 'communities')->count() >= 5;
                    })
                    ->mutateFormDataUsing(function (array $data) {
                        $data['view'] = 'communities';

                        Button::handleLinkFormatting($data);

                        return $data;
                    })
                    ->using(function (RelationManager $livewire, array $data) {

                        if (
                            $livewire->ownerRecord
                            ->buttons()
                            ->where('view', 'communities')
                            ->count() >= 5
                        ) {

                            Notification::make()
                                ->warning()
                                ->title('User can only add up to 5 community links!')
                                ->persistent()
                                ->send();

                            return null; // ❌ stop creation
                        }

                        $button = $livewire->ownerRecord->buttons()->create($data);

                        Notification::make()
                            ->title('Success')
                            ->body('Community link created successfully.')
                            ->success()
                            ->send();

                        return $button; // ✅ required
                    }),


            ])
            ->actions([
                Tables\Actions\EditAction::make()
                    ->mutateFormDataUsing(function (array $data) {
                        Button::handleLinkFormatting($data);
                        return $data;
                    })

                    ->using(function (Model $record, array $data) {
                        $record->update($data);

                        Notification::make()
                            ->title('Success')
                            ->body('Community link has been updated successfully.')
                            ->success()
                            ->send();

                        return $record;
                    }),

                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
