<?php

namespace App\Filament\Resources\UserResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Tables\Actions\Action;
use Filament\Support\Enums\IconPosition;
use Filament\Tables\Actions\EditAction;
use App\Models\Button;
use Filament\Tables\Actions\CreateAction;
use Filament\Notifications\Notification;
use Illuminate\Database\Eloquent\Model;

class PaymentRelationManager extends RelationManager
{
    protected static string $relationship = 'buttons';

    public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    {
        return in_array($ownerRecord->user_type, ['Basic', 'Gold', 'Platinum', 'Sponsored', 'Influencer']);
    }

    protected static ?string $title = 'Payments';
    protected static ?string $modelLabel = 'Payment';

    // public static function getEloquentQuery(): Builder
    // {
    //     return parent::getEloquentQuery()->where('view', 'payments')->orderBy('order');
    // }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Select::make('type')
                    ->required()
                    ->label('Link Type')
                    ->reactive()
                    ->afterStateUpdated(function ($state, callable $get, callable $set) {
                        $urls = [
                            'cashapp' => 'https://cash.app/$',
                            'venmo' => 'https://venmo.com/u/',
                            'paypal_link' => 'https://www.paypal.me/',
                        ];

                        $titles = [
                            'chime' => 'Chime',
                            'square' => 'Square',
                            'cashapp' => 'Cashapp',
                            'paypal_link' => 'Paypal Link',
                            'paypal_email' => 'Paypal Email',
                            'apple_pay' => 'Apple Pay',
                            'venmo' => 'Venmo',
                            'credit' => 'Credit Card',
                            'zelle' => 'Zelle',
                        ];

                        $set('link', $urls[$state] ?? '');
                        $set('title', $titles[$state] ?? '');
                    })
                    ->options([
                        'payment_other' => 'Payment (Other)',
                        'chime' => 'Chime',
                        'square' => 'Square',
                        'cashapp' => 'Cashapp',
                        'paypal_link' => 'Paypal Link',
                        'paypal_email' => 'Paypal Email',
                        'apple_pay' => 'Apple Pay',
                        'venmo' => 'Venmo',
                        'credit' => 'Credit Card',
                        'zelle' => 'Zelle',
                    ]),
                Forms\Components\TextInput::make('title')
                    ->required()
                    ->maxLength(255),
                Forms\Components\TextInput::make('link')
                    ->columnSpanFull()
                    ->required()
                    ->maxLength(255),
                Forms\Components\Toggle::make('is_active')
                    ->columnSpanFull()
                    ->label('Active')
                    ->default(true)
                    ->required(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->description('All links listed/added here will be displayed as a payment icon on the card.')
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->whereNull('page_id')->where('view', 'payments')->orderBy('order')
            )
            ->columns([
                // USED FOR DEBUGGING
                // Tables\Columns\TextColumn::make('page_id')->label('Page ID'),
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive
                // Tables\Columns\TextColumn::make('title')
                //     ->searchable(),
                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    // ->url(fn($record) => $record->link)
                    ->url(
                        fn($record) =>
                        $record->type === 'email' ? 'mailto:' . $record->link : ($record->type === 'sms' ? 'sms:' . $record->link : ($record->type === 'phone' ? 'tel:' . $record->link : $record->link))
                    )
                    ->openUrlInNewTab(),

                Tables\Columns\TextColumn::make('type')
                    ->searchable()
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->before(function (CreateAction $action, $livewire) {
                        // Access form data from the action
                        $data = $action->getFormData();

                        // Validate and process the link
                        try {
                            // Assuming $data is the input data for button creation
                            Button::handleLinkFormatting($data);

                            // Check if the user is on a free plan and has exceeded the button limit
                            if ($livewire->ownerRecord->user_type == 'Free' && $livewire->ownerRecord->buttons()->count() >= 4) {
                                Notification::make()
                                    ->warning()
                                    ->title('Free card holders can only add up to 4 button links!')
                                    ->persistent()
                                    ->send();

                                // Cancel the action to prevent further processing
                                $action->cancel(); // This ensures that the action is canceled and no further action is executed
                            } else {
                                // Proceed to create the button if no issues
                                $livewire->ownerRecord->buttons()->create($data);

                                Notification::make()
                                    ->title('Success')
                                    ->body('Payment link created successfully.')
                                    ->success()  // Success notification style
                                    ->send();

                                // Cancel the action after manual creation to prevent default action execution
                                $action->cancel();  // Prevent the default action (which would also try to create the button)
                            }
                        } catch (\InvalidArgumentException $e) {
                            Notification::make()
                                ->title('Validation Error')
                                ->body($e->getMessage())  // Show the exception message
                                ->danger()  // Set notification style to error
                                ->send();

                            // Cancel the action to prevent the model creation
                            $action->halt(); // Halt the action to prevent further processing
                        }
                    })
                    ->mutateFormDataUsing(function (array $data) {
                        $data['view'] = 'payments';
                        return $data;
                    }),
            ])
            ->actions([
                Tables\Actions\EditAction::make()->before(function (EditAction $action, $livewire) {
                    $data = $action->getFormData();
                    try {
                        Button::handleLinkFormatting($data);

                        // Fetch the existing button to update
                        $button = $action->getRecord();

                        if ($livewire->ownerRecord->user_type == 'Free' && $livewire->ownerRecord->buttons()->count() >= 4) {
                            Notification::make()
                                ->warning()
                                ->title('Free card holders can only add up to 4 button links!')
                                ->persistent()
                                ->send();

                            $action->cancel();
                        } else {
                            $button->update($data);

                            Notification::make()
                                ->title('Success')
                                ->body('Payment link has been updated successfully.')
                                ->success()
                                ->send();

                            $action->cancel();
                        }
                    } catch (\InvalidArgumentException $e) {
                        Notification::make()
                            ->title('Validation Error')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();

                        $action->halt();
                    }
                }),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
