<?php

namespace App\Filament\Resources\UserResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Model;
use App\Settings\CardDefaultsSetting;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;

class SubcardsRelationManager extends RelationManager
{
    protected static string $relationship = 'subcards';

    public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    {
        return $ownerRecord->user_type == 'Platinum' || $ownerRecord->user_type == 'Gold';
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                // Forms\Components\TextInput::make('title')
                //     ->required()
                //     ->maxLength(255),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('user.firstname')
                    ->label('Owner')
                    ->getStateUsing(function ($record) {
                        return $record->user ? $record->user->firstname . ' ' . $record->user->lastname : 'N/A';
                    })
                    ->searchable(['firstname', 'lastname']),
                Tables\Columns\TextColumn::make('username')
                    ->badge()
                    ->searchable()
                    ->label('Card Link')
                    ->color('primary')
                    ->searchable(query: function ($query, $search) {
                        $query->orWhere('username', 'like', '%' . ltrim($search, '@') . '%');
                    })
                    ->url(function ($record) {
                        $siteUrl = app(CardDefaultsSetting::class)->site_url;
                        return "{$siteUrl}{$record->username}";
                    })
                    ->formatStateUsing(fn($state) => '@' . $state)
                    ->openUrlInNewTab(),

                Tables\Columns\TextColumn::make('email')
                    ->searchable()
                    ->label('Email'),

                Tables\Columns\TextColumn::make('firstname')
                    ->label('Name on Card')
                    ->getStateUsing(function ($record) {
                        return $record ? $record->firstname . ' ' . $record->lastname : 'N/A';
                    })
                    ->searchable(['firstname', 'lastname']),

                Tables\Columns\TextColumn::make('created_at')
                    ->label('Created')
                    ->sortable()
                    //  ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y | D | h:i a')),
                    ->formatStateUsing(fn($state) => Carbon::parse($state)->format('M d, Y')),

            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\Action::make('customAction')
                    ->label('New Subcard') // Button text
                    // ->url(app(CardDefaultsSetting::class)->site_url)
                    ->url(route('filament.admin.resources.sub-cards.create'))
                // ->openUrlInNewTab()

            ])
            ->actions([
                Tables\Actions\Action::make('edit')
                    ->label('Edit')
                    ->url(fn($record) => route('filament.admin.resources.sub-cards.edit', $record))
                    ->openUrlInNewTab(false),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
