<?php

namespace App\Filament\Resources\UserResource\RelationManagers;

use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Forms\Set;
use App\Models\SubCard;
use App\Models\User;
use Filament\Forms\Components\Select;
use Filament\Tables\Actions\Action;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;

class TrustedNetworkRelationManager extends RelationManager
{
    protected static string $relationship = 'trusted_network';

    public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    {
        // return $ownerRecord->user_type == 'Platinum' || $ownerRecord->user_type == 'Gold';
        return $ownerRecord->user_type == 'Platinum';
    }

    // public function mutateFormDataBeforeCreate(array $data): array
    // {
    //     if (isset($data['member_id']) && !isset($data['card_type'])) {
    //         if (isset($data['user_id']) && $data['user_id']) {
    //             $data['card_type'] = 'main';
    //         } elseif (isset($data['page_id']) && $data['page_id']) {
    //             $data['card_type'] = 'sub';
    //         }
    //     }

    //     return $data;
    // }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Hidden::make('card_type'),

                Select::make('member_id')
                    ->label('Member')
                    ->columnSpanFull()
                    ->placeholder('Search for member')
                    ->getSearchResultsUsing(function (string $search): array {
                        $users = User::query()
                            ->where(function ($query) use ($search) {
                                if (strpos($search, '@') === 0 && strlen($search) > 1) {
                                    $query->where('username', 'like', "%" . substr($search, 1) . "%");
                                } else {
                                    $query->where('firstname', 'like', "%{$search}%")
                                        ->orWhere('lastname', 'like', "%{$search}%")
                                        ->orWhere('username', 'like', "%{$search}%");
                                }
                            })
                            ->limit(25)
                            ->get()
                            ->mapWithKeys(fn($user) => [
                                "user-{$user->id}" => "{$user->firstname} {$user->lastname} - @$user->username (User)"
                            ]);

                        $subCards = SubCard::query()
                            ->where(function ($query) use ($search) {
                                $query->where('firstname', 'like', "%{$search}%")
                                    ->orWhere('lastname', 'like', "%{$search}%")
                                    ->orWhere('username', 'like', "%{$search}%");
                            })
                            ->limit(25)
                            ->get()
                            ->mapWithKeys(fn($sub) => [
                                "sub-{$sub->id}" => "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)"
                            ]);

                        return $users->toArray() + $subCards->toArray();
                    })
                    ->getOptionLabelUsing(function ($value) {
                        if (!$value) return '';

                        // Accept either "user-5" or just "5"
                        if (!str_contains($value, '-')) {
                            // Try to infer type from DB
                            $user = User::find($value);
                            if ($user) {
                                return "{$user->firstname} {$user->lastname} - @$user->username (User)";
                            }

                            $sub = SubCard::find($value);
                            if ($sub) {
                                return "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)";
                            }

                            return '';
                        }

                        [$type, $id] = explode('-', $value);

                        if ($type === 'user') {
                            $user = User::find($id);
                            return $user ? "{$user->firstname} {$user->lastname} - @$user->username (User)" : '';
                        }

                        if ($type === 'sub') {
                            $sub = SubCard::find($id);
                            return $sub ? "{$sub->firstname} {$sub->lastname} - @$sub->username (SubCard)" : '';
                        }

                        return '';
                    })
                    // ->formatStateUsing(function ($state, $record) {
                    //     if (!$record) return $state;

                    //     if ($record->card_type === 'main' && $record->user_id) {
                    //         return "user-{$record->user_id}";
                    //     }

                    //     if ($record->card_type === 'sub' && $record->page_id) {
                    //         return "sub-{$record->page_id}";
                    //     }

                    //     return $state;
                    // })

                    ->formatStateUsing(function ($state, $record) {

                        Log::info('🟦 formatStateUsing()', [
                            'record_id' => $record?->id,
                            'card_type' => $record?->card_type,
                            'user_id'   => $record?->user_id,
                            'page_id'   => $record?->page_id,
                            'member_id' => $state,
                        ]);

                        if (!$record) return $state;

                        // if ($record->card_type === 'main' && $record->user_id) {
                        //     return "user-{$record->user_id}";
                        // }

                        // if ($record->card_type === 'sub' && $record->page_id) {
                        //     return "sub-{$record->page_id}";
                        // }

                        if ($record->card_type === 'main') {
                            return "user-{$record->member_id}";
                        }

                        if ($record->card_type === 'sub') {
                            return "sub-{$record->member_id}";
                        }

                        return $state;
                    })

                    // ->afterStateUpdated(function (Set $set, $state) {
                    //     if (!$state) return;

                    //     [$type, $id] = explode('-', $state);

                    //     $set('member_id', (int) $id);

                    //     if ($type === 'user') {
                    //         $set('user_id', $id);
                    //         $set('page_id', null);
                    //         $set('card_type', 'main');
                    //     } elseif ($type === 'sub') {
                    //         $set('page_id', $id);
                    //         $set('user_id', null);
                    //         $set('card_type', 'sub');
                    //     }
                    // })
                    ->afterStateUpdated(function (Set $set, $state) {
                        Log::info('🟩 afterStateUpdated()', [
                            'raw_state' => $state,
                        ]);

                        if (!$state) return;

                        [$type, $id] = explode('-', $state);

                        Log::info('🟩 Parsed selection', [
                            'type' => $type,
                            'id'   => $id,
                        ]);

                        $set('member_id', (int) $id);

                        if ($type === 'user') {
                            $set('user_id', $id);
                            $set('page_id', null);
                            $set('card_type', 'main');
                        } elseif ($type === 'sub') {
                            $set('page_id', $id);
                            $set('user_id', null);
                            $set('card_type', 'sub');
                        }
                    })

                    // ->dehydrateStateUsing(fn($state) => is_string($state) && str_contains($state, '-') ? (int) explode('-', $state)[1] : (int)$state)

                    ->dehydrateStateUsing(function ($state) {

                        Log::info('🟥 dehydrateStateUsing()', [
                            'incoming' => $state,
                        ]);

                        if (is_string($state) && str_contains($state, '-')) {
                            $id = (int) explode('-', $state)[1];

                            Log::info('🟥 dehydrated to ID only', [
                                'dehydrated' => $id,
                            ]);

                            return $id;
                        }

                        Log::info('🟥 dehydrated unchanged', [
                            'dehydrated' => (int)$state,
                        ]);

                        return (int)$state;
                    })


                    ->searchable()
                    ->required(),

                Forms\Components\Select::make('type')
                    ->label('Network Type')
                    ->helperText('If you leave it empty, it will be saved as Network.')
                    ->columnSpanFull()
                    ->options([
                        'network' => 'Network',
                        'eat' => 'Eat',
                        'expert' => 'Expert',
                        'event' => 'Event',
                    ]),

                Forms\Components\Toggle::make('is_accepted')
                    ->columnSpanFull()
                    ->label('Active')
                    ->default(true),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->modifyQueryUsing(
                fn(Builder $query, RelationManager $livewire) =>
                $query->orderBy('order', 'asc')
            )
            ->columns([

                Tables\Columns\ToggleColumn::make('is_accepted')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\ImageColumn::make('image')
                    ->label('Image')
                    ->defaultImageUrl(asset('/img/profile-placeholder.jpg'))
                    ->getStateUsing(function ($record) {
                        if (!$record) return 'N/A';

                        // If card_type is main, fetch from users table
                        if ($record->card_type === 'main' && $record->user) {
                            $user = $record->user;
                            return "$user->profile_pic";
                        }

                        // If card_type is sub, fetch from subcards table
                        if ($record->card_type === 'sub' && $record->subcard) {
                            $sub = $record->subcard;
                            return "$sub->profile_pic";
                        }

                        return 'N/A';
                    }),

                Tables\Columns\TextColumn::make('member_id')
                    ->label('Member')
                    ->getStateUsing(function ($record) {
                        if (!$record) return 'N/A';

                        // If card_type is main, fetch from users table
                        if ($record->card_type === 'main' && $record->user) {
                            $user = $record->user;
                            return "{$user->firstname} {$user->lastname} - @$user->username (Main Card)";
                        }

                        // If card_type is sub, fetch from subcards table
                        if ($record->card_type === 'sub' && $record->subcard) {
                            $sub = $record->subcard;
                            return "{$sub->firstname} {$sub->lastname} - @$sub->username (Sub Card)";
                        }

                        return 'N/A';
                    }),
                Tables\Columns\TextColumn::make('type')
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make(),
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }
}
