<?php

namespace App\Filament\Resources\UserResource\RelationManagers;

use App\Models\Video;
use App\Settings\CardDefaultsSetting;
use Filament\Tables\Actions\Action;
use Filament\Actions\CreateAction;
use Filament\Forms;
use Filament\Forms\Components\Select;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Tables;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Filament\Support\Enums\IconPosition;
use Filament\Forms\Components\FileUpload;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Support\Str;

class VideosRelationManager extends RelationManager
{
    public $videos;
    protected static string $relationship = 'videos';

    public static function canViewForRecord(Model $ownerRecord, string $pageClass): bool
    {
        return in_array($ownerRecord->user_type, ['Influencer', 'Sponsored', 'Gold', 'Platinum', 'Basic']);
        // $ownerRecord->user_type == 'Platinum' || $ownerRecord->user_type == 'Sponsored';
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\TextInput::make('title')
                    ->required()
                    ->maxLength(255),
                Select::make('video_type')
                    ->label('Vide Type')
                    ->required()
                    ->searchable()
                    ->options([
                        'youtube' => 'Youtube',
                        'vimeo' => 'Vimeo',
                    ]),
                Forms\Components\TextInput::make('link')
                    ->label('Video Link')
                    ->url()
                    ->required()
                    ->columnSpanFull()
                    ->maxLength(255),

                FileUpload::make('thumbnail')
                    ->label("Custom Video Thumbnail")
                    ->placeholder('Upload Custom Video Thumbnail')
                    ->directory('video_thumbnails')
                    ->disk('public')
                    ->image()
                    ->imageEditor()
                    ->imageResizeMode('cover')
                    ->columnSpanFull()
                    ->imageCropAspectRatio('1280:720')
                    ->imageResizeTargetWidth('1280')
                    ->imageResizeTargetHeight('720')
                    ->getUploadedFileNameForStorageUsing(
                        fn(TemporaryUploadedFile $file): string =>
                        (string) Str::random(40) . '.' . $file->getClientOriginalExtension()
                    ),

                Forms\Components\TextInput::make('button_title')
                    ->label('Button Title')
                    ->placeholder('Optional')
                    ->maxLength(255),
                Forms\Components\TextInput::make('button_link')
                    ->label('Button Link')
                    ->url()
                    ->placeholder('Optional')
                    ->maxLength(255),
                Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
                    ->columnSpanFull()
                    ->required(),
            ]);
    }

    public function table(Table $table): Table
    {
        return $table
            ->reorderable('order')
            ->reorderRecordsTriggerAction(
                fn(Action $action, bool $isReordering) => $action
                    ->button()
                    ->label($isReordering ? 'Disable reordering' : 'Enable reordering'),
            )
            ->description(function (RelationManager $livewire) {
                $limits = [
                    'Platinum'   => 5,
                    'Gold'       => 3,
                    'Sponsored'  => 1,
                    'Influencer' => 1,
                    'Basic'      => 1,
                ];

                $userType = $livewire->ownerRecord->user_type;
                $settings = app(CardDefaultsSetting::class);

                $terms = [
                    'Platinum'   => $settings->term_platinum,
                    'Gold'       => $settings->term_gold,
                    'Basic'      => $settings->term_basic,
                    'Sponsored'  => 'Sponsored',
                    'Influencer' => 'Influencer',
                ];

                if (! isset($limits[$userType])) {
                    return null;
                }

                $userTypeTerm = $terms[$userType] ?? $userType;

                return "{$userTypeTerm} card holders can add up to {$limits[$userType]} "
                    . ($limits[$userType] > 1 ? 'videos' : 'video');
            })

            // ->modifyQueryUsing(
            //     fn(Builder $query, RelationManager $livewire) =>
            //     $query->whereNull('page_id')
            //         ->where('user_id', $livewire->ownerRecord->id)->orderBy('order')
            // )
            ->modifyQueryUsing(function (Builder $query, RelationManager $livewire) {
                $query->whereNull('page_id')
                    ->where('user_id', $livewire->ownerRecord->id)
                    ->orderBy('order');

                if ($livewire->ownerRecord->user_type === 'Basic') {
                    $query->limit(1);
                }

                return $query;
            })
            ->paginated(fn(RelationManager $livewire) => $livewire->ownerRecord->user_type !== 'Basic')


            ->recordTitleAttribute('user_id')
            ->columns([
                Tables\Columns\ToggleColumn::make('is_active')
                    ->label('Active')
                    ->onColor('success') // ✅ green when active
                    ->offColor('danger'), // ❌ red when inactive

                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->limit(40)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();

                        if (strlen($state) <= $column->getCharacterLimit()) {
                            return null;
                        }

                        // Only render the tooltip if the column content exceeds the length limit.
                        return $state;
                    }),

                Tables\Columns\TextColumn::make('link')
                    ->searchable()
                    ->limit(40)
                    ->icon('heroicon-m-arrow-top-right-on-square')
                    ->iconPosition(IconPosition::After)
                    ->url(fn($record) => $record->link)
                    ->openUrlInNewTab(),
                // ->tooltip(function (TextColumn $column): ?string {
                //     $state = $column->getState();

                //     if (strlen($state) <= $column->getCharacterLimit()) {
                //         return null;
                //     }

                //     // Only render the tooltip if the column content exceeds the length limit.
                //     return $state;
                // }),
                Tables\Columns\TextColumn::make('video_type')
                    ->label('Video Type')
                    ->searchable()
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->filters([
                //
            ])
            ->headerActions([
                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->disabled(function (RelationManager $livewire) {
                        $videoCount = $livewire->ownerRecord
                            ->videos()
                            ->whereNull('page_id')
                            ->count();

                        $userType = $livewire->ownerRecord->user_type;

                        $limits = [
                            'Platinum'   => 5,
                            'Gold'       => 3,
                            'Sponsored'  => 1,
                            'Influencer' => 1,
                            'Basic'      => 1,
                        ];

                        // If user type is not limited, allow action
                        if (! isset($limits[$userType])) {
                            return false;
                        }

                        // Disable when limit is reached
                        return $videoCount >= $limits[$userType];
                    })
                    ->mutateFormDataUsing(function (array $data, RelationManager $livewire) {
                        $data['user_id'] = $livewire->ownerRecord->user_id;
                        return $data;
                    })
                // ->before(function (Tables\Actions\CreateAction $action, RelationManager $livewire) {
                //     $videoCount = $livewire->ownerRecord->videos()->whereNull('page_id')->count();
                //     $userType = $livewire->ownerRecord->user_type;

                //     $limits = [
                //         'Platinum' => 5,
                //         'Sponsored' => 1,
                //         'Influencer' => 1,
                //         'Basic' => 1,
                //     ];

                //     if (isset($limits[$userType]) && $videoCount >= $limits[$userType]) {
                //         $settings = app(CardDefaultsSetting::class);

                //         if ($userType == 'Gold') {
                //             $userTypeTerm = $settings->term_gold;
                //         } else if ($userType == 'Platinum') {
                //             $userTypeTerm = $settings->term_platinum;
                //         }
                //         Notification::make()
                //             ->warning()
                //             ->title("{$userTypeTerm} card holders can only add up to {$limits[$userType]} " . ($userType === 'Platinum' ? 'videos' : 'video') . "!")
                //             ->persistent()
                //             ->send();

                //         $action->halt();
                //     }
                // })
                // ->disabled(function (RelationManager $livewire) {
                //     $limits = [
                //         'Platinum' => 5,
                //         'Sponsored' => 1,
                //         'Influencer' => 1,
                //         'Basic' => 1,
                //     ];

                //     $userType = $livewire->ownerRecord->user_type;
                //     $videoCount = $livewire->ownerRecord->videos()->whereNull('page_id')->count();

                //     return isset($limits[$userType]) && $videoCount >= $limits[$userType];
                // })
            ])
            ->actions([
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make()
                    ->modalHeading('Are you sure you want to delete this video?')
                    ->modalDescription('This action cannot be undone.')
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make()
                        ->modalHeading('Are you sure you want to delete these videos?')
                        ->modalDescription('This action cannot be undone.'),
                ]),
            ]);
    }
}
