<?php

namespace App\Filament\Resources;

use App\Filament\Resources\VideoResource\Pages;
use App\Filament\Resources\VideoResource\RelationManagers;
use App\Models\User;
use App\Models\Video;
use Filament\Forms;
use Filament\Forms\Components\Select;
use Filament\Forms\Form;
use Filament\Forms\Set;
use Filament\Notifications\Notification;
use Filament\Resources\RelationManagers\RelationManager;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\SoftDeletingScope;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\HtmlString;

class VideoResource extends Resource
{
    protected static ?string $model = Video::class;

    protected static ?string $navigationIcon = 'heroicon-o-video-camera';

    protected static ?string $navigationGroup = 'Client Management';

    protected static ?int $navigationSort = 5;

    // hide this resource
    public static function shouldRegisterNavigation(): bool
    {
        return false;
    }

    // public static function form(Form $form): Form
    // {
    //     return $form
    //         ->schema([
    //             Select::make('user_id')
    //                 ->label('Owner')
    //                 ->getSearchResultsUsing(
    //                     fn(string $search): array =>
    //                     User::where(function ($query) use ($search) {
    //                         $query->where('firstname', 'like', "%{$search}%")
    //                             ->orWhere('lastname', 'like', "%{$search}%");
    //                     })
    //                         ->where('user_type', 'Platinum')  // AND condition for user_type
    //                         ->limit(50)
    //                         ->pluck(DB::raw('CONCAT(firstname, " ", lastname)'), 'id')
    //                         ->toArray()
    //                 )
    //                 ->getOptionLabelUsing(
    //                     fn($value) =>
    //                     optional(User::find($value))->firstname . ' ' . optional(User::find($value))->lastname
    //                 )
    //                 ->searchable()
    //                 ->required()
    //                 ->helperText(new HtmlString('Only <strong>Platinum</strong> card holders can add video(s)')),
    //             Forms\Components\TextInput::make('title')
    //                 ->required()
    //                 ->maxLength(255),
    //             Forms\Components\TextInput::make('link')
    //                 ->label('Youtube Link')
    //                 ->required()
    //                 ->columnSpanFull()
    //                 ->maxLength(255),
    //             Forms\Components\Toggle::make('is_active')
    //                 ->label('Active')
    //                 ->default(true)
    //                 ->columnSpanFull()
    //                 ->required(),
    //         ]);
    // }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Select::make('user_id')
                    ->label('Owner')
                    ->getSearchResultsUsing(
                        fn(string $search): array =>
                        User::where(function ($query) use ($search) {
                            $query->where('firstname', 'like', "%{$search}%")
                                ->orWhere('lastname', 'like', "%{$search}%");
                        })
                            ->where('user_type', 'Platinum')  // Only Platinum users
                            ->limit(50)
                            ->pluck(DB::raw('CONCAT(firstname, " ", lastname)'), 'id')
                            ->toArray()
                    )
                    ->getOptionLabelUsing(
                        fn($value) =>
                        optional(User::find($value))->firstname . ' ' . optional(User::find($value))->lastname
                    )
                    ->searchable()
                    ->required()
                    ->helperText(new HtmlString('Only <strong>Platinum</strong> card holders can add video(s)'))
                    ->afterStateUpdated(function (Set $set, $state) {
                        // Check if the selected user has 5 or more videos
                        $user = User::find($state);
                        if ($user->user_type == "Platinum" && $user->videos()->count() >= 5) {
                            // Trigger a notification
                            Notification::make()
                                ->title('Limit reached')
                                ->body('Platinum card holders can only add up to 5 videos!')
                                ->warning()
                                ->send();
                            // Optionally, you can reset the state or show a validation error:
                            $set('user_id', null); // Reset the user_id if limit is reached
                        }
                    }),
                Forms\Components\TextInput::make('title')
                    ->required()
                    ->maxLength(255),
                Select::make('video_type')
                    ->label('Vide Type')
                    ->required()
                    ->searchable()
                    ->options([
                        'youtube' => 'Youtube',
                        'vimeo' => 'Vimeo',
                    ]),
                Forms\Components\TextInput::make('link')
                    ->label('Video Link')
                    ->url()
                    ->required()
                    ->maxLength(255),
                Forms\Components\Toggle::make('is_active')
                    ->label('Active')
                    ->default(true)
                    ->columnSpanFull()
                    ->required(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\IconColumn::make('is_active')
                    ->label('Active')
                    ->boolean(),
                Tables\Columns\TextColumn::make('user.firstname')
                    ->label('Owner')
                    ->getStateUsing(function ($record) {
                        return $record->user ? $record->user->firstname . ' ' . $record->user->lastname : 'N/A';
                    })
                    ->sortable()
                    ->searchable(['firstname', 'lastname']),
                Tables\Columns\TextColumn::make('title')
                    ->searchable()
                    ->limit(40)
                    ->tooltip(function (TextColumn $column): ?string {
                        $state = $column->getState();

                        if (strlen($state) <= $column->getCharacterLimit()) {
                            return null;
                        }

                        // Only render the tooltip if the column content exceeds the length limit.
                        return $state;
                    }),
                Tables\Columns\TextColumn::make('video_type')
                    ->searchable()
                    ->formatStateUsing(fn($state) => ucfirst($state)),
            ])
            ->headerActions([
                // Tables\Actions\CreateAction::make()
                //     ->createAnother(false)
                //     ->before(function (Tables\Actions\CreateAction $action, RelationManager $livewire) {
                //         // dd($livewire->ownerRecord->videos()->count());
                //         if ($livewire->ownerRecord->videos()->count() >= 5) {
                //             Notification::make()
                //                 ->warning()
                //                 ->title('Platinum card holders can only add up to 5 videos!')
                //                 // ->body('Choose a plan to continue.')
                //                 ->persistent()
                //                 ->send();
                //             // $action->halt();
                //             $action->cancel();
                //         }
                //     })

                Tables\Actions\CreateAction::make()
                    ->createAnother(false)
                    ->before(function (Tables\Actions\CreateAction $action, RelationManager $livewire) {
                        if ($livewire->ownerRecord->videos()->count() >= 5) {
                            Notification::make()
                                ->warning()
                                ->title('Platinum card holders can only add up to 5 videos!')
                                ->persistent()
                                ->send();

                            $action->halt(); // Prevent proceeding to the creation form
                        }
                    })

            ])
            ->actions([
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                // Tables\Actions\DeleteAction::make()
                //     ->modalHeading('Are you sure you want to delete this video?')
                //     ->modalDescription('This action cannot be undone.')
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                ]),
            ]);
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListVideos::route('/'),
            'create' => Pages\CreateVideo::route('/create'),
            'view' => Pages\ViewVideo::route('/{record}'),
            'edit' => Pages\EditVideo::route('/{record}/edit'),
        ];
    }
}
