<?php

namespace App\Helpers;

use App\Settings\CardDefaultsSetting;

class SavingLinkHelpers
{
    /**
     *
     * @param string $input
     * @return string
     */

    public static function saveEventVideoLink(array|string $input): array
    {
        $result = [];

        // ---- If input is a repeater array: [['link' => '...'], ...] ----
        if (is_array($input) && isset($input[0]) && is_array($input[0]) && array_key_exists('link', $input[0])) {
            foreach ($input as $item) {
                $link = $item['link'] ?? '';
                if (is_string($link) && !empty(trim($link))) {
                    $result[] = ['link' => self::validateSingleLink($link)];
                }
            }
            return $result;
        }

        // ---- If input is array of strings: ['https://youtu.be/...', ...] ----
        if (is_array($input)) {
            foreach ($input as $link) {
                if (is_string($link) && !empty(trim($link))) {
                    $result[] = ['link' => self::validateSingleLink($link)];
                }
            }
            return $result;
        }

        // ---- If input is single string ----
        if (is_string($input) && !empty(trim($input))) {
            $result[] = ['link' => self::validateSingleLink($input)];
            return $result;
        }

        throw new \InvalidArgumentException('No valid video links provided.');
    }

    // protected static function validateSingleLink(string $input): string
    // {
    //     $input = trim($input);

    //     // Ensure protocol
    //     if (!preg_match('/^https?:\/\//i', $input)) {
    //         $input = 'https://' . ltrim($input, '/');
    //     }

    //     if (!filter_var($input, FILTER_VALIDATE_URL)) {
    //         throw new \InvalidArgumentException("Invalid URL format.");
    //     }

    //     $host  = parse_url($input, PHP_URL_HOST);
    //     $path  = trim(parse_url($input, PHP_URL_PATH) ?? '', '/');
    //     $query = parse_url($input, PHP_URL_QUERY);

    //     // ---- YouTube ----
    //     if (strpos($host, 'youtube.com') !== false || strpos($host, 'youtu.be') !== false) {
    //         $videoId = null;

    //         if (strpos($host, 'youtu.be') !== false) {
    //             $videoId = $path;
    //             return 'https://www.youtube.com/watch?v=' . $videoId;
    //         }

    //         if ($query) {
    //             parse_str($query, $queryParams);
    //             if (!empty($queryParams['v'])) {
    //                 return 'https://www.youtube.com/watch?v=' . $queryParams['v'];
    //             }
    //         }

    //         if (strpos($path, 'embed/') === 0) {
    //             $videoId = substr($path, strlen('embed/'));
    //             return 'https://www.youtube.com/watch?v=' . $videoId;
    //         }

    //         if (strpos($path, 'shorts/') === 0) {
    //             $videoId = substr($path, strlen('shorts/'));
    //             return 'https://www.youtube.com/shorts/' . $videoId;
    //         }

    //         throw new \InvalidArgumentException("Invalid YouTube video URL.");
    //     }

    //     // ---- Vimeo ----
    //     if (strpos($host, 'vimeo.com') !== false) {
    //         $videoId = null;
    //         $hash    = null;

    //         if (preg_match('#^(\d+)(?:/([a-zA-Z0-9]+))?#', $path, $matches)) {
    //             $videoId = $matches[1];
    //             $hash    = $matches[2] ?? null;
    //         } elseif (preg_match('#^video/(\d+)#', $path, $matches)) {
    //             $videoId = $matches[1];
    //             if ($query) {
    //                 parse_str($query, $queryParams);
    //                 if (!empty($queryParams['h'])) {
    //                     $hash = $queryParams['h'];
    //                 }
    //             }
    //         }

    //         if (!$videoId) {
    //             throw new \InvalidArgumentException("Invalid Vimeo video URL.");
    //         }

    //         return 'https://player.vimeo.com/video/' . $videoId . ($hash ? '?h=' . $hash : '');
    //     }

    //     throw new \InvalidArgumentException("The input must be a valid YouTube or Vimeo video link.");
    // }

    protected static function validateSingleLink(string $input): string
    {
        $input = trim($input);

        // Ensure protocol
        if (!preg_match('/^https?:\/\//i', $input)) {
            $input = 'https://' . ltrim($input, '/');
        }

        if (!filter_var($input, FILTER_VALIDATE_URL)) {
            throw new \InvalidArgumentException("Invalid URL format.");
        }

        $host  = strtolower(parse_url($input, PHP_URL_HOST));
        $path  = trim(parse_url($input, PHP_URL_PATH) ?? '', '/');
        $query = parse_url($input, PHP_URL_QUERY);

        // ---------------------------------------------------------
        //                       YOUTUBE
        // ---------------------------------------------------------
        if (str_contains($host, 'youtube.com') || str_contains($host, 'youtu.be')) {
            // youtu.be/VIDEOID
            if (str_contains($host, 'youtu.be')) {
                return "https://www.youtube.com/watch?v={$path}";
            }

            // youtube.com/watch?v=xxxx
            parse_str($query ?? '', $params);
            if (!empty($params['v'])) {
                return "https://www.youtube.com/watch?v={$params['v']}";
            }

            // youtube.com/embed/VIDEOID
            if (str_starts_with($path, 'embed/')) {
                return "https://www.youtube.com/watch?v=" . substr($path, 6);
            }

            // youtube.com/shorts/VIDEOID
            if (str_starts_with($path, 'shorts/')) {
                return "https://www.youtube.com/shorts/" . substr($path, 7);
            }

            // Accept ANY YouTube URL
            return $input;
        }

        // ---------------------------------------------------------
        //                       VIMEO
        // ---------------------------------------------------------
        if (str_contains($host, 'vimeo.com')) {
            $videoId = null;
            $hash    = null;

            // /123456/abcdef
            if (preg_match('#^(\d+)(?:/([A-Za-z0-9]+))?#', $path, $m)) {
                $videoId = $m[1];
                $hash    = $m[2] ?? null;
            }

            // /video/123456?h=xxxx
            if (!$videoId && preg_match('#video/(\d+)#', $path, $m)) {
                $videoId = $m[1];
                parse_str($query ?? '', $params);
                if (!empty($params['h'])) {
                    $hash = $params['h'];
                }
            }

            // If Vimeo ID found → normalize player URL
            if ($videoId) {
                return "https://player.vimeo.com/video/{$videoId}" . ($hash ? "?h=$hash" : "");
            }

            // Accept any remaining Vimeo URLs
            return $input;
        }

        // ---------------------------------------------------------
        //                       TIKTOK
        // ---------------------------------------------------------
        if (
            str_contains($host, 'tiktok.com') ||
            str_contains($host, 'vm.tiktok.com') ||
            str_contains($host, 'vt.tiktok.com')
        ) {
            // Pass through untouched — TikTok URLs vary a lot
            return $input;
        }

        // ---------------------------------------------------------
        //                      INSTAGRAM
        // ---------------------------------------------------------
        if (
            str_contains($host, 'instagram.com') ||
            str_contains($host, 'instagr.am')
        ) {
            // Reels, posts, IGTV, short links — accept all
            return $input;
        }

        throw new \InvalidArgumentException("The input must be a valid video link (YouTube, Vimeo, TikTok, or Instagram).");
    }



    public static function saveWhatsApp(string $input): string
    {
        $input = trim($input);

        // Accept WhatsApp group links
        if (
            filter_var($input, FILTER_VALIDATE_URL) &&
            preg_match('#^https://chat\.whatsapp\.com/[A-Za-z0-9]{20,}$#', $input)
        ) {
            return $input;
        }

        // Accept wa.me links with phone numbers
        if (
            filter_var($input, FILTER_VALIDATE_URL) &&
            str_starts_with($input, 'https://wa.me/')
        ) {
            $phoneNumber = str_replace('https://wa.me/', '', $input);
            $phoneNumber = preg_replace('/[^\d+]/', '', $phoneNumber);

            if (preg_match('/^\+?\d{10,15}$/', $phoneNumber)) {
                return 'https://wa.me/' . ltrim($phoneNumber, '+'); // wa.me does not require '+'
            }

            throw new \InvalidArgumentException("The WhatsApp link must contain a valid phone number.");
        }

        // Accept plain phone number input
        if (preg_match('/^\+?\d{10,15}$/', preg_replace('/[^\d+]/', '', $input))) {
            $phoneNumber = preg_replace('/[^\d+]/', '', $input);
            return 'https://wa.me/' . ltrim($phoneNumber, '+');
        }

        throw new \InvalidArgumentException("The input must be a valid WhatsApp number or group link.");
    }

    public static function saveWeChat(string $input): string
    {
        $input = trim($input);

        // Accept WeChat official links (e.g. https://weixin.qq.com/r/XXXX)
        if (
            filter_var($input, FILTER_VALIDATE_URL) &&
            preg_match('#^https://weixin\.qq\.com/[A-Za-z0-9/_\-]+$#i', $input)
        ) {
            return $input;
        }

        // Accept WeChat deep links (e.g. weixin://dl/scan or weixin://dl/chat?username=...)
        if (
            str_starts_with($input, 'weixin://') &&
            preg_match('#^weixin://[A-Za-z0-9_\-/?=.&]+$#', $input)
        ) {
            return $input;
        }

        // Accept plain WeChat ID (letters, numbers, underscores, dashes, 6–20 chars)
        if (preg_match('/^[a-zA-Z][a-zA-Z0-9_-]{5,19}$/', $input)) {
            // Normalize and return custom URL scheme
            return 'weixin://dl/chat?username=' . $input;
        }

        throw new \InvalidArgumentException("The input must be a valid WeChat ID or link.");
    }

    public static function saveZalo(string $input): string
    {
        $input = trim($input);

        // Accept official Zalo profile links (e.g. https://zalo.me/1234567890)
        if (
            filter_var($input, FILTER_VALIDATE_URL) &&
            preg_match('#^https://zalo\.me/[0-9]{6,20}$#', $input)
        ) {
            return $input;
        }

        // Accept Zalo deep links (e.g. zalo://conversation?uid=1234567890)
        if (
            str_starts_with($input, 'zalo://') &&
            preg_match('#^zalo://[A-Za-z0-9_\-/?=.&]+$#', $input)
        ) {
            return $input;
        }

        // Accept plain phone number input
        $cleaned = preg_replace('/[^\d+]/', '', $input);
        if (preg_match('/^\+?\d{8,15}$/', $cleaned)) {
            // Remove leading '+' for Zalo link format
            return 'https://zalo.me/' . ltrim($cleaned, '+');
        }

        throw new \InvalidArgumentException("The input must be a valid Zalo number or link.");
    }

    // Save Telegram Link, username or number and make it secured
    public static function saveTelegram(string $input): string
    {
        // Check if the input is a valid Telegram link (https://t.me/)
        if (filter_var($input, FILTER_VALIDATE_URL) && strpos($input, 'https://t.me/') === 0) {
            // Extract the username or phone number from the URL (strip the 'https://t.me/' part)
            $usernameOrPhone = str_replace('https://t.me/', '', $input);

            // Check if the username or phone number is valid (Telegram usernames can contain letters, numbers, and underscores)
            if (preg_match('/^[A-Za-z0-9_]+$/', $usernameOrPhone) || preg_match('/^\+?[0-9]{10,15}$/', $usernameOrPhone)) {
                // Return the valid Telegram link (ensure it's in the correct format)
                return 'https://t.me/' . $usernameOrPhone;
            } else {
                throw new \InvalidArgumentException("The Telegram link must contain a valid username or phone number.");
            }
        }

        // If the input is just a phone number, create a Telegram link
        if (preg_match('/^\+?[0-9]{10,15}$/', $input)) {
            // Format it to a valid Telegram link
            return 'https://t.me/' . $input;
        }

        // If the input is just a valid username, create a Telegram link
        if (preg_match('/^[A-Za-z0-9_]+$/', $input)) {
            // Format it to a valid Telegram link
            return 'https://t.me/' . $input;
        }

        // If the input is not valid, throw an exception or return an error message
        throw new \InvalidArgumentException("The input must be a valid phone number or Telegram link.");
    }

    // public static function saveSecureLink(string $input): string
    // {
    //     // Trim the input to remove leading or trailing spaces
    //     $input = trim($input);

    //     // If the input does not have a scheme (http:// or https://)
    //     if (strpos($input, '://') === false) {
    //         // Prepend https:// to the input to make it a valid URL
    //         $input = 'https://' . $input;
    //     }

    //     // Validate the URL using filter_var
    //     if (!filter_var($input, FILTER_VALIDATE_URL)) {
    //         throw new \InvalidArgumentException("The input must be a valid URL.");
    //     }

    //     // Parse the URL to extract the host/domain
    //     $url = parse_url($input);

    //     // Ensure the domain part exists
    //     if (isset($url['host'])) {
    //         $domain = $url['host'];

    //         // Check if the domain is in a valid format (e.g., site.com, site.org)
    //         if (!preg_match('/^(?!-)(?:[A-Za-z0-9-]{1,63}\.?)+[A-Za-z]{2,6}$/', $domain)) {
    //             throw new \InvalidArgumentException("The input must be a valid domain (e.g., site.com, site.org).");
    //         }

    //         // Check if the domain is resolvable via DNS
    //         // if (!checkdnsrr($domain, 'A') && !checkdnsrr($domain, 'AAAA')) {
    //         //     throw new \InvalidArgumentException("The input must be a valid and existing domain.");
    //         // }
    //     } else {
    //         throw new \InvalidArgumentException("The input must contain a valid domain.");
    //     }

    //     // If the URL is not HTTPS, convert it to HTTPS
    //     if (isset($url['scheme']) && $url['scheme'] !== 'https') {
    //         // Ensure we replace the scheme with HTTPS
    //         $input = 'https://' . ltrim($input, 'http://');
    //     }

    //     // Return the secure URL
    //     return $input;
    // }

    public static function saveSecureLink(string $input): string
    {
        // Normalize input
        $input = trim($input);

        // Add https:// if no scheme
        if (!preg_match('#^https?://#i', $input)) {
            $input = 'https://' . $input;
        }

        // Validate full URL
        if (!filter_var($input, FILTER_VALIDATE_URL)) {
            throw new \InvalidArgumentException("Please enter a valid URL.");
        }

        // Parse URL parts
        $url = parse_url($input);

        // Domain must exist
        if (empty($url['host'])) {
            throw new \InvalidArgumentException("URL must contain a valid domain.");
        }

        $domain = $url['host'];

        // Validate domain format (no localhost, no IP, no weird TLDs)
        if (!preg_match('/^(?!\-)([A-Za-z0-9\-]{1,63}\.)+[A-Za-z]{2,}$/', $domain)) {
            throw new \InvalidArgumentException("Please enter a valid domain like example.com.");
        }

        // Force HTTPS
        $secureUrl = 'https://' . $domain;

        // Rebuild full URL path/query/fragment
        if (isset($url['path'])) {
            $secureUrl .= $url['path'];
        }
        if (isset($url['query'])) {
            $secureUrl .= '?' . $url['query'];
        }
        if (isset($url['fragment'])) {
            $secureUrl .= '#' . $url['fragment'];
        }

        return $secureUrl;
    }


    // Helper function to validate the domain part
    private static function isValidDomain(string $domain): bool
    {
        // Check if the domain has a valid format (letters, numbers, hyphens, and dots)
        return (bool) preg_match('/^(?!-)(?:[A-Za-z0-9-]{1,63}\.?)+[A-Za-z]{2,6}$/', $domain);
    }

    public static function saveCashApp(string $input): string
    {
        $input = trim($input);

        // If input is a full URL
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            // Normalize host
            $host = str_replace('www.', '', $host);

            if ($host !== 'cash.app') {
                throw new \InvalidArgumentException("Only Cash App URLs are allowed.");
            }

            $path = trim(parse_url($input, PHP_URL_PATH), '/');

            if (empty($path)) {
                throw new \InvalidArgumentException("Invalid Cash App link.");
            }

            // Ensure the path starts with "$"
            if (strpos($path, '$') !== 0) {
                $path = '$' . $path;
            }

            return 'https://cash.app/' . $path;
        }

        // If input starts with "cash.app/" or "www.cash.app/"
        if (strpos($input, 'cash.app/') === 0 || strpos($input, 'www.cash.app/') === 0) {
            $path = trim(strstr($input, '/'), '/');

            if (strpos($path, '$') !== 0) {
                $path = '$' . $path;
            }

            return 'https://cash.app/' . $path;
        }

        // If input is just the username
        if (!empty($input)) {
            if (strpos($input, '$') !== 0) {
                $input = '$' . $input;
            }
            return 'https://cash.app/' . $input;
        }

        throw new \InvalidArgumentException("The input must be a valid Cash App link or username.");
    }

    public static function saveZelle(string $input): string
    {
        // $phoneRegex = '/^(\+?[0-9]{10,15}|\(\d{3}\) \d{3}-\d{4})$/';
        // if (preg_match($phoneRegex, $input)) {
        //     // Return the phone number (Zelle works with phone numbers)
        //     return $input;
        // }

        // Normalize the input to remove spaces, hyphens, and parentheses for validation
        $normalized = preg_replace('/[\s\-\(\)]/', '', $input);

        // Match phone numbers with optional + and 10 to 15 digits
        $phoneRegex = '/^\+?[0-9]{10,15}$/';

        if (preg_match($phoneRegex, $normalized)) {
            // Return the original input if it's a valid formatted number
            return $input;
        }

        // Check if the input is a valid email address
        if (filter_var($input, FILTER_VALIDATE_EMAIL)) {
            // Return the email address (Zelle can also use email addresses)
            return $input;
        }

        // If the input is neither a valid phone number nor a valid email, throw an exception
        throw new \InvalidArgumentException("The input must be a valid phone number or email address.");
    }

    public static function savePaypalEmail(string $input): string
    {
        // Check if the input is a valid email address
        if (filter_var($input, FILTER_VALIDATE_EMAIL)) {
            // Return the email address if valid
            return $input;
        }

        // If the input is not a valid email, throw an exception
        throw new \InvalidArgumentException("The input must be a valid email address.");
    }

    public static function saveContactNumber(string $input): string
    {
        // Trim the input and remove all characters except digits and +
        $input = trim($input);

        // Preserve + if it's at the start; remove all other non-digit characters
        if (str_starts_with($input, '+')) {
            $cleanInput = '+' . preg_replace('/[^\d]/', '', substr($input, 1));
        } else {
            $cleanInput = preg_replace('/[^\d]/', '', $input);
        }

        // Validate the cleaned input: must start with + or digit and be 10-15 digits total
        if (preg_match('/^\+?\d{10,15}$/', $cleanInput)) {
            return $cleanInput;
        }

        throw new \InvalidArgumentException("The input must be a valid phone number.");
    }

    public static function saveApplePay(string $input): string
    {
        // Normalize the input to remove spaces, hyphens, and parentheses for validation
        $normalized = preg_replace('/[\s\-\(\)]/', '', $input);

        // Match phone numbers with optional + and 10 to 15 digits
        $phoneRegex = '/^\+?[0-9]{10,15}$/';

        if (preg_match($phoneRegex, $normalized)) {
            // Return the original input if it's a valid formatted number
            return $input;
        }

        // Check if the input is a valid email address
        if (filter_var($input, FILTER_VALIDATE_EMAIL)) {
            return $input;
        }

        throw new \InvalidArgumentException("The input must be a valid phone number or email address.");
    }

    // public static function saveVenmo(string $input): string
    // {
    //     $input = trim($input);

    //     // If input is a valid URL
    //     if (filter_var($input, FILTER_VALIDATE_URL)) {
    //         $host = parse_url($input, PHP_URL_HOST);
    //         $path = trim(parse_url($input, PHP_URL_PATH) ?? '', '/');

    //         $host = str_replace('www.', '', strtolower($host));

    //         if ($host !== 'venmo.com' || empty($path)) {
    //             throw new \InvalidArgumentException("Only valid Venmo profile URLs are allowed.");
    //         }

    //         return 'https://venmo.com/' . $path;
    //     }

    //     // If input starts with "venmo.com/" or "www.venmo.com/"
    //     if (str_starts_with($input, 'venmo.com/') || str_starts_with($input, 'www.venmo.com/')) {
    //         $path = trim(str_replace(['venmo.com/', 'www.venmo.com/'], '', $input), '/');
    //         if (empty($path)) {
    //             throw new \InvalidArgumentException("Missing Venmo username.");
    //         }
    //         return 'https://venmo.com/' . $path;
    //     }

    //     // If input is a venmo://user/ deep link
    //     if (str_starts_with($input, 'venmo://user/')) {
    //         $username = substr($input, strlen('venmo://user/'));
    //         if (empty($username)) {
    //             throw new \InvalidArgumentException("Invalid Venmo username in app link.");
    //         }
    //         return 'https://venmo.com/' . $username;
    //     }

    //     // If input contains forbidden domains
    //     if (str_contains($input, '.com') || str_contains($input, '/')) {
    //         throw new \InvalidArgumentException("Only Venmo usernames are allowed, not other domains.");
    //     }

    //     // Otherwise, treat as plain username
    //     if (!empty($input)) {
    //         return 'https://venmo.com/' . ltrim($input, '@');
    //     }

    //     throw new \InvalidArgumentException("The input must be a valid Venmo link or username.");
    // }

    // public static function saveVenmo(string $input): string
    // {
    //     $input = trim($input);

    //     // If input is a valid URL
    //     if (filter_var($input, FILTER_VALIDATE_URL)) {
    //         $host = parse_url($input, PHP_URL_HOST);
    //         $path = trim(parse_url($input, PHP_URL_PATH) ?? '', '/');
    //         $query = parse_url($input, PHP_URL_QUERY);

    //         $host = str_replace('www.', '', strtolower($host));

    //         if ($host !== 'venmo.com') {
    //             throw new \InvalidArgumentException("Only valid Venmo profile URLs are allowed.");
    //         }

    //         // Case 1: Direct profile (https://venmo.com/username)
    //         if (!empty($path) && $path !== 'code') {
    //             return 'https://venmo.com/' . $path;
    //         }

    //         // Case 2: QR/Code links (https://venmo.com/code?... with user_id)
    //         if ($path === 'code' && !empty($query) && str_contains($query, 'user_id=')) {
    //             return 'https://venmo.com/code?' . $query;
    //         }

    //         throw new \InvalidArgumentException("Invalid Venmo profile URL format.");
    //     }

    //     // If input starts with "venmo.com/" or "www.venmo.com/"
    //     if (str_starts_with($input, 'venmo.com/') || str_starts_with($input, 'www.venmo.com/')) {
    //         $path = trim(str_replace(['venmo.com/', 'www.venmo.com/'], '', $input), '/');
    //         if (empty($path)) {
    //             throw new \InvalidArgumentException("Missing Venmo username.");
    //         }
    //         return 'https://venmo.com/' . $path;
    //     }

    //     // If input is a venmo://user/ deep link
    //     if (str_starts_with($input, 'venmo://user/')) {
    //         $username = substr($input, strlen('venmo://user/'));
    //         if (empty($username)) {
    //             throw new \InvalidArgumentException("Invalid Venmo username in app link.");
    //         }
    //         return 'https://venmo.com/' . $username;
    //     }

    //     // If input contains forbidden domains
    //     if (str_contains($input, '.com') || str_contains($input, '/')) {
    //         throw new \InvalidArgumentException("Only Venmo usernames are allowed, not other domains.");
    //     }

    //     // Otherwise, treat as plain username
    //     if (!empty($input)) {
    //         return 'https://venmo.com/' . ltrim($input, '@');
    //     }

    //     throw new \InvalidArgumentException("The input must be a valid Venmo link or username.");
    // }

    public static function saveVenmo(string $input): string
    {
        $input = trim($input);

        // If it's a full URL
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);
            $path = trim(parse_url($input, PHP_URL_PATH) ?? '', '/');
            $query = parse_url($input, PHP_URL_QUERY);

            $host = strtolower($host);

            // ✅ Accept any link with "venmo" in the hostname
            if (!str_contains($host, 'venmo')) {
                throw new \InvalidArgumentException("Only Venmo URLs are allowed.");
            }

            // Normalize: remove leading "www."
            $host = str_replace('www.', '', $host);

            // Case: account.venmo.com/u/{username}
            if ($host === 'account.venmo.com' && str_starts_with($path, 'u/')) {
                return 'https://venmo.com/' . substr($path, 2);
            }

            // Case: venmo.com/{username}
            if ($host === 'venmo.com' && !empty($path) && $path !== 'code') {
                return 'https://venmo.com/' . $path;
            }

            // Case: venmo.com/code?... (QR link)
            if ($host === 'venmo.com' && $path === 'code' && !empty($query) && str_contains($query, 'user_id=')) {
                return 'https://venmo.com/code?' . $query;
            }

            // Otherwise, just return the original valid Venmo link
            return $input;
        }

        // If input starts with venmo.com/... without scheme
        if (str_starts_with($input, 'venmo.com/') || str_starts_with($input, 'www.venmo.com/')) {
            $path = trim(str_replace(['venmo.com/', 'www.venmo.com/'], '', $input), '/');
            if (empty($path)) {
                throw new \InvalidArgumentException("Missing Venmo username.");
            }
            return 'https://venmo.com/' . $path;
        }

        // Deep link format: venmo://user/username
        if (str_starts_with($input, 'venmo://user/')) {
            $username = substr($input, strlen('venmo://user/'));
            if (empty($username)) {
                throw new \InvalidArgumentException("Invalid Venmo username in app link.");
            }
            return 'https://venmo.com/' . $username;
        }

        // Reject if contains other domains or slashes
        if (str_contains($input, '.com') || str_contains($input, '/')) {
            throw new \InvalidArgumentException("Only Venmo usernames are allowed, not other domains.");
        }

        // Otherwise treat as plain username
        if (!empty($input)) {
            return 'https://venmo.com/' . ltrim($input, '@');
        }

        throw new \InvalidArgumentException("The input must be a valid Venmo link or username.");
    }

    // public static function savePaypalLink(string $input): string
    // {
    //     $input = trim($input);

    //     // If it's a full URL
    //     if (filter_var($input, FILTER_VALIDATE_URL)) {
    //         $host = parse_url($input, PHP_URL_HOST);

    //         // Normalize host to remove "www."
    //         $host = str_replace('www.', '', $host);

    //         if ($host !== 'paypal.me' && $host !== 'paypal.com') {
    //             throw new \InvalidArgumentException("Only PayPal URLs are allowed.");
    //         }

    //         // Normalize paths if necessary
    //         $path = trim(parse_url($input, PHP_URL_PATH), '/');

    //         if (empty($path)) {
    //             throw new \InvalidArgumentException("Invalid PayPal link.");
    //         }

    //         // If URL is already in the paypal.me format
    //         if ($host === 'paypal.me') {
    //             return 'https://www.paypal.me/' . $path;
    //         }

    //         // If URL is in paypal.com/paypalme format
    //         if ($host === 'paypal.com' && strpos($path, 'paypalme/') === 0) {
    //             $username = substr($path, strlen('paypalme/'));
    //             return 'https://www.paypal.me/' . $username;
    //         }

    //         throw new \InvalidArgumentException("Invalid PayPal link format.");
    //     }

    //     // If input is partial like "paypal.me/username" or "www.paypal.me/username"
    //     if (strpos($input, 'paypal.me/') === 0 || strpos($input, 'www.paypal.me/') === 0) {
    //         $username = substr(strstr($input, '/'), 1);
    //         return 'https://www.paypal.me/' . $username;
    //     }

    //     if (strpos($input, 'paypal.com/paypalme/') === 0 || strpos($input, 'www.paypal.com/paypalme/') === 0) {
    //         $username = substr(strstr($input, 'paypalme/'), strlen('paypalme/'));
    //         return 'https://www.paypal.me/' . $username;
    //     }

    //     // If it's just a username
    //     if (!empty($input) && strpos($input, '/') === false && strpos($input, ' ') === false) {
    //         return 'https://www.paypal.me/' . $input;
    //     }

    //     throw new \InvalidArgumentException("The input must be a valid PayPal link or username.");
    // }

    public static function savePaypalLink(string $input): string
    {
        $input = trim($input);

        // If it's a full URL
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            // Normalize host to remove "www."
            $host = str_replace('www.', '', $host);

            if (!in_array($host, ['paypal.me', 'paypal.com'])) {
                throw new \InvalidArgumentException("Only PayPal URLs are allowed.");
            }

            $path = trim(parse_url($input, PHP_URL_PATH), '/');

            if (empty($path)) {
                throw new \InvalidArgumentException("Invalid PayPal link.");
            }

            // paypal.me direct
            if ($host === 'paypal.me') {
                return 'https://www.paypal.me/' . $path;
            }

            // paypal.com/paypalme/{username}
            if ($host === 'paypal.com' && strpos($path, 'paypalme/') === 0) {
                $username = substr($path, strlen('paypalme/'));
                return 'https://www.paypal.me/' . $username;
            }

            // NEW: paypal.com/biz/profile/{username}
            if ($host === 'paypal.com' && strpos($path, 'biz/profile/') === 0) {
                $username = substr($path, strlen('biz/profile/'));
                return 'https://www.paypal.me/' . $username;
            }

            throw new \InvalidArgumentException("Invalid PayPal link format.");
        }

        // Partial formats
        if (strpos($input, 'paypal.me/') === 0 || strpos($input, 'www.paypal.me/') === 0) {
            $username = substr(strstr($input, '/'), 1);
            return 'https://www.paypal.me/' . $username;
        }

        if (strpos($input, 'paypal.com/paypalme/') === 0 || strpos($input, 'www.paypal.com/paypalme/') === 0) {
            $username = substr(strstr($input, 'paypalme/'), strlen('paypalme/'));
            return 'https://www.paypal.me/' . $username;
        }

        // NEW: partial biz/profile format
        if (strpos($input, 'paypal.com/biz/profile/') === 0 || strpos($input, 'www.paypal.com/biz/profile/') === 0) {
            $username = substr(strstr($input, 'biz/profile/'), strlen('biz/profile/'));
            return 'https://www.paypal.me/' . $username;
        }

        // If it's just a username
        if (!empty($input) && strpos($input, '/') === false && strpos($input, ' ') === false) {
            return 'https://www.paypal.me/' . $input;
        }

        throw new \InvalidArgumentException("The input must be a valid PayPal link or username.");
    }

    public static function saveClubhouseLink(string $input): string
    {
        // 🚫 Reject all non-Clubhouse URLs early
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);
            if (strpos($host, 'clubhouse.com') === false) {
                throw new \InvalidArgumentException("Only Clubhouse URLs are allowed.");
            }
        }

        // If the input starts with "https://www.clubhouse.com/", return it as is
        if (filter_var($input, FILTER_VALIDATE_URL) && strpos($input, 'https://www.clubhouse.com/') === 0) {
            return $input;
        }

        // If the input starts with "http://www.clubhouse.com/", replace it with "https://"
        if (filter_var($input, FILTER_VALIDATE_URL) && strpos($input, 'http://www.clubhouse.com/') === 0) {
            return str_replace('http://', 'https://', $input);
        }

        // If the input starts with "www.clubhouse.com/", prepend "https://www."
        if (strpos($input, 'www.clubhouse.com/') === 0) {
            return 'https://' . $input;
        }

        // If the input starts with "clubhouse.com/", prepend "https://www.clubhouse.com/"
        if (strpos($input, 'clubhouse.com/') === 0) {
            return 'https://www.' . $input;
        }

        // If the input starts with "https://clubhouse.com/", prepend "https://www.clubhouse.com/"
        if (filter_var($input, FILTER_VALIDATE_URL) && strpos($input, 'https://clubhouse.com/') === 0) {
            return 'https://www.clubhouse.com' . substr($input, 19);
        }

        // If the input starts with "http://clubhouse.com/", replace it with "https://www.clubhouse.com/"
        if (filter_var($input, FILTER_VALIDATE_URL) && strpos($input, 'http://clubhouse.com/') === 0) {
            return 'https://www.clubhouse.com' . substr($input, 18);
        }

        // If the input starts with "https://www.clubhouse.com/club/", return it as is
        if (filter_var($input, FILTER_VALIDATE_URL) && strpos($input, 'https://www.clubhouse.com/club/') === 0) {
            return $input;
        }

        // If the input starts with "http://www.clubhouse.com/club/", replace it with "https://"
        if (filter_var($input, FILTER_VALIDATE_URL) && strpos($input, 'http://www.clubhouse.com/club/') === 0) {
            return str_replace('http://', 'https://', $input);
        }

        // If the input starts with "www.clubhouse.com/club/", prepend "https://www."
        if (strpos($input, 'www.clubhouse.com/club/') === 0) {
            return 'https://' . $input;
        }

        // If the input starts with "clubhouse.com/club/", prepend "https://www.clubhouse.com/"
        if (strpos($input, 'clubhouse.com/club/') === 0) {
            return 'https://www.' . $input;
        }

        // If the input is just a username (starts with "@"), prepend "https://www.clubhouse.com/"
        if (strpos($input, '@') === 0) {
            return 'https://www.clubhouse.com/' . $input;
        }

        // If the input is just a club name (does not start with "@"), prepend "https://www.clubhouse.com/club/"
        if (!empty($input)) {
            $input = trim($input);
            if (strpos($input, '@') !== 0) {
                return 'https://www.clubhouse.com/club/' . $input;
            }
        }

        // If the input does not match any valid format, throw an exception
        throw new \InvalidArgumentException("The input must be a valid Clubhouse link, username, or club name.");
    }

    // public static function saveFacebookLink(string $input): string
    // {
    //     $input = trim($input);

    //     // If it's a full URL, make sure it's a Facebook URL
    //     if (filter_var($input, FILTER_VALIDATE_URL)) {
    //         $host = parse_url($input, PHP_URL_HOST);

    //         // Block all non-facebook domains
    //         if (strpos($host, 'facebook.com') === false) {
    //             throw new \InvalidArgumentException("Only Facebook URLs are allowed.");
    //         }

    //         // Normalize "http://www.facebook.com/" → "https://"
    //         if (strpos($input, 'http://www.facebook.com/') === 0) {
    //             return str_replace('http://', 'https://', $input);
    //         }

    //         // Normalize "http://facebook.com/" → "https://www.facebook.com/"
    //         if (strpos($input, 'http://facebook.com/') === 0) {
    //             return 'https://www.facebook.com' . substr($input, strlen('http://facebook.com/'));
    //         }

    //         // Normalize "https://facebook.com/" → "https://www.facebook.com/"
    //         if (strpos($input, 'https://facebook.com/') === 0) {
    //             return 'https://www.facebook.com' . substr($input, strlen('https://facebook.com/'));
    //         }

    //         // Already valid full form: "https://www.facebook.com/"
    //         if (strpos($input, 'https://www.facebook.com/') === 0) {
    //             return $input;
    //         }
    //     }

    //     // Normalize input like "facebook.com/..." or "www.facebook.com/..."
    //     if (strpos($input, 'facebook.com/') === 0) {
    //         return 'https://www.' . $input;
    //     }

    //     if (strpos($input, 'www.facebook.com/') === 0) {
    //         return 'https://' . $input;
    //     }

    //     // If input starts with "@username"
    //     if (strpos($input, '@') === 0) {
    //         return 'https://www.facebook.com/' . ltrim($input, '@');
    //     }

    //     // If input is just a username (no @), like "giovani.brasileno"
    //     if (!empty($input) && strpos($input, '/') === false && strpos($input, ' ') === false) {
    //         return 'https://www.facebook.com/' . $input;
    //     }

    //     // Nothing matched — throw error
    //     throw new \InvalidArgumentException("The input must be a valid Facebook link or username.");
    // }

    public static function saveFacebookLink(string $input): string
    {
        $input = trim($input);

        // If input starts with "@username"
        if (str_starts_with($input, '@')) {
            return 'https://www.facebook.com/' . ltrim($input, '@');
        }

        // If input is a full URL
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $parts = parse_url($input);

            // Make sure host contains facebook.com
            if (!isset($parts['host']) || !str_contains($parts['host'], 'facebook.com')) {
                throw new \InvalidArgumentException("Only Facebook URLs are allowed.");
            }

            // Normalize host to www.facebook.com
            $host = 'www.facebook.com';

            // Preserve the path (remove leading @ if exists)
            $path = $parts['path'] ?? '';
            $path = ltrim($path, '@');

            return 'https://' . $host . $path;
        }

        // Input is just a username without @
        if (!empty($input) && !str_contains($input, ['/', ' '])) {
            return 'https://www.facebook.com/' . $input;
        }

        throw new \InvalidArgumentException("The input must be a valid Facebook link or username.");
    }

    public static function saveInstagramLink(string $input): string
    {
        $input = trim($input);

        // If it's a full URL, validate and check domain
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            // Block non-instagram domains
            if (strpos($host, 'instagram.com') === false) {
                throw new \InvalidArgumentException("Only Instagram URLs are allowed.");
            }

            // Normalize "http://www.instagram.com/" → "https://"
            if (strpos($input, 'http://www.instagram.com/') === 0) {
                return str_replace('http://', 'https://', $input);
            }

            // Normalize "http://instagram.com/" → "https://www.instagram.com/"
            if (strpos($input, 'http://instagram.com/') === 0) {
                return 'https://www.instagram.com' . substr($input, strlen('http://instagram.com/'));
            }

            // Normalize "https://instagram.com/" → "https://www.instagram.com/"
            if (strpos($input, 'https://instagram.com/') === 0) {
                return 'https://www.instagram.com' . substr($input, strlen('https://instagram.com/'));
            }

            // Already valid: "https://www.instagram.com/"
            if (strpos($input, 'https://www.instagram.com/') === 0) {
                return $input;
            }
        }

        // Normalize input like "instagram.com/..." or "www.instagram.com/..."
        if (strpos($input, 'instagram.com/') === 0) {
            return 'https://www.' . $input;
        }

        if (strpos($input, 'www.instagram.com/') === 0) {
            return 'https://' . $input;
        }

        // If it's a username like "@username", convert to link
        if (strpos($input, '@') === 0) {
            return 'https://www.instagram.com/' . ltrim($input, '@');
        }

        // Throw exception if nothing matches
        throw new \InvalidArgumentException("The input must be a valid Instagram link or username.");
    }

    public static function saveThreadsLink(string $input): string
    {
        $input = trim($input);

        // If it's a full URL, validate and check domain
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            // Block non-threads domains
            if (strpos($host, 'threads.net') === false) {
                throw new \InvalidArgumentException("Only Threads URLs are allowed.");
            }

            // Normalize "http://www.threads.net/" → "https://"
            if (strpos($input, 'http://www.threads.net/') === 0) {
                return str_replace('http://', 'https://', $input);
            }

            // Normalize "http://threads.net/" → "https://www.threads.net/"
            if (strpos($input, 'http://threads.net/') === 0) {
                return 'https://www.threads.net' . substr($input, strlen('http://threads.net/'));
            }

            // Normalize "https://threads.net/" → "https://www.threads.net/"
            if (strpos($input, 'https://threads.net/') === 0) {
                return 'https://www.threads.net' . substr($input, strlen('https://threads.net/'));
            }

            // Already valid: "https://www.threads.net/"
            if (strpos($input, 'https://www.threads.net/') === 0) {
                return $input;
            }
        }

        // Normalize input like "threads.net/..." or "www.threads.net/..."
        if (strpos($input, 'threads.net/') === 0) {
            return 'https://www.' . $input;
        }

        if (strpos($input, 'www.threads.net/') === 0) {
            return 'https://' . $input;
        }

        // If it's a username like "@username" or "username", convert to link
        if (strpos($input, '@') === 0) {
            return 'https://www.threads.net/' . ltrim($input, '@');
        } elseif (!preg_match('/^https?:\/\//', $input) && strpos($input, '.') === false) {
            return 'https://www.threads.net/' . $input;
        }

        // Throw exception if nothing matches
        throw new \InvalidArgumentException("The input must be a valid Threads link or username.");
    }

    public static function saveTwitterXLink(string $input): string
    {
        $input = trim($input);

        // If it's a full URL
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            // Allow only twitter.com or x.com
            if (
                strpos($host, 'twitter.com') === false &&
                strpos($host, 'x.com') === false
            ) {
                throw new \InvalidArgumentException("Only Twitter or X URLs are allowed.");
            }

            $path = parse_url($input, PHP_URL_PATH);
            return 'https://www.x.com' . $path;
        }

        // Partial input like "twitter.com/user" or "x.com/user"
        if (strpos($input, 'twitter.com/') === 0 || strpos($input, 'x.com/') === 0) {
            return 'https://www.x.com/' . substr($input, strpos($input, '/') + 1);
        }

        // Handles "www.twitter.com/user" or "www.x.com/user"
        if (strpos($input, 'www.twitter.com/') === 0 || strpos($input, 'www.x.com/') === 0) {
            return 'https://www.x.com/' . substr($input, strpos($input, '/') + 1);
        }

        // Handles @username
        if (strpos($input, '@') === 0) {
            return 'https://www.x.com/' . ltrim($input, '@');
        }

        throw new \InvalidArgumentException("The input must be a valid Twitter/X link or username.");
    }

    // public static function saveYoutubeChannel(string $input): string
    // {
    //     $input = trim($input);

    //     // If it's a full URL
    //     if (filter_var($input, FILTER_VALIDATE_URL)) {
    //         $host = parse_url($input, PHP_URL_HOST);

    //         if (
    //             strpos($host, 'youtube.com') === false &&
    //             strpos($host, 'youtu.be') === false
    //         ) {
    //             throw new \InvalidArgumentException("Only YouTube URLs are allowed.");
    //         }

    //         $path = trim(parse_url($input, PHP_URL_PATH), '/');
    //         $query = parse_url($input, PHP_URL_QUERY);

    //         // Channel URL
    //         if (strpos($path, 'channel/') === 0) {
    //             $url = 'https://www.youtube.com/' . $path;
    //         }
    //         // Handle URL
    //         elseif (strpos($path, '@') === 0) {
    //             $url = 'https://www.youtube.com/' . $path;
    //         }
    //         // Custom URL (e.g., /c/LarryWRobinsonmedia)
    //         elseif (strpos($path, 'c/') === 0) {
    //             $url = 'https://www.youtube.com/' . $path;
    //         } else {
    //             throw new \InvalidArgumentException("Invalid YouTube channel URL.");
    //         }

    //         if ($query) {
    //             $url .= '?' . $query;
    //         }

    //         return $url;
    //     }

    //     // Partial URL like "youtube.com/@channelname" or "youtube.com/channel/..."
    //     if (
    //         strpos($input, 'youtube.com/channel/') === 0 ||
    //         strpos($input, 'youtube.com/@') === 0 ||
    //         strpos($input, 'youtube.com/c/') === 0
    //     ) {
    //         return 'https://www.' . $input;
    //     }

    //     // www prefix
    //     if (
    //         strpos($input, 'www.youtube.com/channel/') === 0 ||
    //         strpos($input, 'www.youtube.com/@') === 0 ||
    //         strpos($input, 'www.youtube.com/c/') === 0
    //     ) {
    //         return 'https://' . $input;
    //     }

    //     // Handle only
    //     if (strpos($input, '@') === 0) {
    //         return 'https://www.youtube.com/' . $input;
    //     }

    //     throw new \InvalidArgumentException("The input must be a valid YouTube channel link or handle.");
    // }

    public static function saveYoutubeChannel(string $input): string
    {
        $input = trim($input);

        // If it's a full URL
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            if (
                strpos($host, 'youtube.com') === false &&
                strpos($host, 'youtu.be') === false
            ) {
                throw new \InvalidArgumentException("Only YouTube URLs are allowed.");
            }

            $path = trim(parse_url($input, PHP_URL_PATH), '/');
            $query = parse_url($input, PHP_URL_QUERY);

            // Playlist
            if (strpos($path, 'playlist') === 0 && $query) {
                parse_str($query, $queryParams);
                if (!empty($queryParams['list'])) {
                    return 'https://www.youtube.com/playlist?list=' . $queryParams['list'];
                } else {
                    throw new \InvalidArgumentException("Invalid YouTube playlist URL.");
                }
            }

            // Channel URL
            if (strpos($path, 'channel/') === 0) {
                return 'https://www.youtube.com/' . $path . ($query ? '?' . $query : '');
            }

            // Handle URL
            if (strpos($path, '@') === 0) {
                return 'https://www.youtube.com/' . $path . ($query ? '?' . $query : '');
            }

            // Custom URL
            if (strpos($path, 'c/') === 0) {
                return 'https://www.youtube.com/' . $path . ($query ? '?' . $query : '');
            }

            throw new \InvalidArgumentException("Invalid YouTube channel or playlist URL.");
        }

        // Partial playlist link
        if (strpos($input, 'youtube.com/playlist?list=') === 0 || strpos($input, 'www.youtube.com/playlist?list=') === 0) {
            parse_str(parse_url('https://' . ltrim($input, '/'), PHP_URL_QUERY), $queryParams);
            if (!empty($queryParams['list'])) {
                return 'https://www.youtube.com/playlist?list=' . $queryParams['list'];
            } else {
                throw new \InvalidArgumentException("Invalid YouTube playlist input.");
            }
        }

        // Partial channel URLs
        if (
            strpos($input, 'youtube.com/channel/') === 0 ||
            strpos($input, 'youtube.com/@') === 0 ||
            strpos($input, 'youtube.com/c/') === 0
        ) {
            return 'https://www.' . $input;
        }

        // www prefix channel
        if (
            strpos($input, 'www.youtube.com/channel/') === 0 ||
            strpos($input, 'www.youtube.com/@') === 0 ||
            strpos($input, 'www.youtube.com/c/') === 0
        ) {
            return 'https://' . $input;
        }

        // Handle only
        if (strpos($input, '@') === 0) {
            return 'https://www.youtube.com/' . $input;
        }

        throw new \InvalidArgumentException("The input must be a valid YouTube channel, handle, or playlist link.");
    }


    public static function saveTiktokLink(string $input): string
    {
        $input = trim($input);

        // If input is a full URL
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            // Normalize host to ignore "www."
            $host = str_replace('www.', '', $host);

            if (strpos($host, 'tiktok.com') === false) {
                throw new \InvalidArgumentException("Only TikTok URLs are allowed.");
            }

            $path = trim(parse_url($input, PHP_URL_PATH), '/');

            if (strpos($path, '@') === 0) {
                return 'https://www.tiktok.com/' . $path;
            }

            throw new \InvalidArgumentException("Invalid TikTok profile URL.");
        }

        // If input is partial like "tiktok.com/@username" or "www.tiktok.com/@username"
        if (
            strpos($input, 'tiktok.com/@') === 0 ||
            strpos($input, 'www.tiktok.com/@') === 0
        ) {
            $path = strstr($input, '@');
            return 'https://www.tiktok.com/' . $path;
        }

        // If input starts with @username
        if (strpos($input, '@') === 0) {
            return 'https://www.tiktok.com/' . $input;
        }

        throw new \InvalidArgumentException("The input must be a valid TikTok profile link or username.");
    }


    // public static function saveLinkedinLink(string $input): string
    // {
    //     $input = trim($input);

    //     // If input is a full URL
    //     if (filter_var($input, FILTER_VALIDATE_URL)) {
    //         $host = parse_url($input, PHP_URL_HOST);

    //         if (strpos($host, 'linkedin.com') === false) {
    //             throw new \InvalidArgumentException("Only LinkedIn URLs are allowed.");
    //         }

    //         $path = trim(parse_url($input, PHP_URL_PATH), '/');

    //         if (strpos($path, 'in/') === 0) {
    //             return 'https://www.linkedin.com/' . $path;
    //         }

    //         throw new \InvalidArgumentException("Invalid LinkedIn profile URL. Must start with /in/.");
    //     }

    //     // If input is partial like "linkedin.com/in/username" or "www.linkedin.com/in/username"
    //     if (
    //         strpos($input, 'linkedin.com/in/') === 0 ||
    //         strpos($input, 'www.linkedin.com/in/') === 0
    //     ) {
    //         $path = strstr($input, 'in/');
    //         return 'https://www.linkedin.com/' . $path;
    //     }

    //     // If input looks like a raw username (no slashes, no @, no spaces)
    //     if (!empty($input) && strpos($input, '/') === false && strpos($input, ' ') === false) {
    //         return 'https://www.linkedin.com/in/' . $input;
    //     }

    //     throw new \InvalidArgumentException("The input must be a valid LinkedIn profile link or username.");
    // }

    public static function saveLinkedinLink(string $input): string
    {
        $input = trim($input);

        // Acceptable LinkedIn path prefixes
        $validPrefixes = ['in/', 'company/'];

        // If input is a full URL
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            if (strpos($host, 'linkedin.com') === false) {
                throw new \InvalidArgumentException("Only LinkedIn URLs are allowed.");
            }

            $path = trim(parse_url($input, PHP_URL_PATH), '/');

            foreach ($validPrefixes as $prefix) {
                if (strpos($path, $prefix) === 0) {
                    return 'https://www.linkedin.com/' . $path;
                }
            }

            throw new \InvalidArgumentException("Invalid LinkedIn URL. Must start with /in/ or /company/.");
        }

        // If input is partial like "linkedin.com/in/username" or "www.linkedin.com/company/name"
        if (
            preg_match('#^(www\.)?linkedin\.com/(in|company)/#', $input)
        ) {
            $path = strstr($input, 'in/') ?: strstr($input, 'company/');
            return 'https://www.linkedin.com/' . $path;
        }

        // If input looks like a raw username or company handle (no slashes, no @, no spaces)
        if (!empty($input) && strpos($input, '/') === false && strpos($input, ' ') === false) {
            // You could optionally validate whether it's a user or company based on a parameter
            return 'https://www.linkedin.com/in/' . $input;
        }

        throw new \InvalidArgumentException("The input must be a valid LinkedIn profile or company link, or a raw username.");
    }

    public static function savePinterestLink(string $input): string
    {
        $input = trim($input);

        // If input is a full URL
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            if (strpos($host, 'pinterest.com') === false) {
                throw new \InvalidArgumentException("Only Pinterest URLs are allowed.");
            }

            $path = trim(parse_url($input, PHP_URL_PATH), '/');

            // Reject if path is empty or has slashes (e.g., boards, pins, etc.)
            if (!empty($path) && strpos($path, '/') === false) {
                return 'https://www.pinterest.com/' . $path . '/';
            }

            throw new \InvalidArgumentException("Invalid Pinterest profile URL. Must point to a user profile.");
        }

        // If input is partial like "pinterest.com/username" or "www.pinterest.com/username"
        if (preg_match('#^(www\.)?pinterest\.com/([^/]+)/?$#', $input, $matches)) {
            return 'https://www.pinterest.com/' . $matches[2] . '/';
        }

        // If input looks like a raw username (no slashes or spaces)
        if (!empty($input) && preg_match('/^[a-zA-Z0-9._]+$/', $input)) {
            return 'https://www.pinterest.com/' . $input . '/';
        }

        throw new \InvalidArgumentException("The input must be a valid Pinterest user profile link or username.");
    }

    // public static function normalizeFanbaseLink(string $input): string
    // {
    //     $input = trim($input);

    //     // Validate and normalize full URLs
    //     if (filter_var($input, FILTER_VALIDATE_URL)) {
    //         $host = parse_url($input, PHP_URL_HOST);

    //         // Ensure the domain is fanbase.app
    //         if (strpos($host, 'fanbase.app') === false) {
    //             throw new \InvalidArgumentException("Only Fanbase URLs are allowed.");
    //         }

    //         // Normalize to HTTPS and remove www subdomain if present
    //         $input = str_replace(['http://', 'www.'], ['https://', ''], $input);

    //         return $input;
    //     }

    //     // Normalize partial URLs
    //     if (strpos($input, 'fanbase.app/') === 0) {
    //         return 'https://' . $input;
    //     } elseif (strpos($input, 'www.fanbase.app/') === 0) {
    //         return 'https://fanbase.app/' . substr($input, strlen('www.fanbase.app/'));
    //     }

    //     // Convert usernames to links
    //     if (strpos($input, '@') === 0) {
    //         return 'https://fanbase.app/' . ltrim($input, '@');
    //     } elseif (!preg_match('/^https?:\/\//', $input) && strpos($input, '.') === false) {
    //         return 'https://fanbase.app/' . $input;
    //     }

    //     // Throw exception for invalid input
    //     throw new \InvalidArgumentException("The input must be a valid Fanbase link or username.");
    // }

    public static function normalizeFanbaseLink(string $input): string
    {
        $input = trim($input);

        // Validate and normalize full URLs
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            // Ensure the domain is fanbase.app
            if ($host !== 'fanbase.app' && $host !== 'www.fanbase.app') {
                throw new \InvalidArgumentException("Only Fanbase URLs are allowed.");
            }

            // Normalize to HTTPS and remove www subdomain if present
            if (parse_url($input, PHP_URL_SCHEME) === 'http') {
                $input = str_replace('http://', 'https://', $input);
            }
            if (strpos($host, 'www.') === 0) {
                $input = str_replace('www.', '', $input);
            }

            return $input;
        }

        // Normalize partial URLs
        if (strpos($input, 'fanbase.app/') === 0) {
            return 'https://' . $input;
        } elseif (strpos($input, 'www.fanbase.app/') === 0) {
            return 'https://fanbase.app/' . substr($input, strlen('www.fanbase.app/'));
        }

        // Convert usernames to links
        if (strpos($input, '@') === 0) {
            return 'https://fanbase.app/' . ltrim($input, '@');
        } elseif (!preg_match('/^https?:\/\//', $input) && strpos($input, '.') === false) {
            return 'https://fanbase.app/' . $input;
        }

        // Throw exception for invalid input
        throw new \InvalidArgumentException("The input must be a valid Fanbase link or username.");
    }

    public static function normalizeSnapchatLink(string $input): string
    {
        $input = trim($input);

        // Validate and normalize full URLs
        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            // Ensure the domain is snapchat.com
            if (strpos($host, 'snapchat.com') === false) {
                throw new \InvalidArgumentException("Only Snapchat URLs are allowed.");
            }

            // Normalize to HTTPS and remove www subdomain if present
            $input = str_replace(['http://', 'www.'], ['https://', ''], $input);

            // Ensure addy format
            if (strpos($input, 'https://snapchat.com/add/') !== 0) {
                $path = parse_url($input, PHP_URL_PATH);
                $input = 'https://snapchat.com/add' . $path;
            }

            return $input;
        }

        // Convert usernames to links
        if (strpos($input, '@') === 0) {
            return 'https://snapchat.com/add/' . ltrim($input, '@');
        } elseif (!preg_match('/^https?:\/\//', $input) && strpos($input, '.') === false) {
            return 'https://snapchat.com/add/' . $input;
        }

        // Normalize partial URLs
        if (strpos($input, 'snapchat.com/add/') === 0) {
            return 'https://' . str_replace('http://', '', $input);
        }

        // Throw exception for invalid input
        throw new \InvalidArgumentException("The input must be a valid Snapchat link or username.");
    }

    public static function normalizeSoundCloudLink(string $input): string
    {
        $input = trim($input);

        if (filter_var($input, FILTER_VALIDATE_URL)) {
            $host = parse_url($input, PHP_URL_HOST);

            if (strpos($host, 'soundcloud.com') === false) {
                throw new \InvalidArgumentException("Only SoundCloud URLs are allowed.");
            }

            $path = parse_url($input, PHP_URL_PATH);
            return 'https://soundcloud.com' . rtrim($path, '/');
        }

        if (!preg_match('/^https?:\/\//', $input) && strpos($input, '.') === false) {
            return 'https://soundcloud.com/' . ltrim($input, '@');
        }

        throw new \InvalidArgumentException("The input must be a valid SoundCloud link or username.");
    }

    public static function normalizeApplePodcastsLink(string $input): string
    {
        $input = trim($input);

        if (!filter_var($input, FILTER_VALIDATE_URL)) {
            throw new \InvalidArgumentException("Apple Podcasts links must be full URLs.");
        }

        $host = parse_url($input, PHP_URL_HOST);

        if (strpos($host, 'podcasts.apple.com') === false) {
            throw new \InvalidArgumentException("Only Apple Podcasts URLs are allowed.");
        }

        $path = parse_url($input, PHP_URL_PATH);
        return 'https://podcasts.apple.com' . rtrim($path, '/');
    }

    public static function normalizeSpotifyLink(string $input): string
    {
        $input = trim($input);

        if (!filter_var($input, FILTER_VALIDATE_URL)) {
            throw new \InvalidArgumentException("Spotify links must be full URLs.");
        }

        $host = parse_url($input, PHP_URL_HOST);

        if (strpos($host, 'spotify.com') === false) {
            throw new \InvalidArgumentException("Only Spotify URLs are allowed.");
        }

        $path = parse_url($input, PHP_URL_PATH);
        return 'https://open.spotify.com' . rtrim($path, '/');
    }

    public static function normalizeDiscordLink(string $input): string
    {
        $input = trim($input);

        if (!filter_var($input, FILTER_VALIDATE_URL)) {
            throw new \InvalidArgumentException("Discord invite must be a full URL.");
        }

        $host = parse_url($input, PHP_URL_HOST);
        $path = parse_url($input, PHP_URL_PATH);

        if (
            strpos($host, 'discord.gg') !== false ||
            (strpos($host, 'discord.com') !== false && strpos($path, '/invite/') === 0)
        ) {
            return 'https://' . $host . rtrim($path, '/');
        }

        throw new \InvalidArgumentException("Only valid Discord invite links are allowed.");
    }

    public static function normalizeSkoolLink(string $input): string
    {
        $input = trim($input);

        // Ensure it's a valid URL
        if (!filter_var($input, FILTER_VALIDATE_URL)) {
            throw new \InvalidArgumentException("Skool invite must be a full URL.");
        }

        $parts = parse_url($input);

        $host = $parts['host'] ?? '';
        $path = $parts['path'] ?? '';

        // Normalize host (allow optional www)
        if (strpos($host, 'skool.com') === false) {
            throw new \InvalidArgumentException("Only valid Skool invite links are allowed.");
        }

        // Path must start with /join/
        if (strpos($path, '/join/') !== 0) {
            throw new \InvalidArgumentException("Only valid Skool invite links are allowed.");
        }

        // Rebuild URL with https and strip trailing slash
        $normalizedUrl = 'https://skool.com' . rtrim($path, '/');

        return $normalizedUrl;
    }
}
