<?php

namespace App\Http\Controllers;

use App\Models\Button;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\DB;
use Livewire\Attributes\On;
use App\Settings\CardDefaultsSetting;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Models\User;
use App\Models\SubCard;
use App\Models\TrustedNetwork;
use App\Models\UserSettings;

class AuthController
{
    public function welcome()
    {
        $default = app(CardDefaultsSetting::class);
        return view('welcome', [
            'site_name' => $default->site_name,
            'site_url' => $default->site_url,
            'site_desc' => $default->site_desc,
            'app_name' => $default->app_name,
            'thumbnail' => $default->thumbnail,
            'site_button_color' => $default->site_button_color,
            'site_button_text_color' => $default->site_button_text_color,
            'site_link_color' => $default->site_link_color,
        ]);
    }

    public function login()
    {
        $default = app(CardDefaultsSetting::class);
        // dd($default->logo);
        return view('login', [
            'site_name' => $default->site_name,
            'site_url' => $default->site_url,
            'site_desc' => $default->site_desc,
            'thumbnail' => $default->thumbnail,
            'app_name' => $default->app_name,
            'logo' => $default->logo,
            'site_button_color' => $default->site_button_color,
            'site_button_text_color' => $default->site_button_text_color,
            'site_link_color' => $default->site_link_color,
        ]);
    }

    public function authenticate(Request $request)
    {
        $trusted_network_username = $request->input('trusted-network');

        $credentials = $request->validate([
            'email' => 'required|email:filter',
            'password' => 'required|string'
        ]);

        if (Auth::attempt($credentials)) {
            $user = Auth::user();

            if (!$user->is_active) {
                Auth::logout(); // Log the user out if they are not active
                throw ValidationException::withMessages([
                    'credentials' => 'Your account is inactive. Please contact support.'
                ]);
            }

            $request->session()->regenerate();

            // if logged in on trusted network
            if ($trusted_network_username) {
                $check = User::where('username', $trusted_network_username)->first()
                    ?? SubCard::where('username', $trusted_network_username)->first();

                $owner_type = $check instanceof User ? 'main' : 'sub';
                $owner_id   = $check->id;

                // Prevent duplicates
                $exists = TrustedNetwork::where('member_id', $user->id)
                    ->where('card_type', 'main')
                    ->where(function ($q) use ($owner_type, $owner_id) {
                        if ($owner_type == 'main') {
                            $q->where('user_id', $owner_id);
                        } else {
                            $q->where('page_id', $owner_id);
                        }
                    })
                    ->exists();

                if (!$exists) {
                    // Create record
                    TrustedNetwork::create([
                        'user_id'     => $owner_type == 'main' ? $owner_id : null,
                        'page_id'     => $owner_type == 'sub' ? $owner_id : null,
                        'member_id'   => $user->id,
                        'card_type'   => 'main', // main or sub from checkbox
                        'type'        => 'network',
                        'is_accepted' => false,
                    ]);
                }

                return redirect()->intended(route('card.trustednetwork', $trusted_network_username))->with('success', 'Requested! Card owner will be notified about the request.');
            }

            // else go to dashboard
            return redirect()->intended(route('dashboard.index'))->with('success', 'You are now logged in!');
        }

        // Throw validation exception if credentials are incorrect
        throw ValidationException::withMessages([
            'credentials' => 'Sorry, incorrect credentials.'
        ]);
    }

    public function logout(Request $request)
    {
        $user = $request->user();

        $activeSubCard = SubCard::where('user_id', $user->id)
            ->where('active_card', true)
            ->first();

        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return to_route('welcome');
    }

    public function show()
    {
        $default = app(CardDefaultsSetting::class);
        return view('register', [
            'site_name' => $default->site_name,
            'site_url' => $default->site_url,
            'site_desc' => $default->site_desc,
            'app_name' => $default->app_name,
            'site_button_color' => $default->site_button_color,
            'site_button_text_color' => $default->site_button_text_color,
            'site_link_color' => $default->site_link_color,
        ]);
    }
}
