<?php

namespace App\Livewire;

use App\Models\Ad;
use Livewire\Component;
use Livewire\Attributes\On;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Validation\ValidationException;

class AdEdit extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;
    public $ad_banner_w;
    public $ad_banner_h;

    public $ad, $link, $caption, $image, $position, $is_active;

    public $new_ad_image;
    public $ad_id;

    public $page_id;
    public $condition;

    #[On('adImage')]
    public function setAdImage($adImageUrl)
    {
        $this->new_ad_image = new TemporaryUploadedFile($adImageUrl, config('filesystems.default'));
    }

    public function updateAd()
    {

        try {

            $validated = $this->validate(
                [
                    'caption' => 'required|string',
                    'link' => 'required|string',
                    'position' => 'required|in:top,bottom',
                ], //Custom error messages
                [
                    'caption.required' => "Ad Title is required.",
                ]
            );

            if (auth()->user()) {

                // check if ad has image
                if ($this->new_ad_image) {
                    // Check if the ad has already a picture and delete it
                    if ($this->image) {
                        unlink(storage_path('app/public/' . $this->image));
                    }

                    // dd('Ad image doesnt exist, upload app/public/' . $this->new_ad_image);
                    $validated['image'] = $this->new_ad_image->storePublicly('ads', 'public');

                    // Get the path of the temporary file (before it's stored in the public directory)
                    $tempFilePath = $this->new_ad_image->getRealPath();

                    if (file_exists($tempFilePath)) {
                        unlink($tempFilePath);  // Delete the temporary file explicitly
                    }
                }

                // Update the user's ad
                Ad::where('id', $this->ad_id)->update($validated);

                session()->flash('success', 'Ad successfully updated!');
            } else {
                session()->flash('error', 'Unauthorized action.');
            }

            return $this->redirect('/dashboard/ads', navigate: true);
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function viewAd($id)
    {
        $this->ad = Ad::where($this->condition)
            ->where('id', $id)
            ->select('id', 'link', 'caption', 'image', 'position', 'is_active')
            ->first();
    }

    public function mount($id)
    {
        $checkifactive = User::where('id', auth()->user()->id)
            ->where('active_card', true)->count();

        //if subcard is  active
        if ($checkifactive < 1) {
            $this->page_id = SubCard::where('user_id', auth()->user()->id)
                ->where('active_card', true)
                ->pluck('id')
                ->first();
            // $validated['page_id'] = $this->page_id;
            $this->condition = ['page_id' => $this->page_id];
        } else {
            $this->condition = ['user_id' => auth()->user()->id, 'page_id' => NULL];
        }

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;
        $this->ad_banner_w = $default->ad_banner_w;
        $this->ad_banner_h = $default->ad_banner_h;

        $this->ad_id = $id;

        $this->viewAd($this->ad_id);
        $this->link = $this->ad->link;
        $this->caption = $this->ad->caption;
        $this->position = $this->ad->position;
        $this->image = $this->ad->image;
        $this->is_active = $this->ad->is_active;

        $this->dispatch('setEditAdImage', $this->image);
    }

    public function render()
    {
        $this->authorize('update', Ad::class);
        return view('livewire.ad-edit')->title('Edit Ad | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
