<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Event;
use App\Models\SubCard;
use App\Models\User;
use App\Models\Button;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Validation\ValidationException;

class AddEvent extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $poster_active = true;
    // public string $event_type = 'Photo'; // 👈 default is Photo
    public $title, $start_datetime, $button_title, $button_link, $description, $image, $partner_link, $shop_link;
    public array $event_video_link = [''];
    public array $optional_cta = [];
    public $title_active = true;
    public $contact_active = true;
    public $shop_active = true;
    public $ad_active = false;

    public $page_id; //subcard id, used in query
    public $contacts;
    public $condition;
    public $checkMain;

    public $contacts_settings;

    public function addCTA()
    {
        $this->optional_cta[] = [
            'title' => '',
            'link' => '',
        ];
    }

    public function removeCTA($index)
    {
        unset($this->optional_cta[$index]);
        $this->optional_cta = array_values($this->optional_cta); // reindex
    }

    #[On('eventImage')]
    public function setEventImage($eventImageUrl)
    {
        $this->image = new TemporaryUploadedFile($eventImageUrl, config('filesystems.default'));
    }

    public function addVideoLink()
    {
        if (count($this->event_video_link) >= 10) {
            session()->flash('error', 'You can only add up to 10 video links.');
            return;
        }

        $this->event_video_link[] = '';
    }

    public function removeVideoLink($index)
    {
        unset($this->event_video_link[$index]);
        $this->event_video_link = array_values($this->event_video_link); // reindex
    }

    public function saveEvent()
    {
        try {
            $rules = [
                'image'                 => 'required|image|max:2048',
                'event_video_link'      => 'nullable|array|max:10',
                'event_video_link.*'    => 'nullable|url',
                'optional_cta'          => 'nullable|array',
                'optional_cta.title'    => 'nullable|string|max:255',
                'optional_cta.link'     => 'nullable|url',
                'poster_active'         => 'required|boolean',
                'ad_active'             => 'required|boolean',
                'title'                 => 'required|string',
                'start_datetime'        => 'nullable|date',
                'description'           => 'nullable|string|max:1000',
                'button_title'          => 'required|string',
                'button_link'           => 'required|url',
                'partner_link'          => 'nullable|url',
                'shop_link'             => 'nullable|url',
                'title_active'          => 'required|boolean',
                'contact_active'        => 'required|boolean',
                'shop_active'           => 'required|boolean',
            ];

            $validated = $this->validate($rules);

            // Handle image upload
            if ($this->image instanceof TemporaryUploadedFile) {
                $validated['event_poster'] = $this->image->storePublicly('events', 'public');

                $tempFilePath = $this->image->getRealPath();
                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath);
                }
            }

            // Filter out empty links and save as JSON array
            $videoLinks = array_filter($this->event_video_link, fn($link) => !empty($link));
            $validated['event_video_link'] = json_encode($videoLinks ?: []);

            // Cleanly save optional cta
            $validated['optional_cta'] = !empty($this->optional_cta)
                ? array_values($this->optional_cta)
                : null;

            // Assign user/page IDs
            $checkIfActive = User::where('id', auth()->id())
                ->where('active_card', true)->exists();

            if (!$checkIfActive) {
                $this->page_id = SubCard::where('user_id', auth()->id())
                    ->where('active_card', true)
                    ->value('id');

                $validated['page_id'] = $this->page_id;
                $validated['user_id'] = null;
            } else {
                $validated['user_id'] = auth()->id();
            }

            try {
                // $validated['contact_settings'] = json_encode($this->contacts_settings);
                // $validated['contact_settings'] = json_encode($this->contacts_settings, JSON_THROW_ON_ERROR);
                $validated['contact_settings'] = $this->contacts_settings;

                Event::create($validated);
                session()->flash('success', 'Event successfully added!');
                return $this->redirect('/dashboard/event', navigate: true);
            } catch (ValidationException $e) {
                // dd('Validation error: ' . $e->getMessage());
                throw $e;
            } catch (\Throwable $e) {
                // dd('Error creating event: ' . $e->getMessage());
                session()->flash('error', 'Something went wrong while creating the event.');
                return redirect()->back()->withInput();
            }
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function toggleAdActive()
    {
        $this->ad_active = !$this->ad_active;
    }

    public function togglePosterActive()
    {
        $this->poster_active = !$this->poster_active;
    }

    public function toggleTitleActive()
    {
        $this->title_active = !$this->title_active;
    }

    public function toggleContactActive()
    {
        $this->contact_active = !$this->contact_active;
    }

    public function toggleShopActive()
    {
        $this->shop_active = !$this->shop_active;
    }

    public function mount()
    {
        $checkIfActive = User::where('id', auth()->id())
            ->where('active_card', true)->exists();

        if (!$checkIfActive) {
            $this->page_id = SubCard::where('user_id', auth()->id())
                ->where('active_card', true)
                ->value('id');
            $this->condition = ['page_id' => $this->page_id];
        } else {
            $this->condition = ['user_id' => auth()->user()->id, 'page_id' => NULL];
        }

        $this->contacts = Button::where($this->condition)
            ->where('is_active', true)
            ->where('view', 'contacts')
            ->orderBy('order', 'asc')
            ->get();

        // ✅ FIX: Ensure contacts_settings is always an array
        if (!is_array($this->contacts_settings)) {
            $this->contacts_settings = [];
        }

        // Ensure every contact gets a toggle entry
        foreach ($this->contacts as $contact) {
            if (!array_key_exists($contact->id, $this->contacts_settings)) {
                $this->contacts_settings[$contact->id] = true; // default enabled
            }
        }
    }

    public function render()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->authorize('create', Event::class);

        return view('livewire.add-event')->title('Add Event | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
