<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\ProductCollection;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use App\Settings\CardDefaultsSetting;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Validation\ValidationException;

class AddProductCollection extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $image, $title, $description;

    public $page_id; //subcard id, used in query

    #[On('bannerImage')]
    public function setBannerImage($adImageUrl)
    {
        $this->image = new TemporaryUploadedFile($adImageUrl, config('filesystems.default'));
    }

    public function saveCollection()
    {
        try {
            $validated = $this->validate(
                [
                    'image'         => 'required|image|mimes:jpeg,png,jpg,webp,heic,heif|max:5120',
                    'title'         => 'required|string',
                    'description'   => 'nullable|string|max:500'
                ],
                [
                    'image.required' => "Product Collection Banner is required.",
                    'image.image' => "The file must be an image (jpeg, png, jpg, heic, heif or webp).",
                ]
            );

            if ($this->image) {
                $validated['image'] =  $this->image->storePublicly('product_collection', 'public');

                // Get the path of the temporary file (before it's stored in the public directory)
                $tempFilePath = $this->image->getRealPath();

                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath);  // Delete the temporary file explicitly
                }
            }

            $user = auth()->user();
            $checkMainActive = User::where('id', $user->id)
                ->where('active_card', true)
                ->exists();

            if ($checkMainActive) {
                $validated['user_id'] = $user->id;
            } else {
                $subCardId = SubCard::where('user_id', $user->id)
                    ->where('active_card', true)
                    ->pluck('id')
                    ->first();

                if (!$subCardId) {
                    session()->flash('error', 'No active subcard found.');
                    return;
                }

                $validated['page_id'] = $subCardId;
            }

            // Create the collection
            $addProductCollection = ProductCollection::create($validated);

            if ($addProductCollection) {
                session()->flash('success', 'Collection successfully added!');
            } else {
                session()->flash('error', 'Failed to add collection.');
            }

            return $this->redirect('/dashboard/product-collections', navigate: true);
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function render()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        // $this->authorize('create', Product::class);
        return view('livewire.add-product-collection')->title('Add Collection | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
