<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Service;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Validation\ValidationException;

class AddService extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $title, $price, $payment_link, $description, $service_image;

    public $page_id; //subcard id, used in query

    #[On('serviceImage')]
    public function setServiceImage($serviceImageUrl)
    {
        $this->service_image = new TemporaryUploadedFile($serviceImageUrl, config('filesystems.default'));
    }

    public function saveService()
    {

        try {

            $validated = $this->validate([
                'title' => 'required|string',
                'price' => 'required|numeric',
                'payment_link' => 'required|url',
                'description' => 'required|string|max:2000'
            ]);

            $validated['user_id'] = auth()->user()->id;


            if ($this->service_image) {
                $validated['service_image'] =  $this->service_image->storePublicly('services', 'public');

                // Get the path of the temporary file (before it's stored in the public directory)
                $tempFilePath = $this->service_image->getRealPath();

                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath);  // Delete the temporary file explicitly
                }
            }

            $checkifactive = User::where('id', auth()->user()->id)
                ->where('active_card', true)->count();

            //if maincard is not active
            if ($checkifactive < 1) {
                $this->page_id = SubCard::where('user_id', auth()->user()->id)
                    ->where('active_card', true)
                    ->pluck('id')
                    ->first();
                $validated['page_id'] = $this->page_id;
            }

            $addService = Service::create($validated);

            if ($addService) {
                session()->flash('success', 'Service successfully added!');
            } else {
                session()->flash('error', 'Unauthorized action.');
            }

            return $this->redirect('/dashboard/service', navigate: true);
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function render()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->authorize('create', Service::class);
        return view('livewire.add-service')->title('Add Service | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
