<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\ProductCategory;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use App\Settings\CardDefaultsSetting;
use Illuminate\Validation\ValidationException;

class EditProductCategory extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $title, $description;

    public $page_id; //subcard id, used in query
    public $condition;
    public $categoryId;
    public $category;

    public function viewCategory($id)
    {
        $this->category = ProductCategory::where($this->condition)
            ->where('id', $id)
            ->select('id', 'title', 'description')
            ->first();

        if (!$this->category) {
            abort(403, 'Access denied');
        }
    }

    public function mount($id)
    {
        $checkIfActive = User::where('id', auth()->id())
            ->where('active_card', true)->exists();

        if (!$checkIfActive) {
            $this->page_id = SubCard::where('user_id', auth()->id())
                ->where('active_card', true)
                ->value('id');
            $this->condition = ['page_id' => $this->page_id];
        } else {
            $this->condition = ['user_id' => auth()->user()->id, 'page_id' => NULL];
        }

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->categoryId = $id;
        $this->viewCategory($this->categoryId);
        $this->title = $this->category->title;
        $this->description = $this->category->description;
    }

    public function updateCategory()
    {
        try {
            $validated = $this->validate([
                'title'       => 'required|string',
                'description' => 'nullable|string|max:500'
            ]);

            $user = auth()->user();
            $checkMainActive = User::where('id', $user->id)
                ->where('active_card', true)
                ->exists();

            if ($checkMainActive) {
                $validated['user_id'] = $user->id;
            } else {
                $subCardId = SubCard::where('user_id', $user->id)
                    ->where('active_card', true)
                    ->pluck('id')
                    ->first();

                if (!$subCardId) {
                    session()->flash('error', 'No active subcard found.');
                    return;
                }

                $validated['page_id'] = $subCardId;
            }

            $update = ProductCategory::where('id', $this->categoryId)->update($validated);

            if ($update) {
                session()->flash('success', 'Category successfully updated!');
            } else {
                session()->flash('error', 'Failed to add category.');
            }

            return $this->redirect('/dashboard/product-categories', navigate: true);
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function render()
    {
        // $this->authorize('update', ProductCategory::class);
        return view('livewire.edit-product-category')->title('Edit Category | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
