<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Sale;
use App\Models\SaleImage;
use App\Models\SaleVideo;
use App\Models\User;
use App\Models\SubCard;
use Livewire\Attributes\On;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use App\Settings\CardDefaultsSetting;

class EditSalesPage extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;

    public $page_id, $condition, $checkifactive;

    public $sales, $title, $description, $is_active;
    public $sales_id;
    public $new_sales_thumbnail = null;
    public $sales_videos;
    public $sales_images;
    public $sales_thumbnail, $old_thumbnail, $image_path;

    #[On('salesImage')]
    public function setSalesImage($salesThumbnailUrl)
    {
        $this->new_sales_thumbnail = new TemporaryUploadedFile(
            $salesThumbnailUrl,
            config('filesystems.default')
        );
    }

    public function updateSales()
    {
        try {
            $validated = $this->validate([
                'title'         => 'required|string',
                'description'   => 'nullable|string',
                'is_active'     => 'required|boolean',
            ]);

            if ($this->checkifactive < 1) {
                $validated['page_id'] = $this->page_id;
            } else {
                $validated['user_id'] = auth()->user()->id;
            }

            // $newImage = SaleImage::findOrFail($this->image);
            $update = Sale::findOrFail($this->sales_id);

            // Handle new image
            if ($this->new_sales_thumbnail) {
                if ($this->old_thumbnail && Storage::disk('public')->exists($this->old_thumbnail)) {
                    Storage::disk('public')->delete($this->old_thumbnail);
                }
                $validated['thumbnail'] = $this->new_sales_thumbnail->storePublicly('sales', 'public');
            }

            $update->fill($validated);
            $update->save();

            session()->flash('success', 'Sales page successfully updated!');

            return $this->redirectRoute('sales.edit', [
                'id' => $this->sales_id,
            ], navigate: true);
        } catch (ValidationException $e) {
            dd($e);
            throw $e;
        } catch (\Throwable $e) {
            session()->flash('error', 'Something went wrong while updating the sales page.');
            return redirect()->back()->withInput();
        }
    }

    public function mount($id)
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;

        $this->sales_id = $id;

        $this->checkifactive = User::where('id', auth()->user()->id)
            ->where('active_card', true)->count();

        //if maincard is not active
        if ($this->checkifactive < 1) {
            $this->page_id = SubCard::where('user_id', auth()->user()->id)
                ->where('active_card', true)
                ->pluck('id')
                ->first();

            $this->condition = ['page_id' => $this->page_id];
        } else {
            $this->condition = ['user_id' => auth()->user()->id, 'page_id' => NULL];
        }

        $this->sales = Sale::where($this->condition)
            ->where('id', $id)
            ->select('id', 'title', 'description', 'is_active', 'thumbnail')
            ->first();

        $this->title = $this->sales->title;
        $this->is_active = $this->sales->is_active;
        $this->description = $this->sales->description;
        $this->sales_thumbnail = $this->sales->thumbnail;
        $this->old_thumbnail =  $this->sales->thumbnail;
        $this->image_path = $this->sales->thumbnail;

        $this->sales_videos = SaleVideo::where('sales_id', $this->sales_id)
            ->select('video_link', 'order', 'is_active', 'shop_online_button', 'shop_online_link')
            ->ordered()->get();

        $this->sales_images = SaleImage::where('sales_id', $this->sales_id)
            ->select('image_path', 'order', 'is_active', 'shop_online_button', 'shop_online_link')
            ->ordered()->get();

        $this->dispatch('setEditSalesImage', $this->image_path);
    }

    //  public function updateOrder(array $order)
    // {
    //     SaleVideo::setNewOrder($order, 1, 'id', function ($query) {
    //         $query->whereBelongsTo(auth()->user());
    //     });
    // }

    public function render()
    {
        // dd($this->sales_thumbnail);
        $this->authorize('update', Sale::class);
        return view('livewire.edit-sales-page')->title('Edit Sales Page | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
        ]);
    }
}
