<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\Attributes\On;
use App\Models\Event;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use App\Models\Button;

class EventEdit extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $event;
    public $title;
    public $start_datetime;
    public $poster_active;
    public $title_active;
    public $contact_active;
    public $shop_active;
    public $event_image;
    public $payment_link;
    public $old_image;
    public $new_event_image = null;
    public $button_title;
    public $button_link, $event_type, $partner_link, $shop_link, $ad_active, $description;
    public array $event_video_link = [''];
    public array $optional_cta = [];
    public $eventId;
    public $page_id;
    public $contacts;
    public $condition;
    public array $contact_settings = [];

    public function addCTA()
    {
        $this->optional_cta[] = [
            'title' => '',
            'link' => '',
        ];
    }

    public function removeCTA($index)
    {
        unset($this->optional_cta[$index]);
        $this->optional_cta = array_values($this->optional_cta); // reindex
    }

    #[On('eventImage')]
    public function setEventImage($eventImageUrl)
    {
        $this->new_event_image = new TemporaryUploadedFile(
            $eventImageUrl,
            config('filesystems.default')
        );
    }

    public function addVideoLink()
    {
        if (count($this->event_video_link) >= 10) {
            session()->flash('error', 'You can only add up to 10 video links.');
            return;
        }

        $this->event_video_link[] = '';
    }

    public function removeVideoLink($index)
    {
        unset($this->event_video_link[$index]);
        $this->event_video_link = array_values($this->event_video_link); // reindex
    }

    public function updateEvent()
    {
        try {
            $validated = $this->validate([
                'event_video_link'      => 'nullable|array|max:10',
                'event_video_link.*'    => 'nullable|url',
                'optional_cta'          => 'nullable|array',
                'optional_cta.title'    => 'nullable|string|max:255',
                'optional_cta.link'     => 'nullable|url',
                'poster_active'         => 'required|boolean',
                'title'                 => 'required|string',
                'start_datetime'        => 'nullable|date',
                'ad_active'             => 'required|boolean',
                'description'           => 'nullable|string|max:1000',
                'button_title'          => 'required|string',
                'button_link'           => 'required|url',
                'partner_link'          => 'nullable|url',
                'shop_link'             => 'nullable|url',
                'title_active'          => 'required|boolean',
                'contact_active'        => 'required|boolean',
                'shop_active'           => 'required|boolean',
            ]);

            $event = Event::findOrFail($this->eventId);

            // Handle new image
            if ($this->new_event_image) {
                if ($this->old_image && Storage::disk('public')->exists($this->old_image)) {
                    Storage::disk('public')->delete($this->old_image);
                }
                $validated['event_poster'] = $this->new_event_image->storePublicly('events', 'public');
            }

            // Always save event_video_link as JSON array
            $validated['event_video_link'] = json_encode(array_filter($this->event_video_link));

            // Cleanly save optional cta
            $validated['optional_cta'] = !empty($this->optional_cta)
                ? array_values($this->optional_cta)
                : null;

            $event->fill($validated);
            // $event->contact_settings = json_encode($this->contact_settings);
            $event->contact_settings = $this->contact_settings;

            $event->save();

            session()->flash('success', 'Event successfully updated!');
            return $this->redirect('/dashboard/event', navigate: true);
        } catch (ValidationException $e) {
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        } catch (\Throwable $e) {
            $this->dispatch('set-submitting', ['value' => false]);
            session()->flash('error', 'Something went wrong while updating the event.');
            return redirect()->back()->withInput();
        }
    }

    public function viewEvent($id)
    {
        $checkIfActive = User::where('id', auth()->id())
            ->where('active_card', true)
            ->exists();

        $this->condition = $checkIfActive
            ? ['user_id' => auth()->id(), 'page_id' => null]
            : ['page_id' => SubCard::where('user_id', auth()->id())->where('active_card', true)->value('id')];

        $this->event = Event::where('id', $id)
            ->where($this->condition)
            // ->select('id', 'title', 'start_datetime', 'button_title', 'button_link', 'event_poster', 'event_video_link', 'poster_active', 'partner_link', 'shop_link', 'title_active', 'contact_active', 'shop_active', 'description', 'ad_active', 'contact_settings')
            ->firstOrFail();

        $this->contacts = Button::where($this->condition)
            ->where('is_active', true)
            ->where('view', 'contacts')
            ->orderBy('order', 'asc')
            ->get();

        // Decode saved contact_settings from the event
        // $this->contact_settings = json_decode($this->event->contact_settings ?? '{}', true);
        $this->contact_settings = $this->event->contact_settings ?? [];

        // Optional: ensure all contacts have an entry (default false if missing)
        if ($this->contacts) {
            foreach ($this->contacts as $contact) {
                if (!isset($this->contact_settings[$contact->id])) {
                    $this->contact_settings[$contact->id] = false;
                }
            }
        }

        // Cast event_video_link to array if not already
        $this->event_video_link = is_array($this->event->event_video_link)
            ? $this->event->event_video_link
            : json_decode($this->event->event_video_link, true) ?? [];

        // $this->optional_cta = is_array($this->event->optional_cta)
        //     ? $this->event->optional_cta
        //     : json_decode($this->event->optional_cta, true) ?? [];
        // dd($this->event);
        $this->optional_cta = $this->event->optional_cta ?? [];

        if (empty($this->optional_cta)) {
            $this->optional_cta[] = ['title' => '', 'link' => ''];
        }

        $this->title = $this->event->title;
        // $this->start_datetime = $this->event->start_datetime;
        $this->start_datetime = $this->event->start_datetime
            ? \Carbon\Carbon::parse($this->event->start_datetime)->format('Y-m-d')
            : null;

        $this->description      = $this->event->description;
        $this->button_title     = $this->event->button_title;
        $this->ad_active        = $this->event->ad_active;
        $this->button_link      = $this->event->button_link;
        $this->poster_active    = $this->event->poster_active;
        $this->title_active     = $this->event->title_active;
        $this->contact_active   = $this->event->contact_active;
        $this->shop_active      = $this->event->shop_active;
        $this->partner_link     = $this->event->partner_link;
        $this->shop_link        = $this->event->shop_link;
        $this->event_image      = $this->event->event_poster;
        $this->old_image        = $this->event->event_poster;
    }

    public function toggleAdActive()
    {
        $this->ad_active = !$this->ad_active;
    }

    public function togglePosterActive()
    {
        $this->poster_active = !$this->poster_active;
    }

    public function toggleTitleActive()
    {
        $this->title_active = !$this->title_active;
    }

    public function toggleContactActive()
    {
        $this->contact_active = !$this->contact_active;
    }

    public function toggleShopActive()
    {
        $this->shop_active = !$this->shop_active;
    }

    public function mount($id)
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->eventId = $id;
        $this->viewEvent($this->eventId);

        $this->dispatch('setEditEventImage', $this->event_image);
    }

    public function render()
    {
        $this->authorize('update', Event::class);
        return view('livewire.event-edit')
            ->title('Edit Event | ' . $this->site_name)
            ->layoutData([
                'thumbnail' => $this->thumbnail,
                'app_name' => $this->app_name,
            ]);
    }
}
