<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\LeadCaptureForm;
use App\Models\User;
use App\Models\SubCard;
use App\Models\UserSettings;
use App\Settings\CardDefaultsSetting;
use Illuminate\Support\Facades\Mail;
use App\Mail\OrderMail;

class FormPage extends Component
{
    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    // form attributes
    public $form;
    public $slug;

    public $card_title;
    public $page_title;

    public $username;

    public $user;
    public $fullname;
    public $favicon;

    //profile layout
    public $profile_layout;

    public $checkMain;
    public $checkSub;
    public bool $mainCard = false;
    public $page_id;
    public $user_id;

    public $color_settings_profile_pic_border;
    public $color_settings_custom_button;
    public $color_settings_custom_button_text;
    public $color_settings_card_light;
    public $color_settings_card_dark;
    public $color_settings_text_light;
    public $color_settings_text_dark;
    public $color_settings_link;

    public $button_style;
    public $is_active;
    public $condition;

    public $firstname, $lastname, $phone,  $referrer, $email;

    public function formSubmit()
    {
        $validated = $this->validate(
            [
                'firstname'         => 'required|string|max:50',
                'lastname'          => 'required|string|max:50',
                'phone'             => 'required|regex:/^[\d\s\-\+\(\)]+$/|max:20',
                'email'             => 'required|email',
                'referrer'          => 'nullable|string|max:500',
            ],
            [
                'phone.regex' => 'Please enter a valid phone number.',
            ]
        );

        try {
            $form = new \App\Models\FormSubmission();

            // Assign validated fields
            $form->firstname    = $validated['firstname'];
            $form->lastname     = $validated['lastname'];
            $form->phone        = $validated['phone'];
            $form->email        = $validated['email'];
            $form->referrer     = $validated['referrer'];
            $form->form_id      = $this->form->id;

            $form->fill($this->condition);

            // Save now
            $form->save();

            $message_to_buyer = "Hey {$validated['firstname']} {$validated['lastname']}!<br><br>"
                . $this->form->email_received . "<br><br>"
                . "<a href='{$this->form->link_attached}' target='_blank'>{$this->form->link_attached}/</a>";

            Mail::to($validated['email'])
                ->queue(new OrderMail($message_to_buyer, $this->form->title . ' | Submission Received!', $this->form->sender_email));

            session()->flash('success', 'Form submitted. Check your email (' . $form->email . ') for the attached link.');
        } catch (\Exception $e) {
            session()->flash('error', $e->getMessage());
        }

        return $this->redirect($this->site_url . $this->username, navigate: true);
    }

    public function displayForm($username, $slug)
    {
        // get user creds
        $this->username = $username;
        $this->slug = $slug;

        if ($this->mainCard == true) {
            $this->user = User::where('username', $this->username)->where('is_active', true)->first();
        } else {
            $this->user = SubCard::join('users', 'subcards.user_id', '=', 'users.id')
                ->where('subcards.username', $this->username)
                ->first();
        }

        if (!isset($this->username)) {
            abort(404, 'Form not found.');
        }

        // If user exist
        if ($this->user) {

            //if user exist, get events
            $this->form = LeadCaptureForm::where('is_active', true)
                ->where($this->condition)
                ->where('slug', $this->slug)
                ->first();

            // Card title
            $this->card_title = UserSettings::where('setting_key', 'card_custom_title')
                ->where($this->condition)
                ->pluck('setting_value')
                ->first();

            // Page title
            $this->page_title = UserSettings::where('setting_key', 'custom_page_title')
                ->where($this->condition)
                ->pluck('setting_value')
                ->first();

            if (!$this->card_title) {
                $this->card_title = $this->fullname;
            }

            if (!$this->page_title) {
                $this->page_title = $this->card_title;
            }

            if (!$this->form) {
                // $this->event = 0;
                abort(404, 'Form not found.');
                $this->card_title = "Form not found";
            }
        } else {
            abort(404, 'Form not found.');
            $this->card_title = "Form not found";
        }
    }

    public function __construct()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->site_url = $default->site_url;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;
    }

    public function mount($username, $slug)
    {
        //used to determine where to check if the username exist
        $this->checkMain = User::where('username', $this->username)->count();
        $this->checkSub = SubCard::where('username', $this->username)->count();

        if ($this->checkMain > 0) {
            $this->mainCard = true;
            $this->user_id = User::where('username', $username)->where('is_active', true)->pluck('id')->first();
            $this->condition = ['user_id' => $this->user_id, 'page_id' => NULL];
        }

        if ($this->checkSub > 0) {
            $this->page_id = SubCard::where('username', $this->username)
                ->pluck('id')
                ->first();
            $this->mainCard = false;
            $this->condition = ['page_id' => $this->page_id];
        }

        $this->color_settings_custom_button = UserSettings::where('setting_key', 'color_settings_custom_button')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_custom_button_text = UserSettings::where('setting_key', 'color_settings_custom_button_text')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_profile_pic_border = UserSettings::where('setting_key', 'color_settings_profile_pic_border')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_card_light = UserSettings::where('setting_key', 'color_settings_card_light')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_card_dark = UserSettings::where('setting_key', 'color_settings_card_dark')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        // Text Color Settings
        $this->color_settings_text_light = UserSettings::where('setting_key', 'color_settings_text_light')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_text_dark = UserSettings::where('setting_key', 'color_settings_text_dark')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_link = UserSettings::where('setting_key', 'color_settings_link')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        //profile layout
        $this->profile_layout = UserSettings::where('setting_key', 'profile_layout')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->button_style = UserSettings::where('setting_key', 'button_style')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->color_settings_custom_button_text = $this->color_settings_custom_button_text ?: '#000';
        $this->color_settings_custom_button = $this->color_settings_custom_button ?: '#facc15';
        $this->color_settings_profile_pic_border = $this->color_settings_profile_pic_border ?: '#ca8a04';
        $this->color_settings_card_light = $this->color_settings_card_light ?: '#fff';
        $this->color_settings_card_dark = $this->color_settings_card_dark ?: '#18181b';
        $this->color_settings_text_light = $this->color_settings_text_light ?: '#1e293b';
        $this->color_settings_text_dark = $this->color_settings_text_dark ?: '#cbd5e1';
        $this->color_settings_link = $this->color_settings_link ?: '#ca8a04';
        $this->profile_layout = $this->profile_layout ?? 'default';
        $this->button_style = $this->button_style ?: 'rounded';

        $this->displayForm($username, $slug);

        if ($this->form->image) {
            $this->favicon = $this->form->image;
        } else if ($this->user->thumbnail) {
            $this->favicon = $this->user->thumbnail;
        } else {
            $this->favicon = $this->user->profile_pic;
        }
    }

    public function render()
    {
        // dd($this->favicon);
        // $event_image = $this->event->event_poster ?? $this->favicon;
        return view('livewire.form-page')->title($this->form->title . ' | ' . $this->page_title ?: $this->card_title)->layoutData([
            'favicon' => $this->favicon,
            'username' => $this->username,
            'site_url' => $this->site_url,
            'thumbnail' => $this->favicon,
            'title' => $this->form->title . ' | ' . $this->page_title ?: $this->card_title,
            'color_settings_card_light' => $this->color_settings_card_light,
            'color_settings_card_dark' => $this->color_settings_card_dark,
            'color_settings_text_light' => $this->color_settings_text_light,
            'color_settings_text_dark' => $this->color_settings_text_dark,
            'color_settings_profile_pic_border' => $this->color_settings_profile_pic_border,
            'app_name' => $this->app_name,
        ]);
    }
}
