<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Image;
use App\Models\SubCard;
use App\Models\User;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use App\Settings\CardDefaultsSetting;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use Livewire\Attributes\On;
use Illuminate\Support\Facades\Storage;

class ImageEdit extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $image, $link;

    public $image_id, $new_image, $old_image, $image_path;
    public $condition, $checkifactive, $page_id;

    #[On('galleryImage')]
    public function setImage($newImageUrl)
    {
        $this->new_image = new TemporaryUploadedFile($newImageUrl, config('filesystems.default'));
    }

    public function updateImage()
    {
        $validated = $this->validate([
            'link' => 'nullable|url',
        ]);

        if ($this->new_image) {
            // Delete the old image if it exists
            if ($this->old_image && Storage::disk('public')->exists($this->old_image)) {
                Storage::disk('public')->delete($this->old_image);
            }

            // Store the new image
            $validated['image_path'] = $this->new_image->storePublicly('images', 'public');

            // Delete temp file if still there (optional, Laravel usually cleans this up)
            $tempFilePath = $this->new_image->getRealPath();
            if (file_exists($tempFilePath)) {
                unlink($tempFilePath);
            }
        }

        // Update the user's image
        $update = Image::where('id', $this->image_id)->update($validated);

        if ($update) {
            session()->flash('success', 'Image successfully updated!');
        } else {
            session()->flash('error', 'Unauthorized action.');
        }
        return $this->redirect('/dashboard/images', navigate: true);
    }

    public function viewImage($id)
    {
        $this->image = Image::where($this->condition)
            ->where('id', $id)
            ->select('id', 'link', 'image_path')
            ->first();

        if (!$this->image) {
            abort(403, 'Access denied');
        }
    }

    public function mount($id)
    {
        $this->checkifactive = User::where('id', auth()->user()->id)
            ->where('active_card', true)->count();

        if ($this->checkifactive < 1) {
            $this->page_id = SubCard::where('user_id', auth()->user()->id)
                ->where('active_card', true)
                ->pluck('id')
                ->first();
            $this->condition = ['page_id' => $this->page_id];
        } else {
            $this->condition = ['user_id' => auth()->user()->id, 'page_id' => NULL];
        }

        $this->image_id = $id;
        $this->viewImage($this->image_id);

        $this->old_image = $this->image->image_path;
        $this->link = $this->image->link;

        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->dispatch('setEditImage', $this->old_image);
    }

    public function render()
    {
        $this->authorize('update', Image::class);
        return view('livewire.image-edit')->title('Update Image | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
