<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\SubCard;
use App\Models\User;
use App\Settings\CardDefaultsSetting;
use App\Models\PushSubscription;

class NotificationSettingsPage extends Component
{
    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    //used to check user type to display proper settings
    public $user_type;

    public $page_id; //subcard id, used in query
    public $checkifactive;

    //used for checking
    public $query;

    public $user;

    public bool $notificationsEnabled = false;

    public function sendTestNotification()
    {
        $userId = auth()->id();
        if (!$userId) return;

        // Get all push subscriptions for the user
        $subscriptions = PushSubscription::where('user_id', $userId)->get();
        // dd($subscriptions);
        if ($subscriptions->isEmpty()) {
            $this->dispatch('alert', ['type' => 'error', 'message' => 'No push subscription found. Please enable notifications first.']);
            return;
        }

        foreach ($subscriptions as $sub) {
            // Prepare payload
            $payload = json_encode([
                'title' => 'Test Notification',
                'body' => 'This is a test notification from your app!',
                'icon' => asset('icon-192.png'),
                'url' => url()->current(),
            ]);

            try {
                // Send push via Web Push protocol
                \Minishlink\WebPush\WebPush::create()->sendNotification(
                    $sub->endpoint,
                    $payload,
                    $sub->keys['p256dh'] ?? null,
                    $sub->keys['auth'] ?? null
                );
            } catch (\Exception $e) {
                info('Push notification failed: ' . $e->getMessage());
            }
        }

        // $this->dispatchBrowserEvent('alert', ['type' => 'success', 'message' => '✅ Test notification sent!']);
        $this->dispatch('alert', ['type' => 'success', 'message' => '✅ Test notification sent!']);
    }

    public function toggleNotifications()
    {
        $this->notificationsEnabled = ! $this->notificationsEnabled;
    }

    public function saveSubscription($subscription)
    {
        $userId = auth()->id();
        if (!$userId) return;

        PushSubscription::updateOrCreate(
            ['endpoint' => $subscription['endpoint']],
            [
                'user_id' => $userId,
                'keys' => $subscription['keys'],
            ]
        );

        $this->notificationsEnabled = true;
        $this->dispatch('push-saved');
    }

    public function deleteSubscription($endpoint)
    {
        PushSubscription::where('endpoint', $endpoint)->delete();
        $this->notificationsEnabled = false;
        $this->dispatch('push-deleted');
    }

    public function mount()
    {

        $userId = auth()->id();
        if (!$userId) return;

        // Check if user already has a subscription
        $this->notificationsEnabled = PushSubscription::where('user_id', $userId)->exists();

        $this->user = auth()->user();
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->checkifactive = User::where('id', $this->user->id)->where('active_card', true)->count();

        if ($this->checkifactive > 0) {
            $this->query = ['user_id' => $this->user->id, 'page_id' => NULL];
        } else {
            $this->page_id = SubCard::where('user_id', $this->user->id)
                ->where('active_card', true)
                ->pluck('id')
                ->first();

            $this->query = ['page_id' => $this->page_id];
        }

        $this->user_type = $this->user->user_type;
    }

    public function render()
    {
        return view('livewire.notification-settings-page')->title('Notifications | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
