<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\User;
use App\Models\Referral;
use App\Models\SubCard;
use App\Models\TrustedNetwork;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Auth\Events\Registered;
use Illuminate\Support\Facades\Auth;
use App\Models\UserSettings;
use Illuminate\Support\Facades\Http;

class RegisterNew extends Component
{
    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $logo;
    public $site_button_color, $site_button_text_color;
    public $mockup;
    public $affiliate_functionality;

    public $default_ads_enabled;
    public $ads_heading;
    public $ads_description;
    public $ad_items;

    public $username, $email, $password, $password_confirmation, $country, $phone;
    public $referrer_code, $referrer, $referrer_id, $referrer_type;
    public $firstname;
    public $lastname;
    public $job_title;
    public $profile_pic;
    public $state;
    public $trusted_network_username;

    public $availableStates = [];
    public array $ads = []; // Each item has user_link

    #[On('profilePic')]
    public function setProfilePic($profileImageUrl)
    {
        $this->profile_pic = new TemporaryUploadedFile($profileImageUrl, config('filesystems.default'));
    }

    public function updatedCountry($value)
    {
        $this->state = null;
        $this->loadStates($value);
    }

    public function loadStates($country)
    {
        if (!$country) {
            $this->availableStates = [];
            return;
        }

        $response = Http::post('https://countriesnow.space/api/v0.1/countries/states', [
            'country' => $country,
        ]);

        $states = data_get($response->json(), 'data.states', []);

        $this->availableStates = collect($states)
            ->pluck('name', 'name')
            ->toArray();
    }

    public function registerUser()
    {
        $validated = $this->validate(
            [
                'username' => [
                    'required',
                    'string',
                    'max:50',
                    'regex:/^[a-zA-Z0-9_.]+$/',
                    'unique:users',
                    'unique:subcards',
                ],
                'profile_pic'       => 'nullable|image|mimes:jpeg,jpg,png,webp|max:5120',
                'firstname'         => 'required|string|max:50',
                'lastname'          => 'required|string|max:50',
                'email'             => 'required|email:filter|unique:users|unique:subcards',
                'password'          => 'required|string|min:8|confirmed',
                'country'           => 'required|string',
                'state'             => 'nullable|string',
                'phone'             => 'required|regex:/^[\d\s\-\+\(\)]+$/|max:20',
                'job_title'         => 'required|string|max:200',

            ],
            //Custom error messages
            [
                'password.confirmed' => "Password didn't match.",
                'username.regex' => "Username may only contain letters, numbers, and underscores.",
                'phone.regex' => 'Please enter a valid phone number.',
            ]
        );

        // if refferrer valid, save referral info
        if (!is_null($this->referrer_id)) {
            $validated['referrer_id'] = $this->referrer_id;
            $validated['referrer_type'] = $this->referrer_type;
        }

        // Initial user type is FREE
        // $validated['user_type'] = 'Free';

        if ($this->profile_pic) {
            $validated['profile_pic'] =  $this->profile_pic->storePublicly('profile_pic', 'public');

            // Get the path of the temporary file (before it's stored in the public directory)
            $tempFilePath = $this->profile_pic->getRealPath();

            if (file_exists($tempFilePath)) {
                unlink($tempFilePath);  // Delete the temporary file explicitly
            }
        }

        // Submit to database
        $create = User::create($validated);

        // Mail::to($create->email)->send(new ConfirmRegistrationMail($create));

        if ($create) {
            if (!is_null($this->referrer_id)) {
                Referral::create([
                    'referrer_id_main' => $this->referrer_type === 'main' ? $this->referrer_id : null,
                    'referrer_id_sub' => $this->referrer_type === 'sub' ? $this->referrer_id : null,
                    'referred_id' => $create->id,
                    'referrer_type' => $this->referrer_type,
                ]);
            }

            // Save the user-created ad links
            foreach ($this->ads as $index => $ad) {
                if (!empty($ad['user_link'])) {
                    UserSettings::create([
                        'user_id' => $create->id,
                        'setting_key'   => "ads.$index.user_link",
                        'setting_value' => $ad['user_link'],
                        'page_id' => null,
                    ]);
                }
            }

            session()->flash('success', 'Almost done! Check your email for a confirmation link to activate your account.');

            Auth::login($create);
            event(new Registered($create));

            // if ($this->trusted_network_username) {
            //     $check = User::where('username', $this->trusted_network_username)->first()
            //         ?? SubCard::where('username', $this->trusted_network_username)->first();

            //     // ❌ Stop if not found
            //     if (! $check) {
            //         // dd('not found');
            //         return;
            //     }

            //     // ❌ If it's a User and NOT Platinum, stop
            //     if ($check instanceof User && $check->user_type !== 'Platinum') {
            //         // dd('not Platinum');
            //         return;
            //     }

            //     $owner_type = $check instanceof User ? 'main' : 'sub';
            //     $owner_id   = $check->id;

            //     // Prevent duplicates
            //     $exists = TrustedNetwork::where('member_id', $create->id)
            //         ->where('card_type', 'main')
            //         ->where(function ($q) use ($owner_type, $owner_id) {
            //             if ($owner_type == 'main') {
            //                 $q->where('user_id', $owner_id);
            //             } else {
            //                 $q->where('page_id', $owner_id);
            //             }
            //         })
            //         ->exists();

            //     if (!$exists) {
            //         // Create record
            //         TrustedNetwork::create([
            //             'user_id'     => $owner_type == 'main' ? $owner_id : null,
            //             'page_id'     => $owner_type == 'sub' ? $owner_id : null,
            //             'member_id'   => $create->id,
            //             'card_type'   => 'main', // main or sub from checkbox
            //             'type'        => 'network',
            //             'is_accepted' => false,
            //         ]);
            //     }
            // }
            if ($this->trusted_network_username) {

                $check = User::where('username', $this->trusted_network_username)->first()
                    ?? SubCard::where('username', $this->trusted_network_username)->first();

                if ($check) {

                    if (!($check instanceof User && $check->user_type !== 'Platinum')) {

                        $owner_type = $check instanceof User ? 'main' : 'sub';
                        $owner_id   = $check->id;

                        $exists = TrustedNetwork::where('member_id', $create->id)
                            ->where('card_type', 'main')
                            ->where(function ($q) use ($owner_type, $owner_id) {
                                $owner_type === 'main'
                                    ? $q->where('user_id', $owner_id)
                                    : $q->where('page_id', $owner_id);
                            })
                            ->exists();

                        if (! $exists) {
                            TrustedNetwork::create([
                                'user_id'     => $owner_type === 'main' ? $owner_id : null,
                                'page_id'     => $owner_type === 'sub' ? $owner_id : null,
                                'member_id'   => $create->id,
                                'card_type'   => 'main',
                                'type'        => 'network',
                                'is_accepted' => false,
                            ]);
                        }
                    }
                }
            }

            return $this->redirect('/email/verify', navigate: true);
        } else {
            session()->flash('error', 'Error creating account.');
            return $this->redirect('/register', navigate: true);
        }
    }

    public function mount()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->site_url = $default->site_url;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;
        $this->logo = $default->logo;
        $this->mockup = $default->mockup;
        $this->site_button_color = $default->site_button_color;
        $this->site_button_text_color = $default->site_button_text_color;
        $this->affiliate_functionality = $default->affiliate_functionality;
        $this->referrer_code = request()->query('ref');
        $this->trusted_network_username = request()->query('trusted-network');

        $this->default_ads_enabled = $default->default_ads_enabled;
        $this->ads_heading = $default->ads_heading;
        $this->ads_description = $default->ads_description;
        $this->ad_items = $default->ad_items;

        // Build ads array structure only for active ads
        $this->ads = []; // reset
        foreach ($this->ad_items as $item) {
            if ($item['is_active'] ?? false) {
                $this->ads[] = [
                    'user_link' => '',
                ];
            }
        }

        if ($this->affiliate_functionality === false && $this->referrer_code) {
            return redirect()->to('/register');
        }

        if ($this->affiliate_functionality && $this->referrer_code) {
            // if ($this->referrer_code) {
            $checkifactive = User::where('username', $this->referrer_code)->count();
            //if main card
            if ($checkifactive > 0) {
                $this->referrer = User::where('username', $this->referrer_code)
                    ->whereIn('user_type', ['Influencer', 'Gold', 'Platinum'])
                    //  ->whereIn('user_type', ['Influencer', 'Sponsored', 'Gold', 'Platinum'])
                    ->first(['firstname', 'lastname', 'id']);
                $this->referrer_type = 'main';
            } else {
                // if subcard
                $this->referrer = SubCard::where('username', $this->referrer_code)
                    ->first(['firstname', 'lastname', 'id']);
                $this->referrer_type = 'sub';
            }
        } else {
            $this->referrer = NULL;
        }

        if (!is_null($this->referrer)) {
            $this->referrer_id = $this->referrer->id;
            $this->referrer = $this->referrer->firstname . ' ' . $this->referrer->lastname;
        }
    }

    public function render()
    {
        return view('livewire.register-new')
            ->title('Register | ' . $this->site_name)
            ->layoutData([
                'thumbnail' => $this->thumbnail,
                'app_name' => $this->app_name,
            ]);
    }
}
