<?php

namespace App\Livewire;

use App\Models\Button;
use App\Models\SubCard;
use Livewire\Component;
use App\Models\User;
use App\Models\Referral;
use App\Models\UserSettings;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;

class RegisterPage extends Component
{
    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $affiliate_functionality;

    public $username, $firstname, $lastname, $phone, $gender, $profile_pic, $job_title, $bio, $company, $country, $email, $password, $password_confirmation;

    public $referrer_code, $referrer, $referrer_id, $referrer_type;

    #[On('profilePic')]
    public function setProfilePic($profileImageUrl)
    {
        $this->profile_pic = new TemporaryUploadedFile($profileImageUrl, config('filesystems.default'));
    }

    public function registerUser()
    {
        $validated = $this->validate(
            [
                // 'username' => 'required|string|max:50|unique:users|unique:subcards',
                'username' => [
                    'required',
                    'string',
                    'max:50',
                    'regex:/^[a-zA-Z0-9_]+$/',
                    'unique:users',
                    'unique:subcards',
                ],
                'firstname' => 'required|string|max:50',
                'lastname' => 'required|string|max:50',
                'phone' => 'required|regex:/^\d+$/|max:20',
                'gender' => 'required|in:Male,Female',
                'profile_pic' => 'nullable|image|mimes:jpeg,jpg,png,webp|max:5120',
                // 'cover_pic' => 'nullable|image|mimes:jpeg,jpg,png,webp|max:5120',
                // 'company_pic' => 'nullable|image|mimes:jpeg,jpg,png,webp|max:2048',
                'job_title' => 'required|string|max:200',
                'bio' => 'nullable|string|max:200',
                'company' => 'nullable|string|max:255',
                // 'company_website' => 'nullable|url',
                // 'language' => 'nullable|string',
                // 'address' => 'nullable|string',
                'country' => 'required|string',
                // 'user_type' => 'required|in:Basic,Gold,Platinum',
                'email' => 'required|email:filter|unique:users|unique:subcards',
                // 'password_confirmation' => 'required|string',
                'password' => 'required|string|min:8|confirmed',
            ],
            //Custom error messages
            [
                'password.confirmed' => "Password didn't match.",
                'username.regex' => "Username may only contain letters, numbers, and underscores.",
                'phone.regex' => 'The phone may only contain numbers.',
            ]
        );

        // if refferrer valid, save referral info
        if (!is_null($this->referrer_id)) {
            $validated['referrer_id'] = $this->referrer_id;
            $validated['referrer_type'] = $this->referrer_type;
            // dd($validated['referrer_id'], $validated['referrer_type']);
        }

        // Initial user type is FREE
        $validated['user_type'] = 'Free';

        if ($this->profile_pic) {
            $validated['profile_pic'] =  $this->profile_pic->storePublicly('profile_pic', 'public');

            // Get the path of the temporary file (before it's stored in the public directory)
            $tempFilePath = $this->profile_pic->getRealPath();

            if (file_exists($tempFilePath)) {
                unlink($tempFilePath);  // Delete the temporary file explicitly
            }
        }

        // Submit to database
        $create = User::create($validated);

        if ($create) {

            // dd($user->referrer_id, $user->referrer_type);
            if (!is_null($this->referrer_id)) {
                Referral::create([
                    'referrer_id_main' => $this->referrer_type === 'main' ? $this->referrer_id : null,
                    'referrer_id_sub' => $this->referrer_type === 'sub' ? $this->referrer_id : null,
                    'referred_id' => $create->id,
                    'referrer_type' => $this->referrer_type,
                ]);
            }

            session()->flash('success', 'User successfully registered.');
            return $this->redirect('/login', navigate: true);
        } else {
            session()->flash('error', 'Error creating account.');
            return $this->redirect('/register', navigate: true);
        }
    }

    // public function mount()
    // {
    //     $default = app(CardDefaultsSetting::class);
    //     $this->site_name = $default->site_name;
    //     $this->site_url = $default->site_url;
    //     $this->thumbnail = $default->thumbnail;
    //     // $this->affiliate_functionality = $default->affiliate_functionality;

    //     $this->referrer_code = request()->query('ref');
    //     if ($this->referrer_code) {
    //         $checkifactive = User::where('username', $this->referrer_code)->count();
    //         //if main card
    //         if ($checkifactive > 0) {
    //             $this->referrer = User::where('username', $this->referrer_code)
    //                 ->whereIn('user_type', ['Gold', 'Platinum'])
    //                 ->first(['firstname', 'lastname', 'id']);
    //             $this->referrer_type = 'main';
    //         }
    //         // else {
    //         //     // if subcard
    //         //     $this->referrer = SubCard::where('username', $this->referrer_code)
    //         //         ->first(['firstname', 'lastname', 'id']);
    //         //     $this->referrer_type = 'sub';
    //         // }
    //     } else {
    //         $this->referrer = NULL;
    //     }

    //     if (!is_null($this->referrer)) {
    //         $this->referrer_id = $this->referrer->id;
    //         $this->referrer = $this->referrer->firstname . ' ' . $this->referrer->lastname;
    //     }
    // }

    public function mount()
    {
        $default = app(CardDefaultsSetting::class);

        $this->affiliate_functionality = $default->affiliate_functionality;
        $this->referrer_code = request()->query('ref');
        // dd($this->affiliate_functionality);

        // If affiliate system is OFF and a ?ref= exists → redirect to clean /register
        if ($this->affiliate_functionality === false && $this->referrer_code) {
            return redirect()->to('/register');
        }

        // Otherwise, handle referral normally
        if ($this->affiliate_functionality && $this->referrer_code) {
            $referrer = User::where('username', $this->referrer_code)
                ->whereIn('user_type', ['Gold', 'Platinum'])
                ->first(['firstname', 'lastname', 'id']);

            if ($referrer) {
                $this->referrer_id = $referrer->id;
                $this->referrer = "{$referrer->firstname} {$referrer->lastname}";
                $this->referrer_type = 'main';
            }
        } else {
            $this->referrer = null;
        }

        $this->site_name = $default->site_name;
        $this->site_url = $default->site_url;
        $this->thumbnail = $default->thumbnail;
    }

    public function render()
    {
        dd('wth');
        return view('livewire.register-page')
            ->title('Register | ' . $this->site_name)
            ->layoutData([
                'thumbnail' => $this->thumbnail,
            ]);
    }
}
