<?php

namespace App\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use App\Settings\CardDefaultsSetting;

class ResetPassword extends Component
{
    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $site_desc;
    public $app_name;

    public $token;
    public $email;
    public $password;
    public $password_confirmation;

    protected $rules = [
        'email' => 'required|email|exists:users,email',
        'password' => 'required|min:8|confirmed',
    ];

    public function resetPassword()
    {
        $this->validate();

        $status = Password::reset(
            [
                'email' => $this->email,
                'password' => $this->password,
                'password_confirmation' => $this->password_confirmation,
                'token' => $this->token,
            ],
            function ($user, $password) {
                $user->forceFill([
                    'password' => Hash::make($password),
                ])->save();
            }
        );

        if ($status === Password::PASSWORD_RESET) {
            session()->flash('success', __($status));
            return redirect()->route('login');
        } else {
            session()->flash('error', __($status));
        }
    }

    public function mount($token)
    {
        $this->token = $token;
        $this->email = request()->query('email');
    }

    public function render()
    {
        $default = app(CardDefaultsSetting::class);

        $this->site_name = $default->site_name;
        $this->site_url = $default->site_url;
        $this->thumbnail = $default->thumbnail;
        $this->site_desc = $default->site_desc;
        $this->app_name = $default->app_name;

        return view('livewire.reset-password')->title('Reset Password ' . $this->site_name)->layoutData([
            'site_name' => $this->site_name,
            'site_url' => $this->site_url,
            'site_desc' => $this->site_desc,
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
