<?php

namespace App\Livewire;

use App\Models\SubCard;
use App\Models\User;
use Livewire\Component;
use App\Settings\CardDefaultsSetting;
use App\Models\TrustedNetwork;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;

class SearchTrustedNetwork extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $keyword = '';
    public $username;
    public $page_id; //subcard id, used in query

    //search results
    public $users = [];

    public function updatedKeyword()
    {
        $this->searchUsers();
    }

    public function searchUsers()
    {
        $query = $this->keyword;

        if ($query) {
            $isUsernameOnlySearch = str_starts_with($query, '@');
            $query = ltrim($query, '@'); // remove @ if present

            // USER QUERY
            $users = User::leftJoin('user_settings', 'users.id', '=', 'user_settings.user_id')
                ->where('users.is_admin', false)
                ->where(function ($queryBuilder) use ($query, $isUsernameOnlySearch) {
                    if ($isUsernameOnlySearch) {
                        $queryBuilder->where('users.username', 'like', "%$query%");
                    } else {
                        $queryBuilder->where('users.username', 'like', "%$query%")
                            ->orWhere('users.email', 'like', "%$query%")
                            ->orWhere('users.firstname', 'like', "%$query%")
                            ->orWhere('users.lastname', 'like', "%$query%")
                            ->orWhere('users.job_title', 'like', "%$query%")
                            ->orWhere('users.company', 'like', "%$query%")
                            ->orWhereRaw("CONCAT(users.firstname, ' ', users.lastname) LIKE ?", ["%$query%"])
                            ->orWhereRaw("CONCAT(users.lastname, ' ', users.firstname) LIKE ?", ["%$query%"]);
                    }
                })
                ->where(function ($queryBuilder) {
                    $queryBuilder->where(function ($subQuery) {
                        $subQuery->where('user_settings.setting_key', 'searchable')
                            ->where('user_settings.setting_value', true)
                            ->whereNull('user_settings.page_id');
                    })
                        ->where('users.is_active', true)
                        ->orWhereNull('user_settings.id'); // Include users without settings
                })
                ->limit(10)
                ->get()
                ->map(function ($user) {
                    return (object)[
                        'username' => $user->username,
                        'firstname' => $user->firstname,
                        'lastname' => $user->lastname,
                        'job_title' => $user->job_title,
                        'profile_pic' => $user->profile_pic,
                        'company' => $user->company,
                        'user_type' => $user->user_type,
                        'id' => $user->id,
                    ];
                });

            // SUBCARD QUERY
            $subcards = SubCard::leftJoin('users', 'subcards.user_id', '=', 'users.id')
                ->leftJoin('user_settings', 'subcards.id', '=', 'user_settings.page_id')
                ->select('subcards.*', 'users.user_type')
                ->where(function ($queryBuilder) use ($query, $isUsernameOnlySearch) {
                    // $subcards = SubCard::leftJoin('users', 'subcards.user_id', '=', 'users.id')
                    //     ->leftJoin('user_settings', 'subcards.id', '=', 'user_settings.page_id')
                    //     ->where(function ($queryBuilder) use ($query, $isUsernameOnlySearch) {
                    if ($isUsernameOnlySearch) {
                        $queryBuilder->where('subcards.username', 'like', "%$query%");
                    } else {
                        $queryBuilder->where('subcards.username', 'like', "%$query%")
                            ->orWhere('subcards.email', 'like', "%$query%")
                            ->orWhere('subcards.firstname', 'like', "%$query%")
                            ->orWhere('subcards.lastname', 'like', "%$query%")
                            ->orWhere('subcards.job_title', 'like', "%$query%")
                            ->orWhere('subcards.company', 'like', "%$query%")
                            ->orWhereRaw("CONCAT(subcards.firstname, ' ', subcards.lastname) LIKE ?", ["%$query%"])
                            ->orWhereRaw("CONCAT(subcards.lastname, ' ', subcards.firstname) LIKE ?", ["%$query%"]);
                    }
                })
                ->where(function ($queryBuilder) {
                    $queryBuilder->where(function ($subQuery) {
                        $subQuery->where('user_settings.setting_key', 'searchable')
                            ->where('user_settings.setting_value', true);
                    })
                        ->where('subcards.is_active', true)
                        ->orWhere('user_settings.id'); // Include users without settings
                })
                ->limit(10)
                ->get()
                ->map(function ($subcard) {
                    return (object)[
                        'username' => $subcard->username,
                        'firstname' => $subcard->firstname,
                        'lastname' => $subcard->lastname,
                        'job_title' => $subcard->job_title,
                        'profile_pic' => $subcard->profile_pic,
                        'company' => $subcard->company,
                        'user_type' => $subcard->user_type,
                        'id' => $subcard->id,
                    ];
                });

            // Merge, deduplicate, and sort
            $merged = array_merge($users->toArray(), $subcards->toArray());
            $merged = collect($merged)->unique('username')->values()->all();

            // usort($merged, function ($a, $b) {
            //     return strcmp($a->firstname, $b->firstname);
            // });

            $keyword = strtolower($this->keyword);

            usort($merged, function ($a, $b) use ($keyword) {
                $aScore = 0;
                $bScore = 0;

                // Convert everything to lowercase
                $aFirst = strtolower($a->firstname);
                $bFirst = strtolower($b->firstname);

                // --- FIRSTNAME PRIORITY ---

                // Exact firstname match = highest priority
                if ($aFirst === $keyword) $aScore += 1000;
                if ($bFirst === $keyword) $bScore += 1000;

                // Firstname starts with search
                if (str_starts_with($aFirst, $keyword)) $aScore += 500;
                if (str_starts_with($bFirst, $keyword)) $bScore += 500;

                // Firstname contains search
                if (str_contains($aFirst, $keyword)) $aScore += 250;
                if (str_contains($bFirst, $keyword)) $bScore += 250;

                // OPTIONAL: fallback sorting if equal score
                if ($aScore === $bScore) {
                    return strcmp($a->firstname, $b->firstname);
                }

                return $bScore <=> $aScore; // Descending: highest score first
            });

            $this->users = collect($merged);
        } else {
            $this->users = [];
        }
    }

    public function mount()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;
        $this->username = User::where('id', auth()->user()->id)->pluck('username')->first();
    }

    public function render()
    {
        $this->authorize('view', TrustedNetwork::class);
        return view(
            'livewire.search-trusted-network',
            [
                'users' => $this->users,
            ]
        )
            ->title('Search Member | ' . $this->site_name)
            ->layoutData([
                'thumbnail' => $this->thumbnail,
                'app_name' => $this->app_name,
            ]);
    }
}
