<?php

namespace App\Livewire;

use App\Models\SendCard as send_card_form;
use Livewire\Component;
use App\Settings\CardDefaultsSetting;
use Livewire\Attributes\On;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;

class SendCard extends Component
{
    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $card_title;
    public $page_title;

    public $user;
    public $fullname;
    public $favicon;



    public ?string $footer_text;
    public ?string $footer_link;

    //footer by user
    public ?string $default_footer_text;
    public ?string $default_footer_link;

    public $site_button_color;
    public $site_button_text_color;

    public $condition;

    public $firstname, $lastname, $phone, $profile_pic, $email, $digital_card, $physical_card, $physical_card_back;

    #[On('profilePic')]
    public function setProfilePic($profileImageUrl)
    {
        $this->profile_pic = new TemporaryUploadedFile($profileImageUrl, config('filesystems.default'));
    }

    #[On('physicalCard')]
    public function setPhysicalCard($physicalCardImageUrl)
    {
        $this->physical_card = new TemporaryUploadedFile($physicalCardImageUrl, config('filesystems.default'));
    }

    #[On('physicalCardBack')]
    public function setPhysicalCardBack($physicalCardBackImageUrl)
    {
        $this->physical_card_back = new TemporaryUploadedFile($physicalCardBackImageUrl, config('filesystems.default'));
    }

    public function mount()
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->site_url = $default->site_url;
        $this->app_name = $default->app_name;
        $this->thumbnail = $default->thumbnail;
        $this->default_footer_text = $default->footer_text;
        $this->default_footer_link = $default->footer_link;

        $this->site_button_text_color = $default->site_button_text_color;
        $this->site_button_color = $default->site_button_color;

        $this->site_button_text_color = $this->site_button_text_color ?: '#000';
        $this->site_button_color = $this->site_button_color ?: '#facc15';
    }

    public function sendCardForm()
    {
        $validated = $this->validate(
            [
                'firstname'                 => 'required|string|max:50',
                'lastname'                  => 'required|string|max:50',
                'phone'                     => 'required|regex:/^[\d\s\-\+\(\)]+$/|max:20',
                'email'                     => 'required|email:filter',
                'profile_pic'               => 'nullable|image|mimes:jpeg,jpg,png,webp|max:5120',
                'digital_card'              => 'nullable|string|max:255',
                'physical_card'             => 'nullable|image|mimes:jpeg,jpg,png,webp|max:5120',
                'physical_card_back'        => 'nullable|image|mimes:jpeg,jpg,png,webp|max:5120',
            ],
            //Custom error messages
            [
                'phone.regex' => 'Please enter a valid phone number.',
            ]
        );

        if ($this->profile_pic) {
            $validated['profile_pic'] =  $this->profile_pic->storePublicly('profile_pic', 'public');

            // Get the path of the temporary file (before it's stored in the public directory)
            $tempFilePath = $this->profile_pic->getRealPath();

            if (file_exists($tempFilePath)) {
                unlink($tempFilePath);  // Delete the temporary file explicitly
            }
        }

        if ($this->physical_card) {
            $validated['physical_card'] =  $this->physical_card->storePublicly('physical_card', 'public');

            // Get the path of the temporary file (before it's stored in the public directory)
            $tempFilePath = $this->physical_card->getRealPath();

            if (file_exists($tempFilePath)) {
                unlink($tempFilePath);  // Delete the temporary file explicitly
            }
        }

        if ($this->physical_card_back) {
            $validated['physical_card_back'] =  $this->physical_card_back->storePublicly('physical_card', 'public');

            // Get the path of the temporary file (before it's stored in the public directory)
            $tempFilePath = $this->physical_card_back->getRealPath();

            if (file_exists($tempFilePath)) {
                unlink($tempFilePath);  // Delete the temporary file explicitly
            }
        }

        try {
            $create = send_card_form::create([
                'firstname'                 => $validated['firstname'],
                'lastname'                  => $validated['lastname'],
                'phone'                     => $validated['phone'],
                'email'                     => $validated['email'],
                'profile_pic'               => $validated['profile_pic'],
                'digital_card'              => $validated['digital_card'],
                'physical_card'             => $validated['physical_card'],
                'physical_card_back'        => $validated['physical_card_back'],
            ]);

            if ($create) {
                session()->flash('success', 'Thank you!');
            } else {
                session()->flash('error', 'Error submitting the form.');
            }
        } catch (\Exception $e) {
            session()->flash('error', $e->getMessage());
        }

        return $this->redirect('/send-card', navigate: true);
    }

    public function render()
    {
        return view('livewire.send-card')->title("Send us your Business Card | " . ($this->page_title ?: $this->card_title))->layoutData([
            'favicon' => $this->favicon,
            'thumbnail' => $this->thumbnail,
            'site_url' => $this->site_url,
            'title' => $this->page_title ?: $this->card_title,
            'site_button_text_color' => $this->site_button_text_color,
            'site_button_color' => $this->site_button_color,
            'app_name' => $this->app_name,
        ]);
    }
}
