<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\Attributes\On;
use App\Models\Service;
use Illuminate\Foundation\Auth\Access\AuthorizesRequests;
use Livewire\Features\SupportFileUploads\TemporaryUploadedFile;
use App\Settings\CardDefaultsSetting;
use Illuminate\Validation\ValidationException;

class ServiceEdit extends Component
{
    use AuthorizesRequests;

    // Site Settings
    public $site_url;
    public $site_name;
    public $thumbnail;
    public $app_name;

    public $service, $title, $description, $price, $service_image, $payment_link, $old_image, $new_service_image;
    public $serviceId;

    #[On('serviceImage')]
    public function setServiceImage($serviceImageUrl)
    {
        $this->new_service_image = new TemporaryUploadedFile($serviceImageUrl, config('filesystems.default'));
    }

    public function updateService()
    {
        try {
            $validated = $this->validate([
                'title' => 'required|string',
                'price' => 'required|numeric',
                'payment_link' => 'required|url',
                'description' => 'required|string|max:2000'
            ]);

            // check if service has image
            if ($this->new_service_image) {
                // Check if the service has already a picture and delete it
                if ($this->old_image) {
                    unlink(storage_path('app/public/' . $this->old_image));
                }

                // dd('service image doesnt exist, upload app/public/' . $this->new_service_image);
                $validated['service_image'] = $this->new_service_image->storePublicly('services', 'public');

                // Get the path of the temporary file (before it's stored in the public directory)
                $tempFilePath = $this->new_service_image->getRealPath();

                if (file_exists($tempFilePath)) {
                    unlink($tempFilePath);  // Delete the temporary file explicitly
                }
            }

            // Update the user's service
            $update = auth()->user()->services()->where('id', $this->serviceId)->update($validated);

            if ($update) {
                session()->flash('success', 'Service successfully updated!');
            } else {
                session()->flash('error', 'Unauthorized action.');
            }
            return $this->redirect('/dashboard/service', navigate: true);
        } catch (ValidationException $e) {
            // stop spinner on validation errors
            $this->dispatch('set-submitting', ['value' => false]);
            throw $e;
        }
    }

    public function viewService($id)
    {
        $this->service = Service::where('user_id', auth()->user()->id)
            ->where('id', $id)
            ->select('id', 'title', 'description', 'price', 'service_image', 'payment_link')
            ->first();

        if (!$this->service) {
            abort(403, 'Access denied');
        }

        // dd($this->service);
    }

    public function mount($id)
    {
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->thumbnail = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->serviceId = $id;
        $this->viewService($this->serviceId);

        $this->title = $this->service->title;
        $this->description = $this->service->description;
        $this->price = $this->service->price;
        $this->service_image = $this->service->service_image;
        $this->old_image = $this->service->service_image;
        $this->payment_link = $this->service->payment_link;

        $this->dispatch('setEditServiceImage', $this->service_image);
    }

    public function render()
    {
        $this->authorize('update', Service::class);
        return view('livewire.service-edit')->title('Edit Service | ' . $this->site_name)->layoutData([
            'thumbnail' => $this->thumbnail,
            'app_name' => $this->app_name,
        ]);
    }
}
