<?php

namespace App\Livewire;

use Livewire\Component;
use App\Settings\CardDefaultsSetting;
use App\Models\UserSettings;
use App\Models\SubscriptionRewards;
use App\Models\User;
use App\Models\SubCard;
use App\Models\Button;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class SubscriptionCart extends Component
{
    use WithFileUploads;

    public $site_url;
    public $site_name;
    public $thumbnail;
    public $favicon;
    public $shop_thumbnail;
    public $app_name;

    public $color_settings_profile_pic_border;
    public $color_settings_custom_button;
    public $color_settings_custom_button_text;
    public $color_settings_card_light;
    public $color_settings_card_dark;
    public $color_settings_text_light;
    public $color_settings_text_dark;

    public $checkMain, $checkSub, $user, $condition, $page_id;
    public $payments;
    public $cartData;
    public $cart;
    public $item;

    public $preferred_contact = 'Text';
    public $username, $firstname, $lastname, $phone, $receipt, $special_note, $sender_email;

    public function mount($username)
    {
        $this->username = $username;
        $this->checkMain = User::where('username', $username)->count();
        $this->checkSub = SubCard::where('username', $username)->count();

        if ($this->checkMain > 0) {
            $this->user = User::where('username', $username)->where('is_active', true)->first();
            $this->condition = ['user_id' => $this->user->id, 'page_id' => NULL];
        }

        if ($this->checkSub > 0) {
            $this->page_id = SubCard::where('username', $username)->value('id');
            $this->user = SubCard::where('username', $username)->where('is_active', true)->first();
            $this->condition = ['page_id' => $this->page_id];
        }

        // Load default and user settings...
        $default = app(CardDefaultsSetting::class);
        $this->site_name = $default->site_name;
        $this->site_url = $default->site_url;
        $this->thumbnail = $default->thumbnail;
        $this->favicon = $default->thumbnail;
        $this->app_name = $default->app_name;

        $this->color_settings_custom_button = UserSettings::where('setting_key', 'color_settings_custom_button')
            ->where($this->condition)
            ->value('setting_value') ?? '#facc15';

        $this->color_settings_custom_button_text = UserSettings::where('setting_key', 'color_settings_custom_button_text')
            ->where($this->condition)
            ->value('setting_value') ?? '#000';

        $this->color_settings_profile_pic_border = UserSettings::where('setting_key', 'color_settings_profile_pic_border')
            ->where($this->condition)
            ->value('setting_value') ?? '#ca8a04';

        $this->color_settings_card_light = UserSettings::where('setting_key', 'color_settings_card_light')
            ->where($this->condition)
            ->value('setting_value') ?? '#fff';

        $this->color_settings_card_dark = UserSettings::where('setting_key', 'color_settings_card_dark')
            ->where($this->condition)
            ->value('setting_value') ?? '#18181b';

        $this->color_settings_text_light = UserSettings::where('setting_key', 'color_settings_text_light')
            ->where($this->condition)
            ->value('setting_value') ?? '#1e293b';

        $this->color_settings_text_dark = UserSettings::where('setting_key', 'color_settings_text_dark')
            ->where($this->condition)
            ->value('setting_value') ?? '#cbd5e1';

        // Shop thumbnail
        $this->shop_thumbnail = UserSettings::where('setting_key', 'shop_thumbnail')
            ->where($this->condition)
            ->pluck('setting_value')
            ->first();

        $this->payments = Button::where('is_active', true)
            ->where($this->condition)
            ->where('view', 'payments')
            ->orderBy('order', 'asc')
            ->get();

        if ($this->payments->isEmpty()) {
            $this->payments = 0;
        }

        if ($this->shop_thumbnail) {
            // Use shop thumbnail if available
            $this->thumbnail = $this->shop_thumbnail;
            $this->favicon = $this->shop_thumbnail;
        } else if ($this->user->thumbnail) {
            // User has a custom thumbnail
            $this->thumbnail = $this->user->thumbnail;
            $this->favicon = $this->user->thumbnail;
        } else if ($this->user->profile_pic) {
            // Use user's profile picture
            $this->thumbnail = $this->user->profile_pic;
            $this->favicon = $this->user->profile_pic;
        } else {
            // Default site thumbnail fallback
            $this->thumbnail = $this->thumbnail ?? null;
            $this->favicon = $this->thumbnail;
        }
    }

    public function checkoutForm()
    {
        // $cart = json_decode($this->cartData);
        $this->cart = json_decode($this->cartData, true);
        $this->item = $this->cart[0] ?? [];

        $validated = $this->validate([
            'firstname' => 'required|string|max:50',
            'lastname' => 'required|string|max:50',
            'special_note' => 'nullable|string|max:100',
            'sender_email' => 'required|email',
            'phone' => 'required|regex:/^[\d\s\-\+\(\)]+$/|max:20',
            'preferred_contact' => 'required|in:Text,WhatsApp,Email',
            'receipt' => 'required|file|mimes:jpg,jpeg,png,pdf|max:2048',
        ], [
            'firstname.required' => 'This field is required.',
            'lastname.required' => 'This field is required.',
            'phone.regex' => 'Please enter a valid phone number.',
            'receipt.max' => 'Receipt must be less than 2MB.',
        ]);

        $receipt_path = $this->receipt->store('receipts', 'public');
        $receipt_url = Storage::url($receipt_path);

        $noteHtml = !empty($validated['special_note'])
            ? "<br><strong>Special Note:</strong> {$validated['special_note']}<br><br>"
            : '';

        $noteText = !empty($validated['special_note'])
            ? "%0A%0ASpecial Note: {$validated['special_note']}"
            : '';

        $link = route('subscription.show', [
            'username' => $this->item['username'],
            'id' => $this->item['id']
        ]);

        $body = "Hey {$this->user->firstname}, this is {$validated['firstname']} {$validated['lastname']}.<br><br>"
            . "I’d like to subscribe to {$this->item['name']} - {$this->item['tier_title']} (<a href='{$link}' target='_blank'>{$link}</a>).<br>"
            . "<strong>Total: $" . number_format($this->item['price'], 2) . "</strong><br><br>"
            . $noteHtml . "<br><br>"
            . "I have attached the payment receipt.<br><br>"
            . "Download Receipt: <a href='" . url($receipt_url) . "' target='_blank'>" . url($receipt_url) . "</a><br><br>"
            . "You can reach me at: <a href='mailto:{$validated['sender_email']}' target='_blank'>{$validated['sender_email']}</a>.<br><br>"
            . "It’s great connecting with you!<br><br>";

        $message = "Hey {$this->user->firstname}, this is {$validated['firstname']} {$validated['lastname']}.%0A%0A"
            . "I’d like to subscribe to {$this->item['name']} - {$this->item['tier_title']} ({$link}).%0A"
            . "Total: $" . number_format($this->item['price'], 2)
            . "%0A"
            . $noteText
            . "%0A"
            . "I have attached the payment receipt.%0A%0A"
            . "Download Receipt: " . url($receipt_url)
            . "%0A%0AYou can reach me at: {$validated['sender_email']}.%0A%0A"
            . "It’s great connecting with you!%0A%0A";
        // . "{$this->site_url}{$this->user->username}/";

        $smsUrl = "sms:{$this->user->phone}?&body={$message}";
        $whatsappUrl = "https://wa.me/{$this->user->phone}?text={$message}";

        try {
            //save to database
            SubscriptionRewards::create(
                array_merge(
                    $this->condition,
                    [
                        'product_id'    => $this->item['id'],
                        'coin'          => $this->item['tier_coin'],
                        'price'         => $this->item['price'],
                        'receipt'       => $receipt_path,
                        'firstname'     => $validated['firstname'],
                        'lastname'      => $validated['lastname'],
                        'email'         => $validated['sender_email'],
                        'phone'         => $validated['phone'],
                        'special_note'  => $validated['special_note']
                    ]
                )
            );

            if ($validated['preferred_contact'] === 'Text') {
                $this->dispatch('cart:clear-sub');
                return redirect()->to($smsUrl);
            } elseif ($validated['preferred_contact'] === 'WhatsApp') {
                $this->dispatch('cart:clear-sub');
                return redirect()->to($whatsappUrl);
            } else {
                Mail::html($body, function ($mail) use ($validated) {
                    $mail->to($this->user->email)
                        ->from(config('mail.from.address'), config('mail.from.name'))
                        ->replyTo($validated['sender_email'])
                        ->subject('Subscription');
                });

                $this->dispatch('cart:clear-sub');
                session()->flash('success', "Email sent to {$this->user->email}");
                return redirect(request()->header('Referer'));
            }
        } catch (\Exception $e) {
            Log::error('Cart email error: ' . $e->getMessage(), [
                'user_id' => $this->user->id ?? null,
                'username' => $this->username ?? null,
                'cartData' => $this->cartData ?? null,
            ]);

            session()->flash('error', "Failed to send email. Please try again later.");
        }
    }

    public function render()
    {
        return view('livewire.subscription-cart')->title('Subscription | Cart')->layoutData([
            'favicon' => $this->thumbnail,
            'site_url' => $this->site_url,
            'thumbnail' => $this->thumbnail,
            'title' => 'Cart',
            'color_settings_card_light' => $this->color_settings_card_light,
            'color_settings_card_dark' => $this->color_settings_card_dark,
            'color_settings_text_light' => $this->color_settings_text_light,
            'color_settings_text_dark' => $this->color_settings_text_dark,
            'color_settings_profile_pic_border' => $this->color_settings_profile_pic_border,
            'color_settings_custom_button_text' => $this->color_settings_custom_button_text,
            'color_settings_custom_button' => $this->color_settings_custom_button,
            'app_name' => $this->app_name,
        ]);
    }
}
