<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\User;
use App\Models\SubCard;
use Spatie\EloquentSortable\Sortable;
use Spatie\EloquentSortable\SortableTrait;
use Illuminate\Support\Str;

class PresentationPage extends Model implements Sortable
{
    use SortableTrait;

    protected $table = 'pages';
    protected $fillable = [
        'user_id',
        'page_id',
        'title',
        'description',
        'order',
        'is_active',
        'contents',
        'call_to_action'
    ];

    protected $casts = [
        'contents' => 'array',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function subcard()
    {
        return $this->belongsTo(SubCard::class, 'page_id');
    }

    public function page_content()
    {
        return $this->hasMany(PageContent::class, 'page_id');
    }

    protected static function booted()
    {
        static::creating(function ($page) {

            if (empty($page->slug)) {
                $baseSlug = Str::slug($page->title);
                $slug = $baseSlug;
                $count = 1;

                while (static::where('slug', $slug)->exists()) {
                    $slug = $baseSlug . '-' . $count++;
                }

                $page->slug = $slug;
            }
        });

        static::saving(function ($page) {

            $contents = $page->contents ?? [];
            foreach ($contents as &$content) {
                if (($content['type'] ?? null) === 'image' && is_array($content['value'])) {
                    $content['value'] = $content['value'][0] ?? null;
                }
            }

            if (empty($page->slug) || $page->isDirty('title')) {
                $baseSlug = Str::slug($page->title);
                $slug = $baseSlug;
                $count = 1;

                while (static::where('slug', $slug)
                    ->where('id', '!=', $page->id)
                    ->exists()
                ) {
                    $slug = $baseSlug . '-' . $count++;
                }

                $page->slug = $slug;
            }
        });

        // static::updating(function ($page) {
        //     if ($page->isDirty('contents')) {
        //         $original = $page->getOriginal('contents');
        //         $updated  = $page->contents;

        //         // decode JSON if still a string
        //         if (is_string($original)) $original = json_decode($original, true);
        //         if (is_string($updated))  $updated  = json_decode($updated, true);

        //         foreach ($original as $index => $oldContent) {
        //             if ($oldContent['type'] === 'image') {
        //                 $oldPath = $oldContent['value'];

        //                 $existsInNew = collect($updated)->contains(function ($c) use ($oldPath) {
        //                     return $c['type'] === 'image' && $c['value'] === $oldPath;
        //                 });

        //                 if (! $existsInNew) {
        //                     @unlink(storage_path('app/public/' . $oldPath));
        //                 }
        //             }
        //         }
        //     }
        // });

        static::updating(function ($page) {
            if ($page->isDirty('contents')) {
                $original = $page->getOriginal('contents');
                $updated  = $page->contents;

                // decode JSON if needed
                if (is_string($original)) $original = json_decode($original, true);
                if (is_string($updated))  $updated  = json_decode($updated, true);

                // Collect all updated image paths
                $updatedImagePaths = collect($updated)
                    ->filter(fn($c) => ($c['type'] ?? null) === 'image' && !empty($c['value']))
                    ->pluck('value')
                    ->toArray();

                foreach ($original as $oldContent) {
                    if (($oldContent['type'] ?? null) === 'image') {
                        $oldPath = $oldContent['value'] ?? null;
                        if ($oldPath && !in_array($oldPath, $updatedImagePaths)) {
                            // Only delete files that are truly removed
                            @unlink(storage_path('app/public/' . $oldPath));
                        }
                    }
                }
            }
        });



        static::deleting(function ($page) {
            foreach ($page->contents as $content) {
                if ($content['type'] === 'image') {
                    @unlink(storage_path('app/public/' . $content['value']));
                }
            }
        });
    }
}
