<?php

namespace App\Models;

use App\Models\Traits\HasProductPhoto;
use Spatie\EloquentSortable\Sortable;
use Spatie\EloquentSortable\SortableTrait;
use Illuminate\Database\Eloquent\Model;
use App\Models\User;
use App\Models\SubCard;
use App\Casts\MoneyCast;

class Product extends Model implements Sortable
{
    use SortableTrait;
    use HasProductPhoto;

    protected $fillable = [
        'user_id',
        'product_name',
        'description',
        'price',
        'type',
        'sizes',
        'colors',
        'styles',
        'category_id',
        'collection_id',
        'shop_online_button',
        'shop_online_link',
        'featured_image',
        'order',
        'is_active',
        'stock',
        'page_id',
        'subscription_options',
        'has_packages',
        'coin',
        'has_coins',
        'is_on_deal',
        'deal_price',
        'deal_starts_at',
        'deal_ends_at',
        'discount_percentage'
    ];

    protected $casts = [
        'deal_starts_at'            => 'datetime',
        'deal_ends_at'              => 'datetime',
        'sizes'                     => 'array',
        'colors'                    => 'array',
        'styles'                    => 'array',
        'subscription_options'      => 'array',
        'category_id'               => 'array',
        'collection_id'             => 'array',
    ];

    // Returns true if deal is currently active
    public function getIsDealActiveAttribute()
    {
        return $this->is_on_deal
            && (!$this->deal_starts_at || $this->deal_starts_at <= now())
            && (!$this->deal_ends_at || $this->deal_ends_at >= now());
    }

    // Returns true if deal is upcoming or active
    public function getIsDealUpcomingOrActiveAttribute()
    {
        return $this->is_on_deal
            && (!$this->deal_ends_at || $this->deal_ends_at >= now());
    }

    public function getFinalPriceAttribute()
    {
        // Deal must be active
        $isActiveDeal = $this->is_on_deal
            && (!$this->deal_starts_at || $this->deal_starts_at <= now())
            && (!$this->deal_ends_at || $this->deal_ends_at >= now());

        if ($isActiveDeal) {

            // If explicit deal_price is set, use it
            if ($this->deal_price !== null) {
                return $this->deal_price;
            }

            // Else calculate using discount_percentage
            if ($this->discount_percentage !== null) {
                return round($this->price * (1 - $this->discount_percentage / 100), 2);
            }
        }

        // Default price
        return $this->price;
    }

    // Automatically update deal_price when discount_percentage changes
    public function setDiscountPercentageAttribute($value)
    {
        $this->attributes['discount_percentage'] = $value;

        if ($value !== null && $this->price !== null) {
            $this->attributes['deal_price'] = round($this->price * (1 - $value / 100), 2);
        }
    }

    // Automatically update discount_percentage if deal_price changes
    public function setDealPriceAttribute($value)
    {
        $this->attributes['deal_price'] = $value;

        if ($value !== null && $this->price !== null && $this->price > 0) {
            $this->attributes['discount_percentage'] = round((1 - $value / $this->price) * 100);
        }
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function subcard()
    {
        return $this->belongsTo(SubCard::class, 'page_id');
    }


    // public function category()
    // {
    //     return $this->belongsToMany(ProductCategory::class, 'category_id');
    // }

    // public function collection()
    // {
    //     return $this->belongsToMany(ProductCollection::class, 'collection_id');
    // }



    // public function category()
    // {
    //     return $this->belongsToMany(ProductCategory::class, 'product_categories', 'product_id', 'id');
    // }

    // public function collection()
    // {
    //     return $this->belongsToMany(ProductCollection::class, 'products_collection', 'product_id', 'id');
    // }

    // public function collection()
    // {
    //     return $this->belongsTo(ProductCollection::class, 'collection_id');
    // }

    public function media()
    {
        return $this->hasMany(ProductMedia::class, 'product_id');
    }

    protected static function booted()
    {
        static::saving(function ($product) {
            // Check if there is an old product image and the product exists
            if ($product->exists && $product->getOriginal('featured_image') && $product->featured_image !== $product->getOriginal('featured_image')) {
                // Get the old product image path
                $oldImagePath = 'app/public/' . $product->getOriginal('featured_image');
                if (file_exists($oldImagePath)) {
                    unlink($oldImagePath);
                }
            }
        });

        // if product is being deleted, check if product exist and product has image. if yes, delete product image
        static::deleting(function ($product) {

            $mediaItems = $product->media()->get();
            //if product is being deleting, delete all media first
            foreach ($mediaItems as $media) {
                if ($media->path) {
                    $path = 'app/public/' . $media->path;
                    if (file_exists(storage_path($path))) {
                        unlink(storage_path($path));
                    }
                }

                $media->delete();
            }

            //then delete the featured image
            if ($product->exists && $product->featured_image) {
                $path = 'app/public/' . $product->featured_image;
                unlink(storage_path($path));
            }
        });
    }
}
