<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Spatie\EloquentSortable\Sortable;
use Spatie\EloquentSortable\SortableTrait;
use App\Models\Sponsor;
use App\Models\User;
use App\Models\SubCard;
use App\Helpers\SavingLinkHelpers;

class SponsorButton extends Model implements Sortable
{
    use SortableTrait;
    protected $table = 'sponsor_default_buttons';
    protected $fillable = [
        'title',
        'link',
        'type',
        'view',
        'user_id',
        'order',
        'sponsor_id',
        'is_active'
    ];

    public function user()
    {
        return $this->belongsTo(Sponsor::class);
    }

    protected static function booted()
    {
        static::saving(function ($button) {
            static::handleLinkFormatting($button);
        });

        static::created(function ($button) {
            $sponsor_id = $button->sponsor_id;
            $view = $button->view;


            if ($sponsor_id) {
                $sponsoredUsers = User::where('sponsor_id', $sponsor_id)->get();

                if ($sponsoredUsers->isNotEmpty()) {
                    // Get the original sponsor button to copy its details
                    $original = SponsorButton::where('is_active', true)
                        ->where('id', $button->id)
                        ->where('sponsor_id', $sponsor_id)
                        ->first();

                    foreach ($sponsoredUsers as $user) {

                        Button::create([
                            'user_id' => $user->id,
                            'title' => $original->title,
                            'link' => $original->link,
                            'type' => $original->type,
                            'view' => $original->view,
                            'page_id' => null,
                        ]);
                    }
                }
            }
        });
    }

    protected static function handleLinkFormatting($button)
    {
        // Validate the links before saving
        if ($button['type'] == 'cashapp') {
            try {
                $cashAppLink = SavingLinkHelpers::saveCashApp($button['link']);
                $button['link'] = $cashAppLink;
            } catch (\InvalidArgumentException $e) {
                throw new \InvalidArgumentException('Invalid input. Please enter a valid $cashtag/link.');
            }
        } else if ($button['type'] == 'email') {
            try {
                $paypalemailLink = SavingLinkHelpers::savePaypalEmail($button['link']);
                $button['link'] = $paypalemailLink;
            } catch (\InvalidArgumentException $e) {
                throw new \InvalidArgumentException('Invalid input. Please enter a valid email.');
            }
        } else if ($button['type'] == 'phone' || $button['type'] == 'sms') {
            try {
                $contactNumber = SavingLinkHelpers::saveContactNumber($button['link']);
                $button['link'] = $contactNumber;
            } catch (\InvalidArgumentException $e) {
                throw new \InvalidArgumentException('Invalid input. Please enter a valid number.');
            }
        } else if ($button['type'] == 'zelle') {
            try {
                $zelleLink = SavingLinkHelpers::saveZelle($button['link']);
                $button['link'] = $zelleLink;
            } catch (\InvalidArgumentException $e) {
                throw new \InvalidArgumentException('Invalid input. Please enter a valid number/email.');
            }
        } else if ($button['type'] == 'paypal_email') {
            try {
                $paypalemailLink = SavingLinkHelpers::savePaypalEmail($button['link']);
                $button['link'] = $paypalemailLink;
            } catch (\InvalidArgumentException $e) {
                throw new \InvalidArgumentException('Invalid input. Please enter a valid email.');
            }
        } else if ($button['type'] == 'paypal_link') {
            try {
                $paypalLink = SavingLinkHelpers::savePaypalLink($button['link']);
                $button['link'] = $paypalLink;
            } catch (\InvalidArgumentException $e) {
                throw new \InvalidArgumentException('Invalid input. Please enter a valid paypal username/link.');
            }
        } else if ($button['type'] == 'apple_pay') {
            try {
                $applepayLink = SavingLinkHelpers::saveApplePay($button['link']);
                $button['link'] = $applepayLink;
            } catch (\InvalidArgumentException $e) {
                throw new \InvalidArgumentException('Invalid input. Please enter a valid number.');
            }
        } else if ($button['type'] == 'venmo') {
            try {
                $venmoLink = SavingLinkHelpers::saveVenmo($button['link']);
                $button['link'] = $venmoLink;
            } catch (\InvalidArgumentException $e) {
                throw new \InvalidArgumentException('Invalid input. Please enter a valid venmo username/link.');
            }
        } else if ($button['type'] == 'whatsapp') {
            try {
                $whatsappLink = SavingLinkHelpers::saveWhatsApp($button['link']);
                $button['link'] = $whatsappLink;
            } catch (\InvalidArgumentException $e) {
                throw new \InvalidArgumentException('Invalid input. Please enter a valid whatsapp link/number.');
            }
        } else if ($button['type'] == 'telegram') {
            try {
                $telegramLink = SavingLinkHelpers::saveTelegram($button['link']);
                $button['link'] = $telegramLink;
            } catch (\InvalidArgumentException $e) {
                throw new \InvalidArgumentException('Invalid input. Please enter a valid telegram link/username/number.');
            }
        } else {
            // if type = link
            try {
                $securedLink = SavingLinkHelpers::saveSecureLink($button['link']);
                $button['link'] = $securedLink;
            } catch (\InvalidArgumentException $e) {
                throw new \InvalidArgumentException('Invalid input. Please enter a valid URL/link.');
            }
        }
    }
}
