<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Models\Button;
use App\Models\Direction;
use App\Models\TrustedNetwork;
use App\Models\Sale;
use App\Models\Social;
use App\Models\Product;
use App\Models\Service;
use App\Models\Video;
use App\Models\User;
use App\Models\Review;
use App\Models\UserSettings;
use App\Models\Ad;
use App\Models\Replay;
use App\Settings\CardDefaultsSetting;
use App\Models\Image;
use App\Models\Traits\HasProfilePhoto;
use Illuminate\Support\Facades\Hash;
use App\Helpers\SavingLinkHelpers;
use Illuminate\Support\Facades\Storage;
use App\Models\PresentationPage;
use App\Models\LeadCaptureForm;
use Illuminate\Notifications\Notifiable;

class SubCard extends Model
{
    use HasProfilePhoto;
    use Notifiable;

    protected $table = 'subcards';
    protected $fillable = [
        'user_id',
        'username',
        'firstname',
        'lastname',
        'email',
        'phone',
        'profile_pic',
        'cover_pic',
        'thumbnail',
        'job_title',
        'bio',
        'bio_long',
        'company',
        'company_website',
        'is_active',
        'fcm_token'
    ];

    public function lead_capture_form()
    {
        return $this->hasMany(LeadCaptureForm::class, 'page_id');
    }

    public function buttons()
    {
        return $this->hasMany(Button::class, 'page_id');
    }

    public function trusted_network()
    {
        return $this->hasMany(TrustedNetwork::class, 'page_id');
    }

    public function replays()
    {
        return $this->hasMany(Replay::class, 'page_id');
    }

    public function reviews()
    {
        return $this->hasMany(Review::class, 'page_id');
    }

    public function pages()
    {
        return $this->hasMany(PresentationPage::class, 'page_id');
    }

    public function sales()
    {
        return $this->hasMany(Sale::class, 'page_id');
    }

    public function products()
    {
        return $this->hasMany(Product::class, 'page_id');
    }

    public function product_category()
    {
        return $this->hasMany(ProductCategory::class, 'page_id');
    }

    public function product_collection()
    {
        return $this->hasMany(ProductCollection::class, 'page_id');
    }

    public function services()
    {
        return $this->hasMany(Service::class, 'page_id');
    }

    public function events()
    {
        return $this->hasMany(Event::class, 'page_id');
    }

    public function socials()
    {
        return $this->hasMany(Social::class, 'page_id');
    }

    public function videos()
    {
        return $this->hasMany(Video::class, 'page_id');
    }

    public function settings()
    {
        return $this->hasMany(UserSettings::class, 'page_id');
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function ads()
    {
        return $this->hasMany(Ad::class, 'page_id');
    }

    public function images()
    {
        return $this->hasMany(Image::class, 'page_id', 'id');
    }

    public function directions()
    {
        return $this->hasMany(Direction::class, 'page_id', 'id');
    }

    public function members()
    {
        return $this->hasMany(Member::class, 'page_id');
    }

    // public function referrals()
    // {
    //     return $this->hasMany(Referral::class, 'referrer_id_sub');
    // }

    public function referralsAsSub()
    {
        return $this->hasMany(Referral::class, 'referrer_id_sub')->where('referrer_type', 'sub');
    }

    protected static function booted()
    {
        static::saving(function ($subcard) {
            if ($subcard->company_website) {
                try {
                    $securedLink = SavingLinkHelpers::saveSecureLink($subcard['company_website']);
                    $subcard['company_website'] = $securedLink;
                } catch (\InvalidArgumentException $e) {
                    throw new \InvalidArgumentException('Invalid input. Please enter a valid URL/link.');
                }
            }

            if ($subcard->phone) {
                try {
                    $contactNumber = SavingLinkHelpers::saveContactNumber($subcard->phone);
                    $subcard->phone = $contactNumber;
                } catch (\InvalidArgumentException $e) {
                    throw new \InvalidArgumentException('Invalid input. Please enter a valid number.');
                }
            }
        });

        // static::updated(function ($subcard) {
        //     if ($subcard->cover_pic == NULL) {
        //         $subcard->settings()
        //             ->where('setting_key', 'profile_layout')
        //             ->update(['setting_value' => 'default']);
        //     }
        // });

        static::updated(function ($subcard) {
            // Get the current profile_layout value from settings
            $layout = $subcard->settings()
                ->where('setting_key', 'profile_layout')
                ->value('setting_value');

            // If the layout is cover-based and there’s no cover_pic, reset to default
            if (in_array($layout, ['cover_only', 'default_cover']) && $subcard->cover_pic === null) {
                $subcard->settings()
                    ->where('setting_key', 'profile_layout')
                    ->update(['setting_value' => 'default']);

                // \Log::info('Profile layout reset to default due to missing cover_pic', [
                //     'subcard_id' => $subcard->id,
                //     'previous_layout' => $layout,
                // ]);
            }
        });

        static::updating(function ($subcard) {

            if ($subcard->phone) {
                try {
                    $contactNumber = SavingLinkHelpers::saveContactNumber($subcard['phone']);
                    $subcard['phone'] = $contactNumber;
                } catch (\InvalidArgumentException $e) {
                    throw new \InvalidArgumentException('Invalid input. Please enter a valid number.');
                }
            }

            // if there's new profile pic, unlink old
            if ($subcard->isDirty('profile_pic') && $subcard->getOriginal('profile_pic')) {
                Storage::disk('public')->delete($subcard->getOriginal('profile_pic'));
            }

            // if there's new cover pic, unlink old
            if ($subcard->isDirty('cover_pic') && $subcard->getOriginal('cover_pic')) {
                Storage::disk('public')->delete($subcard->getOriginal('cover_pic'));
            }

            // if there's new thumbnail, unlink old
            if ($subcard->isDirty('thumbnail') && $subcard->getOriginal('thumbnail')) {
                Storage::disk('public')->delete($subcard->getOriginal('thumbnail'));
            }

            // $settings = $subcard->settings; // load the related settings model

            // if ($settings && $settings->isDirty('shop_thumbnail') && $settings->getOriginal('shop_thumbnail')) {
            //     Storage::disk('public')->delete($settings->getOriginal('shop_thumbnail'));
            // }

            if ($subcard->relationLoaded('settings')) {
                // If already loaded
                $settings = $subcard->settings;
            } else {
                // If not yet loaded, fetch manually
                $settings = $subcard->settings()->get();
            }

            foreach ($settings as $setting) {
                if ($setting->isDirty('shop_thumbnail')) {
                    $oldThumbnail = $setting->getOriginal('shop_thumbnail');

                    if ($oldThumbnail && Storage::disk('public')->exists($oldThumbnail)) {
                        Storage::disk('public')->delete($oldThumbnail);
                    }
                }
            }
        });

        static::created(function ($subcard) {
            $default = app(CardDefaultsSetting::class);

            if (!$subcard->is_duplicated) {

                if (!auth()->user()->is_admin) {
                    //footer enabled by default when subcard is created
                    UserSettings::create([
                        // 'user_id' => $subcard->user_id,
                        'user_id' => NULL,
                        'setting_key' => 'footer_enabled',
                        'setting_value' => true,
                        'page_id' => $subcard->id
                    ]);

                    UserSettings::create([
                        'page_id' => $subcard->id,
                        'setting_key' => 'footer_text',
                        'setting_value' => $default->footer_text,
                        'user_id' => NULL,
                    ]);

                    UserSettings::create([
                        'page_id' => $subcard->id,
                        'setting_key' => 'footer_link',
                        'setting_value' => $default->site_url . 'register?ref=' . $subcard->username,
                        'user_id' => NULL,
                    ]);

                    //subcard is searchable by default when subcard is created
                    UserSettings::create([
                        // 'user_id' => $subcard->user_id,
                        'user_id' => NULL,
                        'setting_key' => 'searchable',
                        'setting_value' => true,
                        'page_id' => $subcard->id
                    ]);

                    //display country true by default
                    UserSettings::create([
                        // 'user_id' => $subcard->user_id,
                        'user_id' => NULL,
                        'setting_key' => 'display_country',
                        'setting_value' => true,
                        'page_id' => $subcard->id,
                    ]);
                }

                //add to contact
                Button::create([
                    'user_id' => $subcard->user_id,
                    'title' => 'Add to Contact',
                    'link' => '',
                    'type' => 'vcf',
                    'view' => 'links',
                    'page_id' => $subcard->id
                ]);

                //How may I help?
                Button::create([
                    'user_id' => $subcard->user_id,
                    'title' => 'How may I help?',
                    'link' => '',
                    'type' => 'help',
                    'view' => 'links',
                    'page_id' => $subcard->id,
                    'is_active' => false,
                ]);

                //Exchange Contacts
                Button::create([
                    'user_id' => $subcard->user_id,
                    // 'title' => 'Exchange Contacts',
                    'title' => "Let's Connect",
                    'link' => '',
                    'type' => 'exchange',
                    'view' => 'links',
                    'page_id' => $subcard->id
                ]);

                //Send your feedback
                Button::create([
                    'user_id' => $subcard->user_id,
                    'title' => 'Send your feedback',
                    'link' => '',
                    'type' => 'feedback',
                    'view' => 'links',
                    'page_id' => $subcard->id,
                    'is_active' => false,
                ]);

                //My Bio
                Button::create([
                    'user_id' => $subcard->user_id,
                    'title' => 'My Bio',
                    'link' => '',
                    'type' => 'bio',
                    'view' => 'links',
                    'page_id' => $subcard->id
                ]);

                //button
                if ($subcard->email) {
                    //add Button, use the email
                    Button::create([
                        'user_id' => $subcard->user_id,
                        'title' => 'Email',
                        'link' => $subcard['email'],
                        'type' => 'email',
                        'view' => 'links',
                        'page_id' => $subcard->id
                    ]);
                }

                //Events
                Button::create([
                    'user_id' => $subcard->user_id,
                    'title' => 'Events & Promotions',
                    'link' => '',
                    'type' => 'events',
                    'view' => 'links',
                    'page_id' => $subcard->id
                ]);

                //Shop
                Button::create([
                    'user_id' => $subcard->user_id,
                    'title' => 'Shop',
                    'link' => '',
                    'type' => 'shop',
                    'view' => 'links',
                    'page_id' => $subcard->id
                ]);

                //Sales Page
                // Button::create([
                //     'user_id' => $subcard->user_id,
                //     'title' => 'Sales',
                //     'link' => '',
                //     'type' => 'sales',
                //     'view' => 'links',
                //     'page_id' => $subcard->id
                // ]);

                //Meet the team
                // Button::create([
                //     'user_id' => $subcard->user_id,
                //     'title' => 'Meet the Team',
                //     'link' => '',
                //     'type' => 'team',
                //     'view' => 'links',
                //     'page_id' => $subcard->id
                // ]);

                Button::create([
                    'user_id' => $subcard->user_id,
                    'title' => 'Driving Directions',
                    'link' => '',
                    'type' => 'driving',
                    'view' => 'links',
                    'page_id' => $subcard->id
                ]);

                Button::create([
                    'user_id' => $subcard->user_id,
                    'title' => 'Zoom Replays',
                    'link' => '',
                    'type' => 'zoom_replay',
                    'view' => 'links',
                    'page_id' => $subcard->id
                ]);

                Button::create([
                    'user_id' => $subcard->user_id,
                    'title' => 'Testimonials',
                    'link' => '',
                    'type' => 'reviews',
                    'view' => 'links',
                    'page_id' => $subcard->id
                ]);

                Button::create([
                    'user_id' => $subcard->user_id,
                    'title' => 'Trusted Network',
                    'link' => '',
                    'type' => 'trusted_network',
                    'view' => 'links',
                    'page_id' => $subcard->id
                ]);

                Button::create([
                    'user_id' => $subcard->user_id,
                    'title' => 'Payment Links',
                    'link' => '',
                    'type' => 'payments',
                    'view' => 'links',
                    'page_id' => $subcard->id
                ]);

                //icon
                if ($subcard->email) {
                    //add Button, use the email
                    Button::create([
                        'user_id' => $subcard->user_id,
                        'title' => 'Email',
                        'link' => $subcard['email'],
                        'type' => 'email',
                        'view' => 'contacts',
                        'page_id' => $subcard->id
                    ]);
                }

                //add Button, use the number for sms
                if ($subcard->phone) {
                    Button::create([
                        'user_id' => $subcard->user_id,
                        'title' => 'SMS/Text',
                        'link' => $subcard['phone'],
                        'type' => 'sms',
                        'view' => 'links',
                        'page_id' => $subcard->id
                    ]);
                }

                //icon
                if ($subcard->phone) {
                    Button::create([
                        'user_id' => $subcard->user_id,
                        'title' => 'SMS/Text',
                        'link' => $subcard['phone'],
                        'type' => 'sms',
                        'view' => 'contacts',
                        'page_id' => $subcard->id
                    ]);
                }

                //add Button, use the number for direct call
                if ($subcard->phone) {
                    Button::create([
                        'user_id' => $subcard->user_id,
                        'title' => 'Direct Call',
                        'link' => $subcard['phone'],
                        'type' => 'phone',
                        'view' => 'links',
                        'page_id' => $subcard->id
                    ]);
                }

                if ($subcard->phone) {
                    //icon
                    Button::create([
                        'user_id' => $subcard->user_id,
                        'title' => 'Direct Call',
                        'link' => $subcard['phone'],
                        'type' => 'phone',
                        'view' => 'contacts',
                        'page_id' => $subcard->id
                    ]);
                }

                if ($subcard->company_website) {
                    Button::create([
                        'user_id' => $subcard->user_id,
                        'title' => 'Company Website',
                        'link' => $subcard->company_website,
                        'type' => 'link',
                        'view' => 'links',
                        'page_id' => $subcard->id
                    ]);
                }
            }
        });

        static::deleting(function ($subcard) {
            //when a subcard is deleted, make the main card active
            User::where('id', $subcard->user_id)->update(['active_card' => true]);

            // if subcard is being deleted, subcard's profile pic will also be deleted
            if ($subcard->profile_pic) {
                $path = storage_path('app/public/' . $subcard->profile_pic);
                if (file_exists($path)) {
                    unlink($path);
                }
            }

            //delete all product images if user is being deleted. that includes the products created in subcards
            $products = Product::where('page_id', $subcard->id)->get();
            if ($products) {

                foreach ($products as $product) {
                    // Delete product media
                    foreach ($product->media as $media) {
                        if ($media->type == 'image') {
                            $path = 'app/public/' . $media->path;
                            if (file_exists(storage_path($path))) {
                                unlink(storage_path($path));
                            }
                        }
                        $media->delete();
                    }

                    // Delete featured image
                    if ($product->featured_image) {
                        $path = 'app/public/' . $product->featured_image;
                        if (file_exists(storage_path($path))) {
                            unlink(storage_path($path));
                        }
                    }
                }
            }

            //delete all service images if user is being deleted. that includes the services created in subcards
            $services = Service::where('page_id', $subcard->id)->get();
            if ($services) {
                foreach ($services as $service) {
                    if ($service->exists && $service->service_image) {
                        $path = 'app/public/' . $service->service_image;
                        unlink(storage_path($path));
                    }
                }
            }

            //delete thumbnail
            if ($subcard->thumbnail) {
                $path = 'app/public/' . $subcard->thumbnail;
                unlink(storage_path($path));
            }

            if ($subcard->cover_pic) {
                $path = storage_path('app/public/' . $subcard->cover_pic);
                if (file_exists($path)) {
                    unlink($path);
                }
            }
        });
    }
}
